// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: boxes.cpp,v 3.7 1999/07/03 13:46:16 kudou Exp $
// manage BeatWord toolboxes

#include "pword.h"
#include "boxes.h"
#include "attribut.h"
#include "bufnew.h"
#include "dialogs.h"
#include "dlg.h"
#include "docedit.h"
#include "docwindo.h"
#include "interval.h"
#include "messbox.h"
#include "mwindow.h"
#include "pmenus.h"
#include "pref.h"
#include "pwordbas.h"
#include "pwordpre.h"
#include "screen.h"
#include "window.h"
#include "xstr.h"

#ifdef BW3_SCEENCUST
#include "scustdlg.h"
#endif /* BW3_SCEENCUST */

#define COPY_BUTTON

class Box;

// index of boxes.
// TOP_RIBBON ... RIGHT_RIBBON must be 0 .. 3.
#define TOP_RIBBON 0
#define BOT_RIBBON 1
#define LEFT_RIBBON 2
#define RIGHT_RIBBON 3
#define MODE_TOOLBOX 4
#define CHAR_TOOLBOX 5
#define PARA_TOOLBOX 6
#define USER_TOOLBOX_1 7
#define USER_TOOLBOX_2 8
#define USER_TOOLBOX_3 9
#define USER_TOOLBOX_4 10
#define BIG_TOOLBOX 11

// Following loop support:
// for (int i = RIBBON_BEGIN; i != RIBBON_END; ++i)
// {
//   Box *box = boxes[i];
//   ...
// }

#define RIBBON_BEGIN 0
#define RIBBON_END 4

// Following loop support:
// for (int i = TOOLBOX_BEGIN; i != TOOLBOX_END; ++i)
// {
//   Box *box = boxes[i];
//   ...
// }

#define TOOLBOX_BEGIN 4
#define TOOLBOX_END	(BIG_TOOLBOX + 1)

// Following loop support:
// for (int i = BOX_BEGIN; i != BOX_END; ++i)
// {
//   Box *box = boxes[i];
//   ...
// }

#define BOX_BEGIN RIBBON_BEGIN
#define BOX_END TOOLBOX_END

#define NUM_BOXES BOX_END
#define NUM_RIBBONS RIBBON_END

#define DEFAULT_RIBBON_SIZE 4
#define MIN_RIBBON_SIZE 4
#define MAX_RIBBON_SIZE 40

#define MIN_TOOLBOX_SIZE 4
#define MAX_TOOLBOX_SIZE 100

#define MIN_BUTTON_SIZE 1
#define MAX_BUTTON_SIZE 40

#define MAX_BUTTONS 200
#define NUM_PROFILE_STRINGS 4

static Box *STATIC_NEAR boxes[NUM_BOXES] = { NULL, };

#define CURSOR_OK 0
#define CURSOR_NO 1
#define CURSOR_TRASH 2
#define CURSOR_SETTING 3
#define NUM_CURSORS 4

static HCURSOR STATIC_NEAR cursors[NUM_CURSORS] = { 0, };

static int STATIC_NEAR global_visible_p = True;
static bool STATIC_NEAR global_customizable_p = false;
static int STATIC_NEAR disable_some_profile = False;

// Button core information
struct Bitmap_Button
{
  int idm;
  int state;
  char* bitmap_name;
  HICON icon_cache;
};

// This vector must synchronized on enum BB.
static Bitmap_Button STATIC_NEAR bitmap_button[] =
{
#define DEFINE_BUTTON(idm, bitmap_name) { idm, 0, bitmap_name, 0, },
#include "boxes_m.h"
};

// button-id, it means index of `bitmap_button'.
enum BB
{
#define DEFINE_BUTTON(idm, butmap_name) BB_ ## idm,
#include "boxes_m.h"
  BB_UNUSED_MAX,
  // Following codes are pseudo operator.
  BB_VISIBLE,
  BB_SIZE,
  BB_VERTICAL,
  BB_SPACE,
  BB_NEWLINE,
  BB_INITIAL_TEXT_BUTTON,
  BB_MINCHO_TEXT_BUTTON,
  BB_GOTHIC_TEXT_BUTTON,
  BB_END,
};

// ------------------------------------------------------------
// profiled information

#define BUTTON_SIZE_AUTO 0
#define BUTTON_SIZE_SMALL 5
#define BUTTON_SIZE_MEDIUM 6
#define BUTTON_SIZE_LARGE 8
#define BUTTON_SIZE_DEFAULT BUTTON_SIZE_MEDIUM

// definition for Pref_button_size
struct Enum_Desc button_size_desc[] =
{
  { S_button_size_auto,		BUTTON_SIZE_AUTO, },
  { S_button_size_large,	BUTTON_SIZE_LARGE, },
  { S_button_size_medium,	BUTTON_SIZE_MEDIUM, },
  { S_button_size_small,	BUTTON_SIZE_SMALL, },
  { S_button_size2_auto,	BUTTON_SIZE_AUTO, },
  { S_button_size2_large,	BUTTON_SIZE_LARGE, },
  { S_button_size2_medium,	BUTTON_SIZE_MEDIUM, },
  { S_button_size2_small,	BUTTON_SIZE_SMALL, },
  { NULL,			BUTTON_SIZE_AUTO, },
};

// 1 unit * UNITS_PER_BUTTON ==> bitmap button size
#define UNITS_PER_BUTTON 4

// Convert floating geometry to button-unit based integer.
#define G(f) ((int) ((f) * UNITS_PER_BUTTON))

// Exact button unit size, this is calculated from Pref_button_size.
static int button_unit;

// module initialized flag.
static int STATIC_NEAR inited = False;

// ------------------------------------------------------------
// generic utilities

static void NEAR
init_fail()
{
  initialization_fail();
}

static HCURSOR NEAR
load_cursor(HANDLE h, char* s)
{
  return (HCURSOR)::LoadCursor((HINSTANCE)h, s);
}

static void NEAR
delete_object(HANDLE h)
{
  if (h != 0)
  {
    ::DeleteObject(h);
  }
}

static void NEAR
destroy_icon(HICON icon)
{
  ::DestroyIcon(icon);
}

static HANDLE NEAR
select_object(HDC dc, HANDLE h)
{
  return(::SelectObject(dc, h));
}

static HCURSOR NEAR
arrow_cursor()
{
  return(load_cursor(0, (char*) IDC_ARROW));
}

static void NEAR
set_bk_mode_transparent(HDC dc)
{
  ::SetBkMode(dc, TRANSPARENT);
}

static void NEAR
beep(int do_beep_p)
{
  if (do_beep_p)
  {
    ::MessageBeep(0);
  }
}

static DWORD NEAR
set_text_color(HDC dc, DWORD color)
{
  return(::SetTextColor(dc, color));
}

static void NEAR
fill_rect(HDC dc, RECT* r, HBRUSH brush)
{
  ::FillRect(dc, r, brush);
}

static void NEAR
fill_solid_color_rect(HDC dc, RECT* r, DWORD color)
{
  DWORD old_color = set_text_color(dc, color);
  fill_rect(dc, r, Screen::black_brush);
  set_text_color(dc, old_color);
}

static void NEAR
fill_color_rect(HDC dc, RECT* r, DWORD color)
{
  HBRUSH brush = ::CreateSolidBrush(color);
  fill_rect(dc, r, brush);
  delete_object(brush);
}

static int NEAR
ensmall_rect(RECT* a, RECT* b, int delta)
{
  a->left = b->left + delta;
  a->right = b->right - delta;
  a->top = b->top + delta;
  a->bottom = b->bottom - delta;
  return(a->left < a->right && a->top < a->bottom);
}

static int NEAR
save_dc(HDC dc)
{
  return(::SaveDC(dc));
}

static void NEAR
restore_dc(HDC dc, int save_id)
{
  ::RestoreDC(dc, save_id);
}

static int NEAR
intersect_clip_rect(HDC dc, RECT* r)
{
  return(::IntersectClipRect(dc, r->left, r->top, r->right, r->bottom)
	  != NULLREGION);
}

static void NEAR
draw_text(HDC dc,
	   char* s,
	   RECT* r_arg, unsigned int flags, int verticalp, DWORD color)
{
  int save_id = save_dc(dc);
  RECT r[1];
  r[0] = r_arg[0];
  if (intersect_clip_rect(dc, r))
  {
    set_text_color(dc, color);
    select_object(dc,
		   (verticalp ? Screen::vertical_system_font
		    : Screen::system_font));
    int x = r->left;
    int y = r->top;
    if (verticalp)
    {
      x = (r->left + r->right + Screen::system_font_height) / 2;
    }
    else
    {
      y = (r->top + r->bottom - Screen::system_font_height) / 2;
    }
    int length = strlen(s);
    if (flags & DT_CENTER)
    {
#ifdef _WIN32
      SIZE width_height;
      ::GetTextExtentPoint32(dc, s, length, &width_height);
      if (verticalp)
      {
	// signed div
	y = (int) (r->top + r->bottom - width_height.cy) / 2;
      }
      else
      {
	// signed div
	x = (int) (r->left + r->right - width_height.cx) / 2;
      }
#else /* _WIN32 */
      DWORD extent = ::GetTextExtent(dc, s, length);
      if (verticalp)
      {
	// signed div
	y = (int) (r->top + r->bottom - HIWORD(extent)) / 2;
      }
      else
      {
	// signed div
	x = (int) (r->left + r->right - LOWORD(extent)) / 2;
      }
#endif /* _WIN32 */
    }
    ::TextOut(dc, x, y, s, length);
  }
  restore_dc(dc, save_id);
}

static void NEAR
position_windows()
{
  main_window->PositionWindows();
}

// ------------------------------------------------------------
// class Super_Box
// The class Super_Box is super class of Bar, and Box.

enum Box_Type
{
  Box_Type_RIBBON,
  Box_Type_TOOLBOX,
  Box_Type_TWISTED_TOOLBOX,
};

#define ribbonp(type) ((type) == Box_Type_RIBBON)
#define toolboxp(type) (!ribbonp(type))
#define twisted_toolbox_p(type) ((type) == Box_Type_TWISTED_TOOLBOX)

class Super_Box
  : public Window
{
public:
  Box_Type box_type;
  
  // Synchronized window show status
  int shownp;

  // This is logical status.
  int visiblep;
  
  // for redisplay
  HDC dc;
  
  // constructor
public:
  NEAR Super_Box(Box_Type box_type);

  // virtual method
public:
  LRESULT_T Super_Box::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam);

  // method
public:
  HDC NEAR Super_Box::begin_paint(PAINTSTRUCT* ps);
  int NEAR Super_Box::get_nonborder_client_rect(RECT* r);
  void NEAR Super_Box::arrange(RECT* rect, int place, int size);
  void NEAR Super_Box::client_to_screen(POINT* p);
  void NEAR Super_Box::draw_border();
  void NEAR Super_Box::draw_frame(RECT* r);
  void NEAR Super_Box::end_paint(PAINTSTRUCT* ps);
  void NEAR Super_Box::fill_background();
  void NEAR Super_Box::get_client_rect(RECT* r);
  void NEAR Super_Box::get_window_rect(RECT* r);
  void NEAR Super_Box::invalidate();
  void NEAR Super_Box::invalidate_rect(RECT* r);
  void NEAR Super_Box::move_window(int x, int y, int width, int height);
  void NEAR Super_Box::show_window(int flag);
};

void NEAR
Super_Box::show_window(int flag)
{
  if (flag)
  {
    if (!this->shownp)
    {
      this->shownp = True;
      this->Show();
    }
  }
  else
  {
    if (this->shownp)
    {
      this->shownp = False;
      this->Hide();
    }
  }
}

void NEAR
Super_Box::invalidate_rect(RECT* r)
{
  ::InvalidateRect(this->GetHandle(), r, True);
}

void NEAR
Super_Box::invalidate()
{
  this->invalidate_rect(NULL);
}

void NEAR
Super_Box::client_to_screen(POINT* p)
{
  ::ClientToScreen(this->GetHandle(), p);
}

void NEAR
Super_Box::get_client_rect(RECT* r)
{
  ::GetClientRect(this->GetHandle(), r);
}

int NEAR
Super_Box::get_nonborder_client_rect(RECT* r)
{
  this->get_client_rect(r);
  return(ensmall_rect(r, r, 1));
}

void NEAR
Super_Box::get_window_rect(RECT* r)
{
  ::GetWindowRect(this->GetHandle(), r);
}

void NEAR
Super_Box::move_window(int x, int y, int width, int height)
{
  ::MoveWindow(this->GetHandle(), x, y, width, height, True);
}

void NEAR
Super_Box::draw_frame(RECT* r)
{
  ::FrameRect(this->dc, r, Screen::stocked_black_brush);
}

void NEAR
Super_Box::fill_background()
{
  RECT r[1];
  this->get_client_rect(r);
  fill_color_rect(dc, r, Screen::button_face_color);
}

void NEAR
Super_Box::draw_border()
{
  RECT r[1];
  this->get_client_rect(r);
  this->draw_frame(r);
  if (ensmall_rect(r, r, 1))
  {
    intersect_clip_rect(this->dc, r);
    this->fill_background();
  }
}

void NEAR
Super_Box::end_paint(PAINTSTRUCT* ps)
{
  this->dc = 0;
  ::EndPaint(this->GetHandle(), ps);
}

HDC NEAR
Super_Box::begin_paint(PAINTSTRUCT* ps)
{
  ::BeginPaint(this->GetHandle(), ps);
  HDC dc = ps->hdc;
  this->dc = dc;
  return(dc);
}

void NEAR
Super_Box::arrange(RECT* rect, int place, int size)
{
  int visiblep = this->visiblep;
  if (visiblep)
  {
    int top_x = rect->left;
    int top_y = rect->top;
    int bot_x = rect->right;
    int bot_y = rect->bottom;
    int width = bot_x - top_x;
    int height = bot_y - top_y;
    ++size;
    visiblep = False;
    switch (place)
    {
     case TOP_RIBBON:
      if (size * 2 <= height)
      {
	visiblep = True;
	this->move_window(top_x - 1, top_y - 1, width + 2, size + 1);
	top_y += size;
	height -= size;
      }
      break;

     case BOT_RIBBON:
      if (size * 2 <= height)
      {
	visiblep = True;
	this->move_window(top_x - 1, bot_y - size, width + 2, size + 1);
	bot_y -= size;
	height -= size;
      }
      break;
      
     case LEFT_RIBBON:
      if (size * 2 <= width)
      {
	visiblep = True;
	this->move_window(top_x - 1, top_y - 1, size + 1, height + 2);
	top_x += size;
	width -= size;
      }
      break;

     case RIGHT_RIBBON:
      if (size * 2 <= width)
      {
	visiblep = True;
	this->move_window(bot_x - size, top_y - 1, size + 1, height + 2);
	bot_x -= size;
	width -= size;
      }
      break;
    }
    rect->left = top_x;
    rect->right = bot_x;
    rect->top = top_y;
    rect->bottom = bot_y;
  }
  this->show_window(visiblep);
}

// virtual method
LRESULT_T
Super_Box::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam)
{
  HWND handle = this->GetHandle();
  if (handle)
  {
    switch (message)
    {
     case WM_DESTROY:
      this->SetHandle(0);
      break;
      
     case WM_CLOSE:
      this->visiblep = False;
      this->show_window(False);
      return(1);
    }
  }
  return(DefWindowProc(message, wParam, lParam));
}

static void NEAR
register_class(WNDCLASS* window_class)
{
  if (RegisterClass(window_class) == 0)
  {
    init_fail();
  }
}

// constructor
NEAR
Super_Box::Super_Box(Box_Type box_type)
{
  this->box_type = box_type;
  this->shownp = False;
  this->visiblep = False;
  this->dc = 0;
  static int STATIC_NEAR firstp = True;
  if (firstp)
  {
    firstp = False;
    WNDCLASS window_class;

    // register ribbon window
    memset(&window_class, 0, sizeof(window_class));
    window_class.lpszClassName = RIBBON_CLASS_NAME;
    window_class.hInstance = PWordPresentation::hInstance;
    window_class.lpfnWndProc = fWndProc;
    window_class.cbWndExtra = DLGWINDOWEXTRA + sizeof(this);
    window_class.hCursor = arrow_cursor();
    window_class.style = (CS_VREDRAW | CS_HREDRAW);
    register_class(&window_class);

    // register toolbox
    memset(&window_class, 0, sizeof(window_class));
    window_class.lpszClassName = TOOLBOX_CLASS_NAME;
    window_class.hInstance = PWordPresentation::hInstance;
    window_class.lpfnWndProc = fWndProc;
    window_class.cbWndExtra = DLGWINDOWEXTRA + sizeof(this);
    window_class.hCursor = arrow_cursor();
    window_class.style = (CS_VREDRAW | CS_HREDRAW | CS_SAVEBITS);
    register_class(&window_class);
  }
  
  HWND handle;
  if (ribbonp(box_type))
  {
    handle = CreateWindow(RIBBON_CLASS_NAME,
			  NULL,
			  (WS_CHILD | WS_CLIPSIBLINGS),
			  0,
			  0,
			  0,
			  0,
			  main_window_handle,
			  0,
			  PWordPresentation::hInstance,
			  NULL);
  }
  else
  {
    handle = CreateDialog(PWordPresentation::hInstance,
			  TOOLBOX_TEMPLATE_NAME, main_window_handle, NULL);
  }
  if (handle == 0)
  {
    init_fail();
  }
  this->SetHandle(handle);
  SetPointer(handle, this, DLGWINDOWEXTRA);
}

// ------------------------------------------------------------
// class Bar

#define NO_RIBBON NUM_RIBBONS

// Internal gap width in message bar.
#define INTERNAL_HEIGHT 1

// definition for Pref_message_bar
struct Enum_Desc message_bar_desc[] =
{
  { "Top",	TOP_RIBBON, },
  { "Bottom",	BOT_RIBBON, },
  { "Left",	LEFT_RIBBON, },
  { "Right",	RIGHT_RIBBON, },
  { "None",	NO_RIBBON, },
  { NULL,	BOT_RIBBON, },
};

class Bar
  : public Super_Box
{
public:
  char* message;
  int bar_type;
  int size;
  
  // constructor
public:
  NEAR Bar();

  // virtual method
public:
  LRESULT_T Bar::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam);

  // method
public:
  long NEAR Bar::set_bar_type(int menu_state_p, int type);
  void NEAR Bar::set_message(char* s);
};

static Bar* STATIC_NEAR message_bar;

// virtual method
LRESULT_T
Bar::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam)
{
  HWND handle = this->GetHandle();
  if (handle)
  {
    switch (message)
    {
     case WM_PAINT:
      {
	PAINTSTRUCT ps;
	HDC dc = this->begin_paint(&ps);
	set_bk_mode_transparent(dc);
	this->draw_border();
	RECT r;
	if (this->get_nonborder_client_rect(&r))
	{
	  int internal_width = Screen::system_font_height / 8 + 1;
	  int verticalp = (this->bar_type == LEFT_RIBBON
			   || this->bar_type == RIGHT_RIBBON);
	  if (verticalp)
	  {
	    r.top += internal_width;
	  }
	  else
	  {
	    r.left += internal_width;
	  }
	  draw_text(dc,
		     this->message,
		     &r, DT_LEFT, verticalp, Screen::button_text_color);
	}
	this->end_paint(&ps);
      }
      return(0);
    }
  }
  return(Super_Box::WndProc(message, wParam, lParam));
}

void NEAR
Bar::set_message(char* s)
{
  if (this->message == NULL || strcmp(this->message, s) != 0)
  {
    xfree(this->message);
    this->message = dup_string(s);
    RECT r[1];
    if (this->get_nonborder_client_rect(r))
    {
      this->invalidate_rect(r);
      ::UpdateWindow(this->GetHandle());
    }
  }
}

long NEAR
Bar::set_bar_type(int menu_state_p, int type)
{
  if (menu_state_p)
  {
    return(this->bar_type == type ? (MF_ENABLED | MF_CHECKED) : MF_ENABLED);
  }
  if (this->bar_type != type)
  {
    this->bar_type = type;
    this->visiblep = (type != NO_RIBBON);
    position_windows();
  }
  return(0);
}

// constructor
NEAR
Bar::Bar()
  : Super_Box(Box_Type_RIBBON)
{
  this->message = dup_string("");
  this->bar_type = NO_RIBBON;
  this->size = MAX(Screen::system_font_height, 10);
}

// ------------------------------------------------------------
// Declaration of class Box and some dialogs

#define MAX_BUTTON_TEXT_LENGTH 40

enum Button_Type
{
  Button_Type_BITMAP,
  Button_Type_TEXT,
  Button_Type_VERTICAL_TEXT,
};

#define bitmap_button_p(type) ((type) == Button_Type_BITMAP)
#define text_button_p(type) (!bitmap_button_p(type))
#define vertical_text_button_p(type) ((type) == Button_Type_VERTICAL_TEXT)

struct Box_Button
{
  Button_Type button_type;
  union
  {
    Bitmap_Button* bb;
    char* text;
  } u;
  
  // command code
  int idm;
  
  // position of button on button-unit coordinate.
  int x;
  int y;
  int width;
  int height;
  
  // local state.
  int state;
};

enum MouseOperation
{
  NO_MOUSE_OPERATION,
  NORMAL_MOUSE_OPERATION,
  CUSTOMIZE_MOUSE_OPERATION,
};

class Box
  : public Super_Box
{
public:
  char* keyword;
  char* name;
  int width;
  int height;
  int customizablep;
  int num_buttons;
  Box_Button* buttons;
  
  MouseOperation mouse_operation;
  int mouse_down_button;
  int mouse_down_button_active_p;

  // for profile
  int* builtin_profile;
  char* initial_profile[NUM_PROFILE_STRINGS];
  int changed;
  
  // constructor
  NEAR Box(char* keyword, char* name, Box_Type box_type, int* profile);

  // virtual method
  LRESULT_T Box::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam);

  // method
  int NEAR Box::add_button(Box_Button* button);
  int NEAR Box::check_button_overlap(int x, int y, int width, int height, Box_Button* ignored_button);
  int NEAR Box::find_button(POINT* p);
  int NEAR Box::expr_next_key();
  int NEAR Box::parse_builtin_profile(int* profile);
  int NEAR Box::parse_profile();
  int NEAR Box::read_profile();
  void NEAR Box::add_bitmap_button(int button_id, int x, int y);
  void NEAR Box::change_button_state(int i, int state);
  void NEAR Box::customize(MSG_T message, WPARAM_T wParam, LPARAM_T lParam);
  void NEAR Box::customize_cancel(int do_beep_p);
  void NEAR Box::customize_move(LPARAM_T lParam);
  void NEAR Box::customize_setting();
  void NEAR Box::drawing_epilogue();
  void NEAR Box::drawing_prologue();
  void NEAR Box::fit_toolbox_size();
  void NEAR Box::flush_expr();
  void NEAR Box::inactivate_mouse(int do_beep_p);
  void NEAR Box::move_toolbox(int x, int y);
  void NEAR Box::no_mouse_operation();
  void NEAR Box::point_to_xy(POINT* p, int* x_ret, int* y_ret);
  void NEAR Box::redisplay_button(int n);
  void NEAR Box::remove_button(int button_index);
  void NEAR Box::restore_button_state(int i);
  void NEAR Box::set_ribbon_size(int size);
  void NEAR Box::twist_toolbox();
  void NEAR Box::write_expr(Pref_Expr* expr);
  void NEAR Box::write_simple_expr(char* op, int num_args, int arg0, int arg1);
  void NEAR Box::write_profile();
  
  // static method
  static Box* NEAR Box::make_ribbon(char* keyword, char* name, int* profile);
  static Box* NEAR Box::make_toolbox(char* keyword, char* name, int* profile);
};

class MenuSelectDialog
  : public PModalDialog
{
private:
  int idm;
  
  // constructor, destructor
public:
  NEAR MenuSelectDialog();
  
  // virtual method
public:
  virtual LRESULT_T MenuSelectDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam);

  // methods
public:
  int NEAR MenuSelectDialog::Go();
};

struct TextButtonDialogValues
{
  Box* box;
  Box_Button* button;
  char text[MAX_BUTTON_TEXT_LENGTH + 1];
  int width;
  int height;
  int verticalp;
  int idm;
};

class TextButtonDialog
  : public PModalDialog
{
private:
  int busy;
  TextButtonDialogValues values;
  
  // constructor, destructor
public:
  NEAR TextButtonDialog(TextButtonDialogValues* values);
  
  // virtual method
public:
  virtual LRESULT_T TextButtonDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam);
  
  // method
private:
  void NEAR TextButtonDialog::check_size(int id, int* place);
  void NEAR TextButtonDialog::init_listbox(int id, Command_Desc** table);
  void NEAR TextButtonDialog::set_command(Command_Desc* command);
  void NEAR TextButtonDialog::set_guide(Command_Desc* command);
  void NEAR TextButtonDialog::set_idm(int idm);
  void NEAR TextButtonDialog::set_listbox(int id, Command_Desc* command, Command_Desc** table);
public:
  void NEAR TextButtonDialog::get_values(TextButtonDialogValues* values);
};

class SettingDialog
  : public NoKanjiModalDialog
{
public:
  int busy;
  int width;
  int height;
  int max_size;
  int idd;
  char* box_name;
  
  // constructor, destructor
public:
  NEAR SettingDialog(char* class_name, char* box_name, int width, int height, int max_size);
  
  // virtual method
public:
  virtual LRESULT_T SettingDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam);
  
  // method
private:
  void NEAR SettingDialog::check_size(int id, int* place);
};

// ------------------------------------------------------------
// implementation of SettingDialog

#undef SUPER_CLASS
#define SUPER_CLASS NoKanjiModalDialog

// private method
void NEAR
SettingDialog::check_size(int id, int* place)
{
  int size;
  int ok = get_item_button_unit(this->GetHandle(), id, &size);
  if (ok)
  {
    *place = size;
    if (this->width < MIN_RIBBON_SIZE
	|| this->max_size < this->width
	|| this->height < MIN_RIBBON_SIZE || this->max_size < this->height)
    {
      ok = False;
    }
  }
  EnableWindow(GetItem(IDOK), ok);
}

// virtual method based on class Window
LRESULT_T
SettingDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam)
{
  HWND handle = this->GetHandle();
  switch (iMessage)
  {
   case WM_INITDIALOG:
    this->busy = True;
    set_item_button_unit(handle, IDD_SETTING_WIDTH, this->width);
    if (this->height != 0)
    {
      set_item_button_unit(handle, IDD_SETTING_HEIGHT, this->height);
    }
    else
    {
      this->height = MIN_RIBBON_SIZE;
    }
    this->busy = False;
    break;
    
   case WM_COMMAND:
    if (!this->busy)
    {
#ifdef _WIN32
      switch (LOWORD(wParam))
#else /* _WIN32 */
      switch (wParam)
#endif /* _WIN32 */
      {
       case IDD_SETTING_DEFAULT:
       case IDD_SETTING_STARTUP:
	if (IssueVA(SF_setting_message,
		     MB_OKCANCEL | MB_ICONHAND,
		     this->box_name,
#ifdef _WIN32
		     (LOWORD(wParam) == IDD_SETTING_DEFAULT ? S_default_setting
#else /* _WIN32 */
		     (wParam == IDD_SETTING_DEFAULT ? S_default_setting
#endif /* _WIN32 */
		      : S_startup_setting))
	    != IDOK)
	{
	  break;
	}
	// fall through
	
       case IDOK:
       case IDCANCEL:
#ifdef _WIN32
	this->idd = LOWORD(wParam);
#else /* _WIN32 */
	this->idd = wParam;
#endif /* _WIN32 */
	End(0);
	return(1);

       case IDD_SETTING_WIDTH:
#ifdef _WIN32
	this->check_size(LOWORD(wParam), &this->width);
#else /* _WIN32 */
	this->check_size(wParam, &this->width);
#endif /* _WIN32 */
	break;
	
       case IDD_SETTING_HEIGHT:
#ifdef _WIN32
	this->check_size(LOWORD(wParam), &this->height);
#else /* _WIN32 */
	this->check_size(wParam, &this->height);
#endif /* _WIN32 */
	break;
      }
    }
    break;
  }
  return(SUPER_CLASS::WndProc(iMessage, wParam, lParam));
}

// public constructor
NEAR
SettingDialog::SettingDialog(char* class_name,
			      char* box_name,
			      int width, int height, int max_size)
{
  this->class_name = class_name;
  this->box_name = box_name;
  this->busy = False;
  this->width = width;
  this->height = height;
  this->max_size = max_size;
  this->idd = IDOK;
}

// ------------------------------------------------------------
// implementation of MenuSelectDialog

#undef SUPER_CLASS
#define SUPER_CLASS PModalDialog

// virtual method based on class Window
LRESULT_T
MenuSelectDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam)
{
  switch (iMessage)
  {
   case WM_COMMAND:
#ifdef _WIN32
    if (IDM_BASE <= (int) LOWORD(wParam))
#else /* _WIN32 */
    if (IDM_BASE <= (int) wParam)
#endif /* _WIN32 */
    {
#ifdef _WIN32
      this->idm = LOWORD(wParam);
#else /* _WIN32 */
      this->idm = wParam;
#endif /* _WIN32 */
      End(1);
      return(1);
    }
    break;
  }
  return(SUPER_CLASS::WndProc(iMessage, wParam, lParam));
}

// public method
int NEAR
MenuSelectDialog::Go()
{
  SUPER_CLASS::Go();
  return(this->idm);
}

// public constructor
NEAR
MenuSelectDialog::MenuSelectDialog()
{
  this->class_name = "MenuSelectDialog";
  this->idm = 0;
}

// ------------------------------------------------------------
// implementation of TextButtonDialog

#undef SUPER_CLASS
#define SUPER_CLASS PModalDialog

// private method
void NEAR
TextButtonDialog::check_size(int id, int* place)
{
  int size;
  int ok = get_item_button_unit(this->GetHandle(), id, &size);
  if (ok)
  {
    *place = size;
    Box_Button* button = this->values.button;
    if (size < MIN_BUTTON_SIZE
	|| MAX_BUTTON_SIZE < size
	|| (this
	    ->values.box
	    ->check_button_overlap(button->x,
				    button->y,
				    this->values.width,
				    this->values.height, button)))
    {
      ok = False;
    }
  }
  EnableWindow(GetItem(IDOK), ok);
}

// private method
void NEAR
TextButtonDialog::set_listbox(int id,
			       Command_Desc* command, Command_Desc** table)
{
  int i = get_command_table_length();
  while (0 <= --i && table[i] != command)
  {
  }
  this->SendItemMessage(id, LB_SETCURSEL, i);
}

// private method
void NEAR
TextButtonDialog::set_guide(Command_Desc* command)
{
  this->set_listbox(IDD_TEXT_BUTTON_GUIDE,
		     command, get_command_by_guide_table());
}

// private method
void NEAR
TextButtonDialog::set_command(Command_Desc* command)
{
  this->set_listbox(IDD_TEXT_BUTTON_COMMAND,
		     command, get_command_by_name_table());
}

// private method
void NEAR
TextButtonDialog::set_idm(int idm)
{
  Command_Desc* command = command_by_idm(idm);
  if (command != NULL)
  {
    this->values.idm = idm;
    this->set_command(command);
    this->set_guide(command);
  }
}

// private method
void NEAR
TextButtonDialog::init_listbox(int id, Command_Desc** table)
{
  for (int length = get_command_table_length();
       length != 0; --length, ++table)
  {
    this->SendItemMessage(id,
			   LB_INSERTSTRING,
			   -1,
			   Ptr2Long(id == IDD_TEXT_BUTTON_COMMAND
				     ? table[0]->name : table[0]->guide));
  }
}

// virtual method based on class Window
LRESULT_T
TextButtonDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam)
{
  switch (iMessage)
  {
   case WM_INITDIALOG:
    this->busy = True;
    {
      HWND handle = this->GetHandle();
      this->SetItemText(IDD_TEXT_BUTTON_TEXT, this->values.text);
      this->SendItemMessage(IDD_TEXT_BUTTON_VERTICAL,
			     BM_SETCHECK, this->values.verticalp);
      set_item_button_unit(handle, IDD_TEXT_BUTTON_WIDTH, this->values.width);
      set_item_button_unit(handle, IDD_TEXT_BUTTON_HEIGHT, this->values.height);
      this->init_listbox(IDD_TEXT_BUTTON_COMMAND,
			  get_command_by_name_table());
      this->init_listbox(IDD_TEXT_BUTTON_GUIDE,
			  get_command_by_guide_table());
      this->set_idm(this->values.idm);
    }
    this->busy = False;
    return(1);
    
   case WM_COMMAND:
    if (!this->busy)
    {
#ifdef _WIN32
      switch (LOWORD(wParam))
#else /* _WIN32 */
      switch (wParam)
#endif /* _WIN32 */
      {
       case IDD_TEXT_BUTTON_COMMAND:
       case IDD_TEXT_BUTTON_GUIDE:
#ifdef _WIN32
	if (HIWORD(wParam) == LBN_SELCHANGE)
#else /* _WIN32 */
	if (HIWORD(lParam) == LBN_SELCHANGE)
#endif /* _WIN32 */
	{
#ifdef _WIN32
	  int i = (int) this->SendItemMessage(LOWORD(wParam), LB_GETCURSEL);
#else /* _WIN32 */
	  int i = (int) this->SendItemMessage(wParam, LB_GETCURSEL);
#endif /* _WIN32 */
	  if ((unsigned int) i < (unsigned int) get_command_table_length())
	  {
#ifdef _WIN32
	    Command_Desc* command = ((LOWORD(wParam) == IDD_TEXT_BUTTON_COMMAND
#else /* _WIN32 */
	    Command_Desc* command = ((wParam == IDD_TEXT_BUTTON_COMMAND
#endif /* _WIN32 */
				      ? get_command_by_name_table()
				      : get_command_by_guide_table())
				     [i]);
	    this->values.idm = command->idm_code;
#ifdef _WIN32
	    if (LOWORD(wParam) == IDD_TEXT_BUTTON_COMMAND)
#else /* _WIN32 */
	    if (wParam == IDD_TEXT_BUTTON_COMMAND)
#endif /* _WIN32 */
	    {
	      Command_Desc* command = (get_command_by_name_table())[i];
	      this->values.idm = command->idm_code;
	      this->set_guide(command);
	    }
	    else
	    {
	      Command_Desc* command = (get_command_by_guide_table())[i];
	      this->values.idm = command->idm_code;
	      this->set_command(command);
	    }
	  }
	}
	break;

       case IDD_TEXT_BUTTON_MENU:
	{
	  MenuSelectDialog dialog;
	  int idm = dialog.Go();
	  if (IDM_BASE < idm)
	  {
	    this->set_idm(idm);
	  }
	}
	break;

       case IDD_TEXT_BUTTON_TEXT:
	{
	  char text[MAX_BUTTON_TEXT_LENGTH + 1];
#ifdef _WIN32
	  GetItemText(LOWORD(wParam), text, MAX_BUTTON_TEXT_LENGTH);
#else /* _WIN32 */
	  GetItemText(wParam, text, MAX_BUTTON_TEXT_LENGTH);
#endif /* _WIN32 */
	  copy_string(this->values.text, text, MAX_BUTTON_TEXT_LENGTH + 1);
	}
	break;
	
       case IDD_TEXT_BUTTON_VERTICAL:
#ifdef _WIN32
	this->values.verticalp = get_item_check(LOWORD(wParam));
#else /* _WIN32 */
	this->values.verticalp = get_item_check(wParam);
#endif /* _WIN32 */
	break;

       case IDD_TEXT_BUTTON_WIDTH:
#ifdef _WIN32
	this->check_size(LOWORD(wParam), &this->values.width);
#else /* _WIN32 */
	this->check_size(wParam, &this->values.width);
#endif /* _WIN32 */
	break;

       case IDD_TEXT_BUTTON_HEIGHT:
#ifdef _WIN32
	this->check_size(LOWORD(wParam), &this->values.height);
#else /* _WIN32 */
	this->check_size(wParam, &this->values.height);
#endif /* _WIN32 */
	break;
      }
    }
    break;
  }
  return(SUPER_CLASS::WndProc(iMessage, wParam, lParam));
}

// public method
void NEAR
TextButtonDialog::get_values(TextButtonDialogValues* values)
{
  memcpy(values, &this->values, sizeof(*values));
}

// public constructor
NEAR
TextButtonDialog::TextButtonDialog(TextButtonDialogValues* values)
{
  this->class_name = "TextButtonDialog";
  this->busy = False;
  memcpy(&this->values, values, sizeof(*values));
}

// ------------------------------------------------------------
// implementation of Box

void NEAR
Box::move_toolbox(int x, int y)
{
  RECT in;
  RECT out;
  POINT p;
  this->get_client_rect(&in);
  this->get_window_rect(&out);
  p.x = in.left;
  p.y = in.top;
  this->client_to_screen(&p);
  int width = out.right - out.left;
  int height = out.bottom - out.top;
  int title = p.y - out.top;
  
  if (Screen::width < x)
  {
    x = Screen::width - width;
  }
  if (x + width <= 0)
  {
    x = 0;
  }
  
  if (Screen::height < y)
  {
    y = Screen::height - height;
  }
  if (y + title <= 0)
  {
    y = 0;
  }
  this->move_window(x, y, width, height);
}

// public method
void NEAR
Box::fit_toolbox_size()
{
  if (toolboxp(this->box_type))
  {
    SET_BOUND(this->width, MIN_TOOLBOX_SIZE, MAX_TOOLBOX_SIZE);
    SET_BOUND(this->height, MIN_TOOLBOX_SIZE, MAX_TOOLBOX_SIZE);
    // Calculate size of client area.
    int width = button_unit * this->width - 1;
    int height = button_unit * this->height - 1;
    RECT in;
    RECT out;
    this->get_client_rect(&in);
    this->get_window_rect(&out);
    this->move_window(out.left,
		       out.top,
		       width + ((out.right - out.left) - (in.right - in.left)),
		       (height
			+ ((out.bottom - out.top) - (in.bottom - in.top))));
    this->move_toolbox(out.left, out.top);
  }
  this->invalidate();
}

// private method
void NEAR
Box::redisplay_button(int n)
{
  Box_Button* button = &this->buttons[n];
  int x = button_unit * button->x;
  int y = button_unit * button->y;
  int width = button_unit * button->width;
  int height = button_unit * button->height;
  int state = button->state;
  HDC dc = this->dc;
  if (ribbonp(this->box_type))
  {
    // Ribbon has border lines in client area.
    ++x;
    ++y;
  }
  RECT frame[1];
  frame->left = x - 1;
  frame->right = x + width;
  frame->top = y - 1;
  frame->bottom = y + height;
  this->draw_frame(frame);
  
  RECT clip[1];
  ensmall_rect(clip, frame, 1);

  DWORD background;
  RECT background_rect[1];
  if (state == 0)
  {
    // popped button
    for (int shadow = (button_unit == BUTTON_SIZE_SMALL ? 1 : 2);
	 shadow != 0; --shadow)
    {
      RECT r[1];
      ensmall_rect(frame, frame, 1);
      *r = *frame;
      r->bottom = r->top + 1;
      fill_rect(dc, r, Screen::stocked_white_brush);
      *r = *frame;
      r->right = r->left + 1;
      fill_rect(dc, r, Screen::stocked_white_brush);
      *r = *frame;
      r->top = r->bottom - 1;
      fill_color_rect(dc, r, Screen::button_shadow_color);
      *r = *frame;
      r->left = r->right - 1;
      fill_color_rect(dc, r, Screen::button_shadow_color);
    }
    ensmall_rect(frame, frame, 1);
    background = Screen::button_face_color;
    *background_rect = *frame;
  }
  else
  {
    // state == 1 for pushed button.
    background = Screen::pushed_button_color;
    *background_rect = *clip;
  }
  
  if (text_button_p(button->button_type))
  {
    fill_solid_color_rect(dc, background_rect, background);
    draw_text(dc,
	       button->u.text,
	       clip,
	       DT_CENTER,
	       vertical_text_button_p(button->button_type),
	       (state == 0 ? Screen::button_text_color
		: Screen::pushed_text_color));
  }
  else
  {
    Bitmap_Button* bb = button->u.bb;
    HICON icon = bb->icon_cache;
    if (icon == 0)
    {
      char icon_name[20];
      strcpy(icon_name, bb->bitmap_name);
      strcat(icon_name, button_unit == BUTTON_SIZE_LARGE ? "27" : "17");
      icon = ::LoadIcon(PWordPresentation::hInstance, icon_name);
      if (icon == 0)
      {
#ifdef BW2_SUPPORT_BITMAP_BUTTON
	char bitmap_name[64];
	char suffix[3];
	suffix[0] = '0' + button_unit;
	suffix[1] = 'a' + button->state;
	suffix[2] = '\0';
	strcpy(bitmap_name, bb->bitmap_name);
	strcat(bitmap_name, suffix);
	HBITMAP bitmap = LoadBitmap(PWordPresentation::hInstance,
				     bitmap_name);
	if (bitmap == 0)
	{
	  return;
	}
	HDC memory_dc = CreateCompatibleDC(dc);
	bitmap = select_object(memory_dc, bitmap);
	BitBlt(dc, x, y, width, height, memory_dc, 0, 0, SRCCOPY);
	delete_object(select_object(memory_dc, bitmap));
	::DeleteDC(memory_dc);
#endif /* BW2_SUPPORT_BITMAP_BUTTON */
	return;
      }
      bb->icon_cache = icon;
    }
    fill_solid_color_rect(dc,
			   background_rect,
			   background == BLACK ? WHITE : background);
    ensmall_rect(clip,
		  clip,
		  (button_unit == BUTTON_SIZE_SMALL ? 1
		   : button_unit == BUTTON_SIZE_MEDIUM ? 3 : 2));
    int save_id = save_dc(dc);
    intersect_clip_rect(dc, clip);
    ::DrawIcon(dc, clip->left, clip->top, icon);
    restore_dc(dc, save_id);
    if (background == BLACK)
    {
      int old_rop2 = SetROP2(dc, R2_NOT);
      HPEN old_pen = (HPEN)select_object(dc, GetStockObject(NULL_PEN));
      Rectangle(dc,
		 background_rect->left,
		 background_rect->top,
		 background_rect->right + 1, background_rect->bottom + 1);
      select_object(dc, old_pen);
      SetROP2(dc, old_rop2);
    }
    if (!Pref_cache_buttons)
    {
      bb->icon_cache = 0;
      destroy_icon(icon);
    }
  }
}

void NEAR
Box::drawing_epilogue()
{
  if (this->dc)
  {
    ReleaseDC(this->GetHandle(), this->dc);
    this->dc = 0;
  }
}

void NEAR
Box::drawing_prologue()
{
  if (this->dc == 0)
  {
    this->dc = GetDC(this->GetHandle());
    set_bk_mode_transparent(this->dc);
    if (ribbonp(this->box_type))
    {
      RECT r[1];
      this->get_client_rect(r);
      if (ensmall_rect(r, r, 1))
      {
	intersect_clip_rect(this->dc, r);
      }
    }
  }
}

void NEAR
Box::change_button_state(int i, int state)
{
  if ((unsigned int) i < (unsigned int) this->num_buttons)
  {
    if (this->buttons[i].state != state)
    {
      this->buttons[i].state = state;
      this->drawing_prologue();
      this->redisplay_button(i);
      this->drawing_epilogue();
    }
  }
}

void NEAR
Box::restore_button_state(int i)
{
  if ((unsigned int) i < (unsigned int) this->num_buttons)
  {
    Box_Button* button = &this->buttons[i];
    int state = text_button_p(button->button_type) ? 0 : button->u.bb->state;
    if (button->state != state)
    {
      this->change_button_state(i, state);
    }
  }
}

void NEAR
Box::point_to_xy(POINT* p, int* x_ret, int* y_ret)
{
  *x_ret = p->x / button_unit;
  *y_ret = p->y / button_unit;
}

// This method returns index of "this->buttons" pointed by P.
// If button is not found, returns -1.
int NEAR
Box::find_button(POINT* p)
{
  int x;
  int y;
  this->point_to_xy(p, &x, &y);
  int i = this->num_buttons;
  Box_Button* button = this->buttons + i;
  while (0 <= --i)
  {
    --button;
    int button_x = button->x;
    int button_y = button->y;
    if (button_x <= x
	&& x < button_x + button->width
	&& button_y <= y && y < button_y + button->height)
    {
      break;
    }
  }
  return(i);
}

void NEAR
Box::set_ribbon_size(int size)
{
  SET_BOUND(size, MIN_RIBBON_SIZE, MAX_RIBBON_SIZE);
  this->width = size;
  this->height = size;
  this->changed = True;
}

int NEAR
Box::check_button_overlap(int x,
			   int y,
			   int width, int height, Box_Button* ignored_button)
{
  int bot_x = x + width;
  int bot_y = y + height;
  Box_Button* button = this->buttons;
  int n = this->num_buttons;
  for (; n != 0; --n, ++button)
  {
    int button_x = button->x;
    int button_y = button->y;
    if (button_x < bot_x
	&& x < button_x + button->width
	&& button_y < bot_y
	&& y < button_y + button->height && button != ignored_button)
    {
      return(True);
    }
  }
  return(False);
}

void NEAR
Box::remove_button(int button_index)
{
  int num_buttons = this->num_buttons;
  if ((unsigned int) button_index < (unsigned int) num_buttons)
  {
    Box_Button* buttons = this->buttons;
    Box_Button* button = &buttons[button_index];
    if (text_button_p(button->button_type))
    {
      xfree(button->u.text);
    }
    --num_buttons;
    this->num_buttons = num_buttons;
    for (int i = button_index; i != num_buttons; ++i)
    {
      buttons[i] = buttons[i + 1];
    }
  }
}

int NEAR
Box::add_button(Box_Button* button)
{
  SET_MAX(button->x, 0);
  SET_MAX(button->y, 0);
  SET_BOUND(button->width, MIN_BUTTON_SIZE, MAX_BUTTON_SIZE);
  SET_BOUND(button->height, MIN_BUTTON_SIZE, MAX_BUTTON_SIZE);
  if (MAX_BUTTONS <= this->num_buttons
      || (this->check_button_overlap(button->x,
				      button->y,
				      button->width, button->height, NULL)))
  {
    return(False);
  }
  
  Box_Button* buttons = this->buttons;
  int n = this->num_buttons;
  int num_buttons = n + 1;
  if (xmalloc_size(buttons) / sizeof(Box_Button)
      < (unsigned int) num_buttons)
  {
    buttons = ((Box_Button*)
	       xrealloc(buttons, sizeof(Box_Button) * num_buttons));
    this->buttons = buttons;
  }
  this->num_buttons = num_buttons;
  buttons[n] = *button;
  return(True);
}

void NEAR
Box::add_bitmap_button(int button_id, int x, int y)
{
  if ((unsigned int) button_id < BB_UNUSED_MAX)
  {
    Box_Button button;
    Bitmap_Button* bb = &bitmap_button[button_id];
    button.button_type = Button_Type_BITMAP;
    button.u.bb = bb;
    button.idm = bb->idm;
    button.x = x;
    button.y = y;
    button.width = UNITS_PER_BUTTON;
    button.height = UNITS_PER_BUTTON;
    button.state = bb->state;
    this->add_button(&button);
  }
}

void NEAR
Box::no_mouse_operation()
{
  this->mouse_operation = NO_MOUSE_OPERATION;
  ::ReleaseCapture();
}

struct Customize_Data
{
  int activep;
  int cursor_index;
  int button_index;
  Box* box;
  int x;
  int y;
  POINT mouse_down_point;
};

static Customize_Data STATIC_NEAR cust = { False, };

static void NEAR
customize_set_cursor(int cursor_index)
{
  if (cust.cursor_index != cursor_index)
  {
    cust.cursor_index = cursor_index;
    SetCursor(cursor_index < 0 ? arrow_cursor() : cursors[cursor_index]);
  }
}

void NEAR
Box::customize_cancel(int do_beep_p)
{
  if (cust.activep)
  {
    this->no_mouse_operation();
    customize_set_cursor(-1);
    cust.activep = False;
    beep(do_beep_p);
  }
}

void NEAR
Box::customize_setting()
{
  char* cannot_target = NULL;
  if (0 <= cust.button_index
      && text_button_p(this->buttons[cust.button_index].button_type))
  {
    // setting for text button
    if (!this->customizablep)
    {
      cannot_target = S_text_button;
    }
    else
    {
      Box_Button* button = &this->buttons[cust.button_index];
      TextButtonDialogValues values;
      values.box = this;
      values.button = button;
      values.idm = button->idm;
      values.width = button->width;
      values.height = button->height;
      values.verticalp = vertical_text_button_p(button->button_type);
      copy_string(values.text, button->u.text, MAX_BUTTON_TEXT_LENGTH + 1);
      int result;
      {
	TextButtonDialog dialog(&values);
	result = dialog.Go();
	dialog.get_values(&values);
      }
      if (result != 0)
      {
	xfree(button->u.text);
	button->u.text = dup_string(values.text);
	button->button_type = (values.verticalp ? Button_Type_VERTICAL_TEXT
			       : Button_Type_TEXT);
	button->idm = values.idm;
	button->width = values.width;
	button->height = values.height;
	SET_BOUND(button->width, MIN_BUTTON_SIZE, MAX_BUTTON_SIZE);
	SET_BOUND(button->height, MIN_BUTTON_SIZE, MAX_BUTTON_SIZE);
	this->invalidate();
      }
    }
  }
  else if (!this->customizablep)
  {
    cannot_target = S_toolbox;
  }
  else
  {
    char* class_name = "RibbonSettingDialog";
    int height = 0;
    int max_size = MAX_RIBBON_SIZE;
    if (toolboxp(this->box_type))
    {
      class_name = "ToolboxSettingDialog";
      height = this->height;
      max_size = MAX_TOOLBOX_SIZE;
    }
    SettingDialog dialog(class_name,
			  this->name, this->width, height, max_size);
    dialog.Go();
    switch (dialog.idd)
    {
     case IDOK:
      if (toolboxp(this->box_type))
      {
	if (this->width != dialog.width || this->height != dialog.height)
	{
	  this->width = dialog.width;
	  this->height = dialog.height;
	  this->changed = True;
	  this->fit_toolbox_size();
	}
      }
      else
      {
	if (this->width != dialog.width)
	{
	  this->set_ribbon_size(dialog.width);
	  position_windows();
	}
      }
      break;

     case IDD_SETTING_STARTUP:
      if (!Pref::begin_profile())
      {
	PWordBase::no_more_memory();
	break;
      }
      // fall through
      
     case IDD_SETTING_DEFAULT:
      this->changed = True;
      while (0 < this->num_buttons)
      {
	this->remove_button(this->num_buttons - 1);
      }
      disable_some_profile = True;
      if (dialog.idd == IDD_SETTING_DEFAULT)
      {
	this->parse_builtin_profile(this->builtin_profile);
      }
      else
      {
	this->parse_profile();
	Pref::end_profile();
      }
      disable_some_profile = False;
      this->fit_toolbox_size();
      position_windows();
    }
  }
  
  if (cannot_target != NULL)
  {
    IssueVA(SF_cannot_customize, MB_OK | MB_ICONEXCLAMATION, cannot_target);
  }
}

void NEAR
Box::customize_move(LPARAM_T lParam)
{
  POINT point;

#ifdef _WIN32
  LPARAM_TO_POINT(lParam, point);
#else /* _WIN32 */
  point = MAKEPOINT(lParam);
#endif /* _WIN32 */

  if (cust.cursor_index == CURSOR_SETTING
      && (cust.button_index < 0
	  || (cust.mouse_down_point.x - button_unit < point.x
	      && point.x < cust.mouse_down_point.x + button_unit
	      && cust.mouse_down_point.y - button_unit < point.y
	      && point.y < cust.mouse_down_point.y + button_unit)))
  {
    return;
  }
  
  this->client_to_screen(&point);
  int cursor_index = this->customizablep ? CURSOR_TRASH : CURSOR_NO;
  HWND handle = ::WindowFromPoint(point);
  if (handle != 0)
  {
    for (int i = BOX_BEGIN; i != BOX_END; ++i)
    {
      Box* box = boxes[i];
      if (box->shownp && box->GetHandle() == handle)
      {
	RECT r[1];
	::ScreenToClient(handle, &point);
	box->get_client_rect(r);
	if (::PtInRect(r, point))
	{
	  cursor_index = CURSOR_NO;
	  if (box->customizablep)
	  {
	    cust.box = box;
	    int x = ((point.x + button_unit / 2) / button_unit
		     - (UNITS_PER_BUTTON / 2));
	    int y = ((point.y + button_unit / 2) / button_unit
		     - (UNITS_PER_BUTTON / 2));
	    if (0 <= x && 0 <= y)
	    {
	      cust.x = x;
	      cust.y = y;
	      Box_Button* button = &this->buttons[cust.button_index];
	      if (!box->check_button_overlap(x,
					      y,
					      button->width,
					      button->height, button))
	      {
		cursor_index = CURSOR_OK;
	      }
	    }
	  }
	}
	break;
      }
    }
  }
  customize_set_cursor(cursor_index);
}

void NEAR
Box::customize(MSG_T message, WPARAM_T wParam, LPARAM_T lParam)
{
  switch (message)
  {
   default:
    this->customize_cancel(True);
    break;
    
   case WM_LBUTTONDBLCLK:
   case WM_LBUTTONDOWN:
    if (cust.activep)
    {
      this->customize_cancel(True);
    }
    else if (!global_customizable_p)
    {
      beep(True);
    }
    else
    {
      ::SetCapture(this->GetHandle());
#ifdef _WIN32
      LPARAM_TO_POINT(lParam, cust.mouse_down_point);
#else /* _WIN32 */
      cust.mouse_down_point = MAKEPOINT(lParam);
#endif /* _WIN32 */
      cust.button_index = ((wParam & MK_SHIFT) != 0 ? -1
			   : this->find_button(&cust.mouse_down_point));
      this->mouse_operation = CUSTOMIZE_MOUSE_OPERATION;
      cust.activep = True;
      cust.cursor_index = -1;
      customize_set_cursor(CURSOR_SETTING);
    }
    break;
    
   case WM_NCMOUSEMOVE:
   case WM_MOUSEMOVE:
    this->customize_move(lParam);
    break;
    
   case WM_NCLBUTTONUP:
   case WM_LBUTTONUP:
    {
      this->customize_move(lParam);
      int do_beep_p = False;
      switch (cust.cursor_index)
      {
       case CURSOR_OK:
	if (MAX_BUTTONS <= cust.box->num_buttons)
	{
	  ::StatusOut("Too many buttons on ribbon.");
	  beep(True);
	}
	else
	{
	  Box_Button button_rec = this->buttons[cust.button_index];
	  if (text_button_p(button_rec.button_type))
	  {
	    button_rec.u.text = dup_string(button_rec.u.text);
	  }
	  if (this->customizablep
#ifdef COPY_BUTTON
	      && cust.box == this
	      && this->check_button_overlap(cust.x,
					     cust.y,
					     button_rec.width,
					     button_rec.height, NULL)
#else /* not COPY_BUTTON */
#endif /* not COPY_BUTTON */
	    )
	  {
	    this->remove_button(cust.button_index);
	    this->invalidate();
	    this->changed = True;
	  }
	  button_rec.x = cust.x;
	  button_rec.y = cust.y;
	  cust.box->add_button(&button_rec);
	  cust.box->invalidate();
	  cust.box->changed = True;
	}
	break;
	
       case CURSOR_TRASH:
	this->remove_button(cust.button_index);
	this->invalidate();
	this->changed = True;
	break;
	
       case CURSOR_SETTING:
	this->customize_cancel(False);
	this->customize_setting();
	break;
	
       default:
	do_beep_p = True;
	break;
      }
      this->customize_cancel(do_beep_p);
    }
    break;
  }
}

void NEAR
Box::inactivate_mouse(int do_beep_p)
{
  switch (this->mouse_operation)
  {
   case CUSTOMIZE_MOUSE_OPERATION:
    this->customize_cancel(True);
    break;
    
   case NORMAL_MOUSE_OPERATION:
    no_mouse_operation();
    if (0 <= this->mouse_down_button)
    {
      this->restore_button_state(this->mouse_down_button);
      this->mouse_down_button = -1;
      this->mouse_down_button_active_p = False;
    }
    beep(do_beep_p);
    break;
  }
}

// private method
void NEAR
Box::twist_toolbox()
{
  Box_Type box_type = this->box_type;
  if (toolboxp(box_type))
  {
    int foo;
    this->box_type = (twisted_toolbox_p(box_type) ? Box_Type_TOOLBOX
		      : Box_Type_TWISTED_TOOLBOX);
    // swap width and height
    foo = this->width;
    this->width = this->height;
    this->height = foo;

    // swap x and y, width and height in all buttons.
    int i = this->num_buttons;
    Box_Button* button = this->buttons;
    for (; i != 0; --i, ++button)
    {
      foo = button->x;
      button->x = button->y;
      button->y = foo;
      foo = button->width;
      button->width = button->height;
      button->height = foo;
      switch (button->button_type)
      {
       case Button_Type_TEXT:
	button->button_type = Button_Type_VERTICAL_TEXT;
	break;
	
       case Button_Type_VERTICAL_TEXT:
	button->button_type = Button_Type_TEXT;
	break;
      }
    }
    this->fit_toolbox_size();
  }
}

// virtual method
LRESULT_T
Box::WndProc(MSG_T message, WPARAM_T wParam, LPARAM_T lParam)
{
  static int STATIC_NEAR kill_flag = False;
  
  HWND handle = this->GetHandle();
  if (handle)
  {
    if (toolboxp(this->box_type))
    {
      switch (message)
      {
       case WM_MOUSEACTIVATE:
	return(MA_NOACTIVATE);
	
       case WM_NCACTIVATE:
	if (wParam == 0)
	{
	  wParam = 1;
	}
	break;
	
       case WM_ACTIVATEAPP:
	kill_flag = (wParam == 0);
	DefWindowProc(WM_NCACTIVATE, wParam, 0);
	return(0);
	
       case WM_KILLFOCUS:
	if (kill_flag)
	{
	  DefWindowProc(message, wParam, lParam);
	  return(DefWindowProc(WM_NCACTIVATE, 0, 0));
	}
	break;
	
       case WM_SETFOCUS:
	PostMessage(main_window_handle, WM_SETFOCUS, (WPARAM_T)this->GetHandle(), 0);
	return(0);
      }
    }
    switch (message)
    {
     case WM_PAINT:
      {
	PAINTSTRUCT ps;
	this->begin_paint(&ps);
	set_bk_mode_transparent(this->dc);
	if (ribbonp(this->box_type))
	{
	  this->draw_border();
	}
	else
	{
	  this->fill_background();
	}
	for (int n = 0; n < this->num_buttons; ++n)
	{
	  this->redisplay_button(n);
	}
	this->end_paint(&ps);
      }
      return(0);
      
     case WM_NCRBUTTONDBLCLK:
     case WM_NCRBUTTONDOWN:
     case WM_RBUTTONDBLCLK:
     case WM_RBUTTONDOWN:
      if (this->mouse_operation != NO_MOUSE_OPERATION)
      {
	this->inactivate_mouse(True);
      }
      else if (PModalDialog::Open())
      {
	beep(True);
      }
      else
      {
	// In case of ribbon, WM_PARENTNOTIFY message is sent to main
	// window.
	if (toolboxp(this->box_type))
	{
	  Boxes::toggle_global_visible();
	}
      }
      break;
      
     case WM_NCLBUTTONDBLCLK:
     case WM_NCLBUTTONDOWN:
      if ((::GetKeyState(VK_CONTROL) & 0x8000) == 0)
      {
	if (message == WM_NCLBUTTONDBLCLK)
	{
	  this->twist_toolbox();
	}
	break;
      }
      message = WM_LBUTTONDOWN;
      wParam = ((::GetKeyState(VK_SHIFT) & 0x8000) ? (MK_SHIFT | MK_CONTROL)
		: MK_CONTROL);
      lParam = MAKELONG(-button_unit, -button_unit);
      // fall through
      
     case WM_LBUTTONDBLCLK:
     case WM_LBUTTONDOWN:
      if (this->mouse_operation != NO_MOUSE_OPERATION)
      {
	this->inactivate_mouse(True);
      }
      else if (PModalDialog::Open())
      {
	beep(True);
      }
      switch (wParam & (MK_SHIFT | MK_CONTROL | MK_RBUTTON | MK_MBUTTON))
      {
       case MK_CONTROL:
       case MK_CONTROL | MK_SHIFT:
	this->customize(message, wParam, lParam);
	break;

       case 0:
	{
#ifdef _WIN32
	  POINT p;
	  LPARAM_TO_POINT(lParam, p);
	  int i = this->find_button(&p);
#else /* _WIN32 */
	  int i = this->find_button(&MAKEPOINT(lParam));
#endif /* _WIN32 */
	  if (0 <= i)
	  {
	    ::SetCapture(handle);
	    this->mouse_operation = NORMAL_MOUSE_OPERATION;
	    this->mouse_down_button = i;
	    this->mouse_down_button_active_p = True;
	    this->change_button_state(i, 1);
	  }
	}
	break;
      }
      break;

     case WM_NCLBUTTONUP:
     case WM_NCMOUSEMOVE:
     case WM_LBUTTONUP:
     case WM_MOUSEMOVE:
      switch (this->mouse_operation)
      {
       case CUSTOMIZE_MOUSE_OPERATION:
	this->customize(message, wParam, lParam);
	break;
	
       case NORMAL_MOUSE_OPERATION:
	{
	  int x;
	  int y;
#ifdef _WIN32
	  POINT p;
	  LPARAM_TO_POINT(lParam, p);
	  this->point_to_xy(&p, &x, &y);
#else /* _WIN32 */
	  this->point_to_xy(&MAKEPOINT(lParam), &x, &y);
#endif /* _WIN32 */
	  int i = this->mouse_down_button;
	  Box_Button* button = &this->buttons[i];
	  int button_x = button->x;
	  int button_y = button->y;
	  int activep = (button_x <= x
			 && x < button_x + button->width
			 && button_y <= y && y < button_y + button->height);
	  if (this->mouse_down_button_active_p != activep)
	  {
	    this->mouse_down_button_active_p = activep;
	    this->change_button_state(i, activep);
	  }
	  
	  if (message == WM_LBUTTONUP)
	  {
	    this->inactivate_mouse(False);
	    if (activep)
	    {
	      dispatch_simple_command(button->idm);
	    }
	  }
	}
	break;

       default:
	if (Pref_guide_button
	    && (message == WM_LBUTTONUP || message == WM_MOUSEMOVE))
	{
#ifdef _WIN32
	  POINT p;
	  LPARAM_TO_POINT(lParam, p);
	  int i = this->find_button(&p);
#else /* _WIN32 */
	  int i = this->find_button(&MAKEPOINT(lParam));
#endif /* _WIN32 */
	  Boxes::guide(i < 0 ? 0 : this->buttons[i].idm);
	}
      }
      break;
    }
  }
  return(Super_Box::WndProc(message, wParam, lParam));
}

#define EXPR_NIL 0
#define EXPR_XY 1
#define EXPR_TWIST 2
#define EXPR_VISIBLE 3
#define EXPR_SIZE 4
#define EXPR_TEXT_BUTTON 5

struct Enum_Desc op_desc[] =
{
  { "visible", EXPR_VISIBLE, },
  { "xy", EXPR_XY, },
  { "size", EXPR_SIZE, },
  { "twist", EXPR_TWIST, },
  { "text-button", EXPR_TEXT_BUTTON, },
  { NULL, EXPR_NIL, },
};

static int STATIC_NEAR expr_string_n;
static int STATIC_NEAR expr_number;
static char* STATIC_NEAR expr_key = NULL;

int NEAR
Box::expr_next_key()
{
  if (expr_string_n == 0)
  {
    expr_key = (char*) xmalloc(strlen(this->keyword) + 1 + 2 + 1);
  }
  else
  {
    if (expr_key != NULL)
    {
      Pref::flush_expr(expr_key);
    }
    if (NUM_PROFILE_STRINGS <= expr_string_n)
    {
      if (expr_key != NULL)
      {
	xfree(expr_key);
	expr_key = NULL;
      }
      return(False);
    }
  }
  wsprintf(expr_key, "%s-%d", this->keyword, expr_string_n);
  ++expr_string_n;
  Pref::start_write_expr();
  return(True);
}

void NEAR
Box::flush_expr()
{
  while (this->expr_next_key())
  {
  }
}

void NEAR
Box::write_expr(Pref_Expr* expr)
{
  if (expr_number == 0)
  {
    if (!this->expr_next_key())
    {
      return;
    }
  }
  Pref::write_expr(expr);
  ++expr_number;
  if (50 <= expr_number)
  {
    expr_number = 0;
  }
}

void NEAR
Box::write_simple_expr(char* op, int num_args, int arg0, int arg1)
{
  Pref_Expr expr;
  expr.op = op;
  expr.num_args = num_args;
  expr.args[0].type = Pref_Value_INTEGER;
  expr.args[0].u.i = arg0;
  expr.args[1].type = Pref_Value_INTEGER;
  expr.args[1].u.i = arg1;
  this->write_expr(&expr);
}

void NEAR
Box::write_profile()
{
  expr_string_n = 0;
  expr_number = 0;
  write_simple_expr("visible", 1, this->visiblep, 0);
  Box_Type box_type = this->box_type;
  if (toolboxp(box_type))
  {
    RECT r;
    this->get_window_rect(&r);
    write_simple_expr("xy", 2, r.left, r.top);
  }
  write_simple_expr("size", 2, this->width, this->height);
  if (twisted_toolbox_p(box_type))
  {
    write_simple_expr("twist", 0, 0, 0);
  }
  int i = 0;
  Box_Button* button = this->buttons;
  for (; i != this->num_buttons; ++i, ++button)
  {
    Command_Desc* command = command_by_idm(button->idm);
    if (command != NULL)
    {
      if (bitmap_button_p(button->button_type))
      {
	write_simple_expr(command->name, 2, button->x, button->y);
      }
      else
      {
	// text-button(text, command, x, y, width, height, vertical)
	Pref_Expr expr;
	expr.op = "text-button";
	expr.num_args = 7;
	for (int n = 2; n != 7; ++n)
	{
	  expr.args[n].type = Pref_Value_INTEGER;
	}
	expr.args[0].type = Pref_Value_STRING;
	expr.args[0].u.s = button->u.text;
	expr.args[1].type = Pref_Value_STRING;
	expr.args[1].u.s = command->name;
	expr.args[2].u.i = button->x;
	expr.args[3].u.i = button->y;
	expr.args[4].u.i = button->width;
	expr.args[5].u.i = button->height;
	expr.args[6].u.i = vertical_text_button_p(button->button_type);
	this->write_expr(&expr);
      }
    }
  }
  this->flush_expr();
}

int NEAR
Box::parse_builtin_profile(int* profile)
{
  int count = 0;
  if (profile != NULL)
  {
    Box_Type type = this->box_type;
    int x = 0;
    int y = 0;
    int* x_ptr = &x;
    int* y_ptr = &y;
    int* width_ptr = &this->width;
    for (;;)
    {
      int op = *profile++;
      if (op == BB_END)
      {
	break;
      }
      switch (op)
      {
       case BB_VISIBLE:
	if (!disable_some_profile)
	{
	  this->visiblep = True;
	}
	break;
	
       case BB_SIZE:
	this->width = *profile++;
	if (toolboxp(type))
	{
	  this->height = *profile++;
	  this->fit_toolbox_size();
	}
	else
	{
	  this->height = this->width;
	}
	break;
	
       case BB_VERTICAL:
	x_ptr = &y;
	y_ptr = &x;
	width_ptr = &this->height;
	break;
	
       case BB_SPACE:
	*x_ptr += 1;
	break;

       case BB_NEWLINE:
	*x_ptr = 0;
	*y_ptr += UNITS_PER_BUTTON;
	break;
	
       case BB_INITIAL_TEXT_BUTTON:
	{
	  Box_Button button;
	  button.button_type = Button_Type_TEXT;
	  button.u.text = dup_string(S_text_button);
	  button.idm = IDM_NoOperation;
	  button.x = x;
	  button.y = y;
	  button.width = G(5);
	  button.height = G(1);
	  button.state = 0;
	  this->add_button(&button);
	}
	x += G(5);
	break;
	
       case BB_MINCHO_TEXT_BUTTON:
	{
	  Box_Button button;
	  button.button_type = Button_Type_TEXT;
	  button.u.text = dup_string(S_min);
	  button.idm = IDM_More_VirtualMincho;
	  button.x = x;
	  button.y = y;
	  button.width = G(1);
	  button.height = G(1);
	  button.state = 0;
	  this->add_button(&button);
	}
	*x_ptr += G(1);
	break;
	
       case BB_GOTHIC_TEXT_BUTTON:
	{
	  Box_Button button;
	  button.button_type = Button_Type_TEXT;
	  button.u.text = dup_string(S_go);
	  button.idm = IDM_More_VirtualGothic;
	  button.x = x;
	  button.y = y;
	  button.width = G(1);
	  button.height = G(1);
	  button.state = 0;
	  this->add_button(&button);
	}
	*x_ptr += G(1);
	break;
	
       default:
	if (toolboxp(type) && *width_ptr <= *x_ptr)
	{
	  *x_ptr = 0;
	  *y_ptr += UNITS_PER_BUTTON;
	}
	this->add_bitmap_button(op, x, y);
	*x_ptr += UNITS_PER_BUTTON;
	break;
      }
    }
  }
  return(count);
}

int NEAR
Box::parse_profile()
{
  if (this->initial_profile[0] == NULL)
  {
    return(this->parse_builtin_profile(this->builtin_profile));
  }

  Box_Type box_type = this->box_type;
  int customizablep = this->customizablep;
  int count = 0;
  for (int i = 0; i != NUM_PROFILE_STRINGS; ++i)
  {
    Pref::set_expr(this->initial_profile[i]);
    Pref_Expr expr;
    while (Pref::read_expr(&expr))
    {
      int op = Pref::enum_value(op_desc, expr.op);
      int arg0 = 1;
      int arg1 = 1;
      int num_integer_args = expr.num_args;
      switch (num_integer_args)
      {
       default:
	if (expr.args[1].type == Pref_Value_INTEGER)
	{
	  arg1 = expr.args[1].u.i;
	}
	else
	{
	  num_integer_args = 1;
	}
	// fall through
	
       case 1:
	if (expr.args[0].type == Pref_Value_INTEGER)
	{
	  arg0 = expr.args[0].u.i;
	}
	else
	{
	  num_integer_args = 0;
	}
	// fall through
	
       case 0:
	break;
      }
      switch (op)
      {
       case EXPR_VISIBLE:
	if (!disable_some_profile)
	{
	  this->visiblep = (arg0 != 0);
	}
	++count;
	break;
	
       case EXPR_XY:
	if (toolboxp(box_type) && 2 <= num_integer_args)
	{
	  if (!disable_some_profile)
	  {
	    this->move_toolbox(arg0, arg1);
	  }
	  ++count;
	}
	break;
	
       case EXPR_SIZE:
	if (ribbonp(box_type))
	{
	  if (1 <= num_integer_args)
	  {
	    this->set_ribbon_size(arg0);
	    ++count;
	  }
	}
	else if (customizablep && 2 <= num_integer_args)
	{
	  this->width = arg0;
	  this->height = arg1;
	  this->fit_toolbox_size();
	  ++count;
	}
	break;
	
       case EXPR_TWIST:
	if (toolboxp(box_type) && !customizablep)
	{
	  this->twist_toolbox();
	  ++count;
	}
	break;
	
       case EXPR_TEXT_BUTTON:
	if (customizablep
	    && expr.num_args == 7
	    && expr.args[0].type == Pref_Value_STRING
	    && expr.args[1].type == Pref_Value_STRING
	    && expr.args[2].type == Pref_Value_INTEGER
	    && expr.args[3].type == Pref_Value_INTEGER
	    && expr.args[4].type == Pref_Value_INTEGER
	    && expr.args[5].type == Pref_Value_INTEGER
	    && expr.args[6].type == Pref_Value_INTEGER)
	{
	  // text-button(text, command, x, y, width, height, vertical)
	  Command_Desc* command = command_by_name(expr.args[1].u.s);
	  if (command != NULL)
	  {
	    Box_Button button;
	    button.button_type = (expr.args[6].u.i == 0 ? Button_Type_TEXT
				  : Button_Type_VERTICAL_TEXT);
	    button.u.text = dup_string(expr.args[0].u.s);
	    button.idm = command->idm_code;
	    button.x = expr.args[2].u.i;
	    button.y = expr.args[3].u.i;
	    button.width = expr.args[4].u.i;
	    button.height = expr.args[5].u.i;
	    button.state = 0;
	    this->add_button(&button);
	    ++count;
	  }
	}
	break;
	    
       default:
	if (customizablep && 2 <= num_integer_args)
	{
	  Command_Desc* command = command_by_name(expr.op);
	  if (command != NULL)
	  {
	    int idm = command->idm_code;
	    int n = 0;
	    Bitmap_Button* bb = bitmap_button;
	    for (; n != BB_UNUSED_MAX; ++n, ++bb)
	    {
	      if (bb->idm == idm)
	      {
		this->add_bitmap_button(n, arg0, arg1);
		++count;
		break;
	      }
	    }
	  }
	}
      }
    }
  }
  return(count);
}

int NEAR
Box::read_profile()
{
  char* key = (char *) xmalloc(strlen(this->keyword) + 1 + 2 + 1);
  for (int i = 0; i != NUM_PROFILE_STRINGS; ++i)
  {
    wsprintf(key, "%s-%d", this->keyword, i);
    char* s = Pref::read_string(key);
    this->initial_profile[i] = dup_string(s);
  }
  xfree(key);
  return(this->parse_profile());
}

// constructor
NEAR
Box::Box(char* keyword, char* name, Box_Type box_type, int* profile)
  : Super_Box(box_type)
{
  this->keyword = keyword;
  this->name = name;
  this->width = DEFAULT_RIBBON_SIZE;
  this->height = DEFAULT_RIBBON_SIZE;
  this->customizablep = True;
  this->num_buttons = 0;
  this->buttons = NULL;
  this->mouse_operation = NO_MOUSE_OPERATION;
  this->mouse_down_button = -1;
  this->mouse_down_button_active_p = False;
  this->builtin_profile = profile;
  for (int i = 0; i != NUM_PROFILE_STRINGS; ++i)
  {
    this->initial_profile[i] = NULL;
  }
  this->changed = False;
}

// static method
Box* NEAR
Box::make_ribbon(char* keyword, char* name, int* profile)
{
  return(new Box(keyword, name, Box_Type_RIBBON, profile));
}

// static method
Box* NEAR
Box::make_toolbox(char* keyword, char* name, int* profile)
{
  Box* box = new Box(keyword, name, Box_Type_TOOLBOX, profile);
  box->fit_toolbox_size();
  return(box);
}

static void NEAR
kill_cursors()
{
  for (int i = 0; i != NUM_CURSORS; ++i)
  {
    HCURSOR cursor = cursors[i];
    if (cursor)
    {
      cursors[i] = 0;
      DestroyCursor(cursor);
    }
  }
}

static void NEAR
load_cursors()
{
  kill_cursors();
  if (global_customizable_p)
  {
    char name[4];
    name[0] = 'c';
    name[1] = '0' + button_unit;
    name[3] = '\0';
    for (int i = 0; i != NUM_CURSORS; ++i)
    {
      name[2] = 'a' + i;
      cursors[i] = load_cursor(PWordPresentation::hInstance, name);
    }
  }
}

// public interface
void
Boxes::clear_bitmap_cache()
{
  Bitmap_Button* bb = bitmap_button;
  int id = 0;
  for (; id != BB_UNUSED_MAX; ++id, ++bb)
  {
    if (bb->icon_cache != 0)
    {
      destroy_icon(bb->icon_cache);
      bb->icon_cache = 0;
    }
  }
}

// public interface
int
Boxes::boxes_window_p(HWND window)
{
  if (inited)
  {
    for (int i = TOOLBOX_BEGIN; i != TOOLBOX_END; ++i)
    {
      if (boxes[i]->GetHandle() == window)
      {
	return(True);
      }
    }
  }
  return(False);
}

// public interface
void
Boxes::enable_modeless(int onoff)
{
  if (inited)
  {
    for (int i = TOOLBOX_BEGIN; i != TOOLBOX_END; ++i)
    {
      boxes[i]->Enable(onoff);
    }
  }
}

// public interface
void
Boxes::update()
{
  if (inited)
  {
    static int last_mode_command = 0;
    
    int changedp = False;
    int mode_command = PWordPresentation::GetEditModeCommand();
    if (last_mode_command != mode_command)
    {
      last_mode_command = mode_command;
      changedp = True;
      static int STATIC_NEAR index[] =
      {
	BB_IDM_Edit_InputText_MODE,
	BB_IDM_Edit_MakeTable_MODE,
	BB_IDM_Edit_SelectFrame_MODE,
	BB_IDM_Edit_Cut_MODE,
	BB_IDM_Edit_DrawLine_MODE,
	BB_IDM_Edit_MoveLine_MODE,
	BB_IDM_Edit_SetMargin_MODE,
	BB_IDM_Edit_FrameMargin_MODE,
	BB_IDM_Edit_SetFlow_MODE,
	BB_IDM_Edit_ExchFlow_MODE,
      };
      for (int i = 0; i != NUMBER_OF(index); ++i)
      {
	Bitmap_Button* bb = &bitmap_button[index[i]];
	bb->state = bb->idm == mode_command;
      }
    }
    
    if (Selected_Window != NULL)
    {
      Interval* i = Selected_Window->GetEditor() ->PrimarySelection();
      if (i != NULL)
      {
	static attr last_char_attr = 0;
	static attr last_para_attr = 0;
	
	attr char_attr = i->GetAtts();
	attr para_attr = i->GetLeft() ->GetParagraph() ->GetAtts();
	if (last_char_attr != char_attr)
	{
	  last_char_attr = char_attr;
	  changedp = True;
	  int* atts = get_atts(char_attr);
	  
	  bitmap_button[BB_IDM_Char_Bold].state = (atts[CA_bold] != 0);
	  bitmap_button[BB_IDM_Char_Italic].state = (atts[CA_italic] != 0);
	  
	  bitmap_button[BB_IDM_Char_Plain].state =
#ifdef BW3_DISPATCH
	  ((dispatch_command(IDM_Char_Plain, PW_MENU_STATE, 0) & MF_CHECKED) != 0);
#else /* BW3_DISPATCH */
	  ((dispatch_command(IDM_Char_Plain, PW_MENU_STATE) & MF_CHECKED) != 0);
#endif /* BW3_DISPATCH */
	  
	  bitmap_button[BB_IDM_Char_Underline].state =
#ifdef BW3_DISPATCH
	  ((dispatch_command(IDM_Char_Underline, PW_MENU_STATE, 0) & MF_CHECKED)
#else /* BW3_DISPATCH */
	  ((dispatch_command(IDM_Char_Underline, PW_MENU_STATE) & MF_CHECKED)
#endif /* BW3_DISPATCH */
	   != 0);
	}
	
	if (last_para_attr != para_attr)
	{
	  last_para_attr = para_attr;
	  changedp = True;
	  int align = GetAttValue(para_attr, PA_alignment);
	  
	  bitmap_button[BB_IDM_Para_AlignLeft].state
	  = (align == AL_leftaligned);
	  
	  bitmap_button[BB_IDM_Para_AlignCenter].state
	  = (align == AL_centered);
	  
	  bitmap_button[BB_IDM_Para_AlignRight].state
	  = (align == AL_rightaligned);
	  
	  bitmap_button[BB_IDM_Para_AlignEquality].state
	  = (align != AL_leftaligned
	     && align != AL_centered
	     && align != AL_rightaligned && align != AL_stretched);
	}
      }
    }
    
    if (changedp)
    {
      for (int i = BOX_BEGIN; i != BOX_END; ++i)
      {
	Box* box = boxes[i];
	int shownp = box->shownp;
	int num_buttons = box->num_buttons;
	Box_Button* button = box->buttons;
	for (int n = 0; n != num_buttons; ++n, ++button)
	{
	  if (bitmap_button_p(button->button_type)
	      && button->state != button->u.bb->state)
	  {
	    button->state = button->u.bb->state;
	    if (shownp)
	    {
	      box->drawing_prologue();
	      box->redisplay_button(n);
	    }
	  }
	}
	box->drawing_epilogue();
      }
    }
  }
}

static void NEAR
control_visible()
{
  for (int i = TOOLBOX_BEGIN; i != TOOLBOX_END; ++i)
  {
    Box* box = boxes[i];
    box->show_window(global_visible_p && box->visiblep);
  }
}

static void NEAR
set_global_visible(int visiblep)
{
  if (global_visible_p != visiblep)
  {
    global_visible_p = visiblep;
    control_visible();
  }
}

// public interface
void
Boxes::toggle_global_visible()
{
  if (inited)
  {
    set_global_visible(!global_visible_p);
  }
}

// public interface
void
Boxes::arrange_ribbons(RECT* rect)
{
  if (inited)
  {
    if (!Pref_internal_message_bar)
    {
      message_bar->arrange(rect,
			    message_bar->bar_type,
			    message_bar->size + INTERNAL_HEIGHT * 2);
    }
    for (int i = RIBBON_BEGIN; i != RIBBON_END; ++i)
    {
      Box* ribbon = boxes[i];
      ribbon->arrange(rect, i, button_unit * ribbon->width);
    }
    if (Pref_internal_message_bar)
    {
      message_bar->arrange(rect,
			    message_bar->bar_type,
			    message_bar->size + INTERNAL_HEIGHT * 2);
    }
  }
}

static long NEAR
customizable(int menu_state_p)
{
  if (menu_state_p)
  {
    return(global_customizable_p ? (MF_ENABLED | MF_CHECKED) : MF_ENABLED);
  }
#ifdef BW3_SCEENCUST
  ScreenCustDialog dlg(global_customizable_p);
  if (dlg.Go())
  {
    global_customizable_p = dlg.GetCustomizeP();
  }
#else /* BW3_SCEENCUST */
  global_customizable_p = !global_customizable_p;
  if (global_customizable_p)
  {
    AboutDialog::About(CUSTOM_NOTE);
  }
#endif /* BW3_SCEENCUST */
  load_cursors();
  return(0);
}

static long NEAR
msgbar(int menu_state_p)
{
  if (menu_state_p)
  {
    return(MF_ENABLED);
  }
  
  MessageDialogValues values;
  values.guidep = Pref_guide_button;
  values.where = message_bar->bar_type;
  int ok;
  // Control dialog life.
  {
    MessageDialog dialog(&values);
    ok = dialog.Go();
    dialog.get_values(&values);
  }
  if (ok)
  {
    Pref_guide_button = values.guidep;
    message_bar->set_bar_type(False, values.where);
  }
  return(0);
}

static long NEAR
toggle_visible(int menu_state_p, int i)
{
  Box* box = boxes[i];
  int visiblep = box->visiblep;
  if (menu_state_p)
  {
    return(visiblep ? (MF_ENABLED | MF_CHECKED) : MF_ENABLED);
  }
  box->visiblep = !visiblep;
  box->changed = True;
  if (ribbonp(box->box_type))
  {
    position_windows();
  }
  else
  {
    global_visible_p = True;
    control_visible();
  }
  return(0);
}

// public interface
long
Boxes::dispatch(int command, int menu_state_p)
{
  if (inited)
  {
    switch (command)
    {
     case IDM_Disp_Customizable:
      return(customizable(menu_state_p));
      
     case IDM_Disp_Bar:
      return(msgbar(menu_state_p));
      
     case IDM_Disp_Ribbon_Top:
      return(toggle_visible(menu_state_p, TOP_RIBBON));
      
     case IDM_Disp_Ribbon_Bot:
      return(toggle_visible(menu_state_p, BOT_RIBBON));
      
     case IDM_Disp_Ribbon_Left:
      return(toggle_visible(menu_state_p, LEFT_RIBBON));
      
     case IDM_Disp_Ribbon_Right:
      return(toggle_visible(menu_state_p, RIGHT_RIBBON));
      
     case IDM_Disp_ModeBox:
      return(toggle_visible(menu_state_p, MODE_TOOLBOX));
      
     case IDM_Disp_CharBox:
      return(toggle_visible(menu_state_p, CHAR_TOOLBOX));
      
     case IDM_Disp_ParaBox:
      return(toggle_visible(menu_state_p, PARA_TOOLBOX));
      
     case IDM_Disp_UserBox1:
      return(toggle_visible(menu_state_p, USER_TOOLBOX_1));
      
     case IDM_Disp_UserBox2:
      return(toggle_visible(menu_state_p, USER_TOOLBOX_2));
      
     case IDM_Disp_UserBox3:
      return(toggle_visible(menu_state_p, USER_TOOLBOX_3));
      
     case IDM_Disp_UserBox4:
      return(toggle_visible(menu_state_p, USER_TOOLBOX_4));
      
     case IDM_Disp_BigBox:
      return(toggle_visible(menu_state_p, BIG_TOOLBOX));
    }
  }
  return(0);
}

static int STATIC_NEAR last_guide_code = 0;

// public interface
void
Boxes::guide(int idm_code)
{
  if (message_bar->bar_type != NO_RIBBON && last_guide_code != idm_code)
  {
    last_guide_code = idm_code;
    Command_Desc* command = command_by_idm(idm_code);
    message_bar->set_message(command == NULL ? "" : command->guide);
  }
}

// public interface
int
Boxes::message(char* s)
{
  if (message_bar->bar_type != NO_RIBBON)
  {
    message_bar->set_message(s);
    return(True);
  }
  return(False);
}

// public interface
int
Boxes::status_out(char* s, int perm)
{
  if (Boxes::message(s))
  {
    return(True);
  }
  if (perm)
  {
    PWordPresentation::GetMessBox() ->DisplayPermanent(s);
  }
  else
  {
    PWordPresentation::GetMessBox() ->Display(s);
  }
  return(False);
}

// public interface
void
Boxes::kill_all_resources()
{
  if (inited)
  {
    kill_cursors();
    Boxes::clear_bitmap_cache();
  }
}

// public interface
void
Boxes::write_profile()
{
  Pref_message_bar = message_bar->bar_type;
  for (int i = BOX_BEGIN; i != BOX_END; ++i)
  {
    Box* box = boxes[i];
    if (box->changed || (toolboxp(box->box_type) && box->visiblep))
    {
      box->write_profile();
    }
  }
}

static int NEAR
calc_button_unit()
{
  int unit = Pref_button_size;
  switch (unit)
  {
   case BUTTON_SIZE_LARGE:
   case BUTTON_SIZE_MEDIUM:
   case BUTTON_SIZE_SMALL:
    break;

   default:
     // 32 is a best number.
     // 640 / 32 -> 20	== small button
     // 800 / 32 -> 25	~= medium button(24 dot)
     // 1024 / 32 -> 32  == large button
    unit = Screen::width / (32 * UNITS_PER_BUTTON);
    unit = (BUTTON_SIZE_LARGE <= unit ? BUTTON_SIZE_LARGE
	    : BUTTON_SIZE_MEDIUM <= unit ? BUTTON_SIZE_MEDIUM
	    : BUTTON_SIZE_SMALL);
    break;
  }
  return(unit);
}

// public interface
void
Boxes::fit_button_size()
{
  int new_unit = calc_button_unit();
  if (button_unit != new_unit)
  {
    button_unit = new_unit;
    Boxes::clear_bitmap_cache();
    load_cursors();
    for (int i = BOX_BEGIN; i != BOX_END; ++i)
    {
      Box* box = boxes[i];
      box->fit_toolbox_size();
    }
    position_windows();
  }
}

// public interface
void
Boxes::read_profile()
{
  button_unit = calc_button_unit();
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(1),
      BB_VISIBLE,
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_SPACE,
      BB_IDM_Char_Plain,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_MINCHO_TEXT_BUTTON,
      BB_GOTHIC_TEXT_BUTTON,
      BB_SPACE,
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_SPACE,
      BB_IDM_Edit_Cut,
      BB_IDM_Edit_Copy,
      BB_IDM_Edit_Paste,
      BB_IDM_Edit_UndoRun,
      BB_IDM_Edit_RedoRun,
      BB_SPACE,
      BB_IDM_Disp_ZoomWindow_Inc,
      BB_IDM_Disp_ZoomWindow_Dec,
      BB_END,
    };
    boxes[TOP_RIBBON] = Box::make_ribbon("top-ribbon", S_top_ribbon, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(1),
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_SPACE,
      BB_IDM_Char_Plain,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_MINCHO_TEXT_BUTTON,
      BB_GOTHIC_TEXT_BUTTON,
      BB_SPACE,
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_SPACE,
      BB_IDM_Edit_Cut,
      BB_IDM_Edit_Copy,
      BB_IDM_Edit_Paste,
      BB_IDM_Edit_UndoRun,
      BB_IDM_Edit_RedoRun,
      BB_SPACE,
      BB_IDM_Disp_ZoomWindow_Inc,
      BB_IDM_Disp_ZoomWindow_Dec,
      BB_END,
    };
    boxes[BOT_RIBBON] = Box::make_ribbon("bottom-ribbon",
					  S_bottom_ribbon, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(1),
      BB_VERTICAL,
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_SPACE,
      BB_IDM_Char_Plain,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_MINCHO_TEXT_BUTTON,
      BB_GOTHIC_TEXT_BUTTON,
      BB_SPACE,
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_SPACE,
      BB_IDM_Edit_Cut,
      BB_IDM_Edit_Copy,
      BB_IDM_Edit_Paste,
      BB_IDM_Edit_UndoRun,
      BB_IDM_Edit_RedoRun,
      BB_SPACE,
      BB_IDM_Disp_ZoomWindow_Inc,
      BB_IDM_Disp_ZoomWindow_Dec,
      BB_END,
    };
    boxes[LEFT_RIBBON] = Box::make_ribbon("left-ribbon",
					   S_left_ribbon, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(1),
      BB_VERTICAL,
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_SPACE,
      BB_IDM_Char_Plain,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_MINCHO_TEXT_BUTTON,
      BB_GOTHIC_TEXT_BUTTON,
      BB_SPACE,
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_SPACE,
      BB_IDM_Edit_Cut,
      BB_IDM_Edit_Copy,
      BB_IDM_Edit_Paste,
      BB_IDM_Edit_UndoRun,
      BB_IDM_Edit_RedoRun,
      BB_SPACE,
      BB_IDM_Disp_ZoomWindow_Inc,
      BB_IDM_Disp_ZoomWindow_Dec,
      BB_END,
    };
    boxes[RIGHT_RIBBON] = Box::make_ribbon("right-ribbon",
					    S_right_ribbon, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(5),
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_END,
    };
    boxes[MODE_TOOLBOX] = Box::make_toolbox("mode-toolbox",
					     S_mode_toolbox, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(4),
      BB_IDM_Char_Plain,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_IDM_Char_Spacing_Inc,
      BB_IDM_Char_Spacing_Dec,
      BB_END,
    };
    boxes[CHAR_TOOLBOX] = Box::make_toolbox("char-toolbox",
					     S_char_toolbox, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(3),
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_END,
    };
    boxes[PARA_TOOLBOX] = Box::make_toolbox("para-toolbox",
					     S_para_toolbox, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(5),
      BB_END,
    };
    boxes[USER_TOOLBOX_1] = Box::make_toolbox("user-toolbox-1",
					       S_user_toolbox_1, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(5),
      BB_END,
    };
    boxes[USER_TOOLBOX_2] = Box::make_toolbox("user-toolbox-2",
					       S_user_toolbox_2, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(5),
      BB_END,
    };
    boxes[USER_TOOLBOX_3] = Box::make_toolbox("user-toolbox-3",
					       S_user_toolbox_3, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(2), G(5),
      BB_END,
    };
    boxes[USER_TOOLBOX_4] = Box::make_toolbox("user-toolbox-4",
					       S_user_toolbox_4, profile);
  }
  {
    static int STATIC_NEAR profile[] =
    {
      BB_SIZE, G(6), G(7),
      BB_IDM_File_New,
      BB_IDM_File_Open,
      BB_IDM_File_Save,
      BB_IDM_File_Print,
      BB_IDM_Disp_ZoomWindow_Inc,
      BB_IDM_Disp_ZoomWindow_Dec,
      BB_NEWLINE,
      BB_IDM_Edit_InputText_MODE,
      BB_IDM_Edit_MoveLine_MODE,
      BB_IDM_Edit_DrawLine_MODE,
      BB_IDM_Edit_SetFlow_MODE,
      BB_IDM_Edit_SetMargin_MODE,
      BB_IDM_Char_Spacing_Inc,
      BB_NEWLINE,
      BB_IDM_Edit_MakeTable_MODE,
      BB_IDM_Edit_SelectFrame_MODE,
      BB_IDM_Edit_Cut_MODE,
      BB_IDM_Edit_ExchFlow_MODE,
      BB_IDM_Edit_FrameMargin_MODE,
      BB_IDM_Char_Spacing_Dec,
      BB_NEWLINE,
      BB_IDM_Char_Plain,
      BB_IDM_Char_Bold,
      BB_IDM_Char_Italic,
      BB_IDM_Char_Underline,
      BB_IDM_Char_Size_Inc,
      BB_IDM_Char_Size_Dec,
      BB_NEWLINE,
      BB_IDM_Para_AlignEquality,
      BB_IDM_Para_AlignRight,
      BB_IDM_Para_AlignCenter,
      BB_IDM_Para_AlignLeft,
      BB_IDM_Para_LineSpacing_Inc,
      BB_IDM_Para_LineSpacing_Dec,
      BB_NEWLINE,
      BB_IDM_Edit_Cut,
      BB_IDM_Edit_Copy,
      BB_IDM_Edit_Paste,
      BB_IDM_Edit_UndoRun,
      BB_IDM_Edit_RedoRun,
      BB_IDM_Again,
      BB_NEWLINE,
      BB_INITIAL_TEXT_BUTTON,
      BB_IDM_NoOperation,
      BB_END,
    };
    Box* box = Box::make_toolbox("big-toolbox", S_big_toolbox, NULL);
    boxes[BIG_TOOLBOX] = box;
    box->customizablep = False;
    box->parse_builtin_profile(profile);
  }

  message_bar = new Bar();
  
  for (int i = BOX_BEGIN; i != BOX_END; ++i)
  {
    Box* box = boxes[i];
    if (box->read_profile() == 0)
    {
      for (int i = 0; i != NUM_PROFILE_STRINGS; ++i)
      {
	xfree(box->initial_profile[i]);
	box->initial_profile[i] = NULL;
      }
      box->parse_profile();
    }
    box->changed = False;
  }

  message_bar->set_bar_type(False, Pref_message_bar);
  global_visible_p = Pref_startup_toolbox_visible;
  position_windows();
  control_visible();
  inited = True;
}

// public interface
void
Boxes::term()
{
  if (inited)
  {
    Boxes::kill_all_resources();
  }
}

// public interface
void
Boxes::init()
{
  // All initialize routines move into Boxes::read_profile()
}

