// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: crosspoi.cpp,v 3.2 1999/05/12 00:22:16 kudou Exp $
// class CrossPoint

#include "pword.h"
#include "frameown.h"
#include "fission.h"
#include "fissioni.h"
#include "borderli.h"
#include <assert.h>

// get fission frame
FissionFrame* 
CrossPoint::GetFissionFrame()
{
  if (vf != 0) 
  {
    return vf->GetFissionFrame();
  }
  if (hf != 0) 
  {
    return hf->GetFissionFrame();
  }
  assert(False);
  return 0;
}

// get vertical prev crosspoint
CrossPoint* 
CrossPoint::FindVPrev()
{
  Iunit y = Y();
  for (CrossPoint* cp = vf->GetStartCrossPoint();
       (cp != 0) && (cp->GetVNext() != 0);
       cp = cp->GetVNext()) 
  {
    if (y <= cp->GetVNext()->Y()) 
    {
      return cp;
    }
  }
  return 0;
}

// get horizontal prev crosspoint
CrossPoint* 
CrossPoint::FindHPrev()
{
  Iunit x = X();
  for (CrossPoint* cp = hf->GetStartCrossPoint();
       (cp != 0) && (cp->GetHNext() != 0);
       cp = cp->GetHNext()) 
  {
    if (x <= cp->GetHNext()->X()) 
    {
      return cp;
    }
  }
  return 0;
}

// constructor, horizontal start crosspoint
// generate start and end crosspoint
CrossPoint::CrossPoint(HFission* f, VFission* start, VFission* end, 
			uword h_border_index)
{
  // start crosspoint
  kind = (CrossKind) (VCross | HStart);
  vf = start;
  hf = f;
  fr = 0;
  SetHLink(0, 0);			// horizontal link
  CrossPoint* prev = FindVPrev();
  SetVLink(prev, prev->GetVNext());	// vertical link
  v_border = prev->v_border;
  h_border = h_border_index;
  
  // end crosspoint
  new CrossPoint(f, this, end);
  ClearSelection();
  ClearAdaptingSelection();
}

// constructor, vertical start crosspoint
// generate start and end crosspoint
CrossPoint::CrossPoint(VFission* f, HFission* start, HFission* end, 
			uword v_border_index)
{
  // start crosspoint
  kind = (CrossKind) (HCross | VStart);
  hf = start;
  vf = f;
  fr = 0;
  SetVLink(0, 0);			// vertical link
  CrossPoint* prev = FindHPrev();
  SetHLink(prev, prev->GetHNext());	// horizontal link
  h_border = prev->h_border;
  v_border = v_border_index;
  
  // end crosspoint
  new CrossPoint(f, this, end);
  ClearSelection();
  ClearAdaptingSelection();
}

// constructor(private), for horizontal end crosspoint
CrossPoint::CrossPoint(HFission* f, CrossPoint* hprev, VFission* end)
{
  kind = (CrossKind) (VCross | HEnd);
  vf = end;
  hf = f;
  fr = 0;
  SetHLink(hprev, 0);			// horizontal link
  CrossPoint* prev = FindVPrev();
  SetVLink(prev, prev->GetVNext());	// vertical link
  v_border = prev->v_border;
  h_border = BorderLine::GetZeroWidthLineIndex();	// default
  ClearSelection();
  ClearAdaptingSelection();
}

// constructor(private), for vertical end crosspoint
CrossPoint::CrossPoint(VFission* f, CrossPoint* vprev, HFission* end)
{
  kind = (CrossKind) (HCross | VEnd);
  vf = f;
  hf = end;
  fr = 0;
  SetVLink(vprev, 0);			// vertical link
  CrossPoint* prev = FindHPrev();
  SetHLink(prev, prev->GetHNext());	// horizontal link
  h_border = prev->h_border;
  v_border = BorderLine::GetZeroWidthLineIndex();	// default
  ClearSelection();
  ClearAdaptingSelection();
}

// make middle crosspoint
void 
CrossPoint::MakeMiddleCrossPoint(HFission* f)
{
  HFissionIterator hfi(f);
  VFission* m;
  while ((m = hfi()) != 0) 
  {
    // middle crosspoint
    new CrossPoint(f, m);
  }
}

// make middle crosspoint
void 
CrossPoint::MakeMiddleCrossPoint(VFission* f)
{
  VFissionIterator vfi(f);
  HFission* m;
  while ((m = vfi()) != 0) 
  {
    // middle crosspoint
    new CrossPoint(m, f);
  }
}

// constructor(private), for middle crosspoint
CrossPoint::CrossPoint(HFission* h, VFission* v)
{
  kind = (CrossKind) (VCross | HCross);
  vf = v;
  hf = h;
  fr = 0;
  
  CrossPoint* prev = FindVPrev();
  assert(prev != 0);
  SetVLink(prev, prev->GetVNext());
  v_border = prev->v_border;
  
  prev = FindHPrev();
  assert(prev != 0);
  SetHLink(prev, prev->GetHNext());
  h_border = prev->h_border;
  
  ClearSelection();
  ClearAdaptingSelection();
}

// constructor, for first FissionFrame
CrossPoint::CrossPoint(HFission* h, VFission* v, CrossKind k)
{
  kind = k;
  vf = v;
  hf = h;
  fr = 0;
  v_border = BorderLine::GetZeroWidthLineIndex();	// default
  h_border = BorderLine::GetZeroWidthLineIndex();	// default
  if (k == (VStart | HStart)) 
  {	// --- 1st. ---
    // left up crosspoint
    v_prev = 0;			// null
    v_next = 0;			// * not yet initialize.
    h_prev = 0;			// null
    h_next = 0;			// * not yet initialize.
  }
  else if (k == (VStart | HEnd)) 
  {	// --- 2nd. ---
    // right up crosspoint
    v_prev = 0;			// null
    v_next = 0;			// * not yet initialize.
    h_prev = h->GetStartCrossPoint();	// horizontal link
    h_next = 0;			// null
    h->GetStartCrossPoint()->h_next = this; // horizontal link
  }
  else if (k == (VEnd | HStart)) 
  {	// --- 3rd. ---
    // left bottom crosspoint
    v_prev = v->GetStartCrossPoint();	// vertical link
    v_next = 0;			// null
    h_prev = 0;			// null
    h_next = 0;			// * not yet initialize
    v->GetStartCrossPoint()->v_next = this; // vertical link
  }
  else if (k == (VEnd | HEnd)) 
  {	// --- 4th. ---
    // right bottom crosspoint
    v_prev = v->GetStartCrossPoint();	// vertical link
    v_next = 0;			// null
    h_prev = h->GetStartCrossPoint();	// horizontal link
    h_next = 0;			// null
    v->GetStartCrossPoint()->v_next = this; // vertical link
    h->GetStartCrossPoint()->h_next = this; // horizontal link
  }
  ClearSelection();
  ClearAdaptingSelection();
}

// constructor for special purpos
CrossPoint::CrossPoint(Fission* f)
{
  kind = Setting;
  fr = 0;
  v_border = 0;
  h_border = 0;
  v_prev = 0;
  v_next = 0;
  h_prev = 0;
  h_next = 0;
  if (f->GetDir() == Fission::Vertical)
  {
    vf = (VFission*)f;
    hf = 0;
  }
  else // horizontal fission
  {
    vf = 0;
    hf = (HFission*)f;
  }
  ClearSelection();
  ClearAdaptingSelection();
}

// destructor
CrossPoint::~CrossPoint()
{
  if (fr) 
  {
    GetFissionFrame()->VanishFrameObject(fr);	// * virtual-interface call !
  }
  if (kind & VCross) 
  {
    UnlinkV();
  }
  if (kind & HCross) 
  {
    UnlinkH();
  }
}

// set frame-object
FrameObject* 
CrossPoint::SetFrameObject(CrossPoint* prev)
{
  if (prev->fr == 0) 
  {
    // empty cell
    return fr = 0;
  }
  else 
  {
    FissionFrame* ff = GetFissionFrame();
    ff->ChangeFrameObject(prev->fr);	// * virtual-interface call !
    return fr = ff->CreateFrameObject(prev->fr, this);
    // * virtual-interface call !
  }
}

// set frame-object
FrameObject* 
CrossPoint::SetFrameObject()
{
  return fr = GetFissionFrame()->CreateFrameObject(0, this);
  // * virtual-interface call !
}

// set frame-object
FrameObject* CrossPoint::SetFrameObject(FrameObject* from)
{
  return fr = GetFissionFrame()->CreateFrameObject(from, this);
  // * virtual-interface call !
}

// update frame-object
void 
CrossPoint::UpdateFrameObject()
{
  if (fr == 0) 
  {
    return;		// empty cell
  }
  GetFissionFrame()->ChangeFrameObject(fr);
}

// get horizontal prev cell
CrossPoint* 
CrossPoint::GetHPrevCell()
{
  if (kind & HStart) 
  {
    return 0;
  }
  for (CrossPoint* cp = GetHPrev(); 
       cp != 0 ; 
       cp = cp->GetHPrev()) 
  {
    if (cp->kind & (VStart | VCross)) 
    {
      return cp;
    }
  }
  return 0;
}

// get vertical prev cell
CrossPoint* 
CrossPoint::GetVPrevCell()
{
  if (kind & VStart) 
  {
    return 0;
  }
  for (CrossPoint* cp = GetVPrev(); 
       cp != 0; 
       cp = cp->GetVPrev()) 
  {
    if (cp->kind & (HStart | HCross)) 
    {
      return cp;
    }
  }
  return 0;
}

// get horizontal next cell
CrossPoint* 
CrossPoint::GetHNextCell()
{
  if (kind & HEnd) 
  {
    return 0;
  }
  for (CrossPoint* cp = GetHNext(); 
       cp != 0 ; 
       cp = cp->GetHNext()) 
  {
    if (cp->kind & (VStart | VCross)) 
    {
      return cp;
    }
  }
  return 0;
}

// get vertical next cell
CrossPoint* 
CrossPoint::GetVNextCell()
{
  if (kind & VEnd) 
  {
    return 0;
  }
  for (CrossPoint* cp = GetVNext(); 
       cp != 0; 
       cp = cp->GetVNext()) 
  {
    if (cp->kind & (HStart | HCross)) 
    {
      return cp;
    }
  }
  return 0;
}

// set vertical crosspoint link
void 
CrossPoint::SetVLink(CrossPoint* prev, CrossPoint* next)
{
  if (next == 0 && prev == 0) 
  {
    v_next = 0;
    v_prev = 0;
  }
  else if (prev == 0) 
  {
    next->v_prev = this;
    v_next = next;
    v_prev = 0;
  }
  else if (next == 0) 
  {
    prev->v_next = this;
    v_prev = prev;
    v_next = 0;
  }
  else 
  {
    prev->v_next = this;
    v_prev = prev;
    next->v_prev = this;
    v_next = next;
  }
}

// set horisontal crosspoint link
void 
CrossPoint::SetHLink(CrossPoint* prev, CrossPoint* next)
{
  if (next == 0 && prev == 0) 
  {
    h_next = 0;
    h_prev = 0;
  }
  else if (prev == 0) 
  {
    next->h_prev = this;
    h_next = next;
    h_prev = 0;
  }
  else if (next == 0) 
  {
    prev->h_next = this;
    h_prev = prev;
    h_next = 0;
  }
  else 
  {
    prev->h_next = this;
    h_prev = prev;
    next->h_prev = this;
    h_next = next;
  }
}

// unlink vertical link
void 
CrossPoint::UnlinkV()
{
  if (v_prev != 0) 
  {
    v_prev->v_next = v_next;
  }
  if (v_next != 0) 
  {
    v_next->v_prev = v_prev;
  }
}

// unlink horizontal link
void 
CrossPoint::UnlinkH()
{
  if (h_prev != 0) 
  {
    h_prev->h_next = h_next;
  }
  if (h_next != 0) 
  {
    h_next->h_prev = h_prev;
  }
}

// get cell size
Rect 
CrossPoint::GetRect()
{
  Iunit x1 = 0;
  Iunit x2 = 0;
  Iunit y1 = 0;
  Iunit y2 = 0;
  if (v_next != 0 && h_next != 0) 
  {
    x1 = X();
    y1 = Y();
    CrossPoint* cp = GetHNext();
    while (cp->kind & VEnd)
    {
      cp = cp->GetHNext();
    }
    x2 = cp->X();
    cp = GetVNext();
    while (cp->kind & HEnd)
    {
      cp = cp->GetVNext();
    }
    y2 = cp->Y();
  }
  return Rect(x1, y1, x2, y2);
}

// get own rect
Rect 
CrossPoint::GetOwnRect()
{
  Iunit x1 = 0;
  Iunit x2 = 0;
  Iunit y1 = 0;
  Iunit y2 = 0;
  if (v_next != 0 && h_next != 0) 
  {
    x1 = X();
    y1 = Y();
    CrossPoint* cp = GetHNext();
    while (cp->kind & VEnd) 
    {
      cp = cp->GetHNext();
    }
    x2 = cp->X() - 1;
    cp = GetVNext();
    while (cp->kind & HEnd) 
    {
      cp = cp->GetVNext();
    }
    y2 = cp->Y() - 1;
  }
  return Rect(x1, y1, x2, y2);
}

// get point coordinates
Vector 
CrossPoint::GetPoint()
{
  return Vector(X(), Y());
}

// vertical fission deleted mark
// fission destructor use only!
void 
CrossPoint::SetVDeletedMark()
{
  vf = 0;
}

// horizontal fission deleted mark
// fission destructor use only!
void 
CrossPoint::SetHDeletedMark()
{
  hf = 0;
}

// set up frame object
void 
CrossPoint::SetupFrameObject(CrossPoint* src)
{
  fr = GetFissionFrame()->CreateFrameObject(src->fr, this);
  // * virtual-interface call !
}

// get internal rect
Rect 
CrossPoint::GetInternalRect()
{
  if (!HaveCell()) 
  {
    // error
    return Rect(0, 0, 0, 0);
  }
  
  // x1
  Iunit x1 = GetInternalX1();
  CrossPoint* cp = this;
  while (cp->GetVNext()->kind & HEnd) 
  {
    cp = cp->GetVNext();
    Iunit tmp = cp->GetInternalX1();
    x1 = (x1 < tmp) ? tmp : x1;
  }
  cp = cp->GetVNext();	// HStart or HCross
  // now cp point the left bottom cross-point.
  
  // y2
  Iunit y2 = cp->GetInternalY2();
  while (cp->GetHNext()->kind & VStart) 
  {
    cp = cp->GetHNext();
    Iunit tmp = cp->GetInternalY2();
    y2 = (y2 > tmp) ? tmp : y2;
  }
  
  // y1
  Iunit y1 = GetInternalY1();
  cp = this;
  while (cp->GetHNext()->kind & VEnd) 
  {
    cp = cp->GetHNext();
    Iunit tmp = cp->GetInternalY1();
    y1 = (y1 < tmp) ? tmp : y1;
  }
  cp = cp->GetHNext();	// VStart or VCross
  // now cp point the right up cross-point
  
  // x2
  Iunit x2 = cp->GetInternalX2();
  while (cp->GetVNext()->kind & VStart) 
  {
    cp = cp->GetVNext();
    Iunit tmp = cp->GetInternalX2();
    x2 = (x2 > tmp) ? tmp : x2;
  }
  return Rect(x1, y1, x2, y2);
}

// get internal position
Vector 
CrossPoint::GetInternalPosition()
{
  if (!HaveCell()) 
  {
    // error
    return Vector(0, 0);
  }
  
  // x1
  Iunit x1 = GetInternalX1();
  CrossPoint* cp = this;
  while (cp->GetVNext()->kind & HEnd) 
  {
    cp = cp->GetVNext();
    Iunit tmp = cp->GetInternalX1();
    x1 = (x1 < tmp) ? tmp : x1;
  }
  
  // y1
  Iunit y1 = GetInternalY1();
  cp = this;
  while (cp->GetHNext()->kind & VEnd) 
  {
    cp = cp->GetHNext();
    Iunit tmp = cp->GetInternalY1();
    y1 = (y1 < tmp) ? tmp : y1;
  }
  return Vector(x1, y1);
}

// get internal left position
Iunit 
CrossPoint::GetInternalX1()
{
  return X() + BorderLine::GetRightWidth(v_border);
}

// get internal top position
Iunit 
CrossPoint::GetInternalY1()
{
  return Y() + BorderLine::GetLowerWidth(h_border);
}

// get internal right position
Iunit 
CrossPoint::GetInternalX2()
{
  return X() - BorderLine::GetLeftWidth(v_border) - 1;
}

// get internal bottom position
Iunit 
CrossPoint::GetInternalY2()
{
  return Y() - BorderLine::GetUpperWidth(h_border) - 1;
}

// have this point
bool 
CrossPoint::HaveThisPoint(Vector& pnt)
{
  if (HaveCell() && GetRect().IsInThisRect(pnt)) 
  {
    return True;		// hit !!
  }
  return False;		// do not have cell
}

// get cell right crosspoint
CrossPoint* 
CrossPoint::GetCellRightCrossPoint()
{
  CrossPoint* cp = GetHNext();
  while (cp->kind & VEnd) 
  {
    cp = cp->GetHNext();
  }
  return cp;
}

// get cell bottom crosspoint
CrossPoint* 
CrossPoint::GetCellBottomCrossPoint()
{
  CrossPoint* cp = GetVNext();
  while (cp->kind & HEnd) 
  {
    cp = cp->GetVNext();
  }
  return cp;
}

// get upper cell
CrossPoint* 
CrossPoint::GetUpperCell()
{
  CrossPoint* cp = this;
  do 
  {
    if (cp->GetKind() & (VCross | VEnd)) 
    {
      cp = cp->GetVPrev();
      while (cp != 0) 
      {
	if (cp->GetKind() & (HCross | HStart)) 
	{
	  return cp;
	}
	cp = cp->GetVPrev();
      }
      return 0;	// may be error.
    }
    cp = cp->GetHPrev();
  } while (cp != 0);
  return 0;
}

// get downward cell
CrossPoint* 
CrossPoint::GetDownwardCell()
{
  CrossPoint* cp = this;
  do 
  {
    if (cp->GetKind() & (VCross | VStart)) 
    {
      return cp;
    }
    cp = cp->GetHPrev();
  } while (cp != 0);
  return 0;
}

// get left cell
CrossPoint* 
CrossPoint::GetLeftCell()
{
  CrossPoint* cp = this;
  do 
  {
    if (cp->GetKind() & (HCross | HEnd)) 
    {
      cp = cp->GetHPrev();
      while (cp != 0) 
      {
	if (cp->GetKind() & (VCross | VStart)) 
	{
	  return cp;
	}
	cp = cp->GetHPrev();
      }
      return 0;	// may be error.
    }
    cp = cp->GetVPrev();
  } while (cp != 0);
  return 0;
}

// get right cell
CrossPoint* 
CrossPoint::GetRightCell()
{
  CrossPoint* cp = this;
  do 
  {
    if (cp->GetKind() & (HCross | HStart)) 
    {
      return cp;
    }
    cp = cp->GetVPrev();
  } while (cp != 0);
  return 0;
}

// move vertical position
void 
CrossPoint::UpdateVertical()
{
  // right side frame-object.
  CrossPoint* cp = GetRightCell();
  if (cp != 0 && cp->fr != 0) 
  {
    GetFissionFrame()->ChangeFrameObject(cp->fr);
    // * virtual-interface call !
  }
  
  // left side frame-object.
  cp = GetLeftCell();
  if (cp != 0 && cp->fr != 0) 
  {
    GetFissionFrame()->ChangeFrameObject(cp->fr);
    // * virtual-interface call !
  }
}

// move horizontal position
void 
CrossPoint::UpdateHorizontal()
{
  // down side frame-object
  CrossPoint* cp = GetDownwardCell();
  if (cp != 0 && cp->fr != 0) 
  {
    GetFissionFrame()->ChangeFrameObject(cp->fr);
    // * virtual-interface call !
  }
  
  // up side frame-object
  cp = GetUpperCell();
  if (cp != 0 && cp->fr != 0) 
  {
    GetFissionFrame()->ChangeFrameObject(cp->fr);
    // * virtual-interface call !
  }
}

// set virtical start crosspoint
void
CrossPoint::SetVStart()
{
  kind = (CrossKind) ((kind & HMask) | VStart);
  v_prev = 0;
}

// set virtical end crosspoint
void
CrossPoint::SetVEnd()
{
  kind = (CrossKind) ((kind & HMask) | VEnd);
  v_next = 0;
  if (fr) 
  {
    GetFissionFrame()->VanishFrameObject(fr);
    // * virtual-interface call !
  }
  fr = 0;
}

// set horizontal start crosspoint
void 
CrossPoint::SetHStart()
{
  kind = (CrossKind) ((kind & VMask) | HStart);
  h_prev = 0;
}

// set horizontal end crosspoint
void 
CrossPoint::SetHEnd()
{
  kind = (CrossKind) ((kind & VMask) | HEnd);
  h_next = 0;
  if (fr) 
  {
    GetFissionFrame()->VanishFrameObject(fr);
    // * virtual-interface call !
  }
  fr = 0;
}

#if 0
// proper virtical border
uword 
CrossPoint::GetProperVBorder(uword index)
{
  if (index == 0) 
  {
    return 0;
  }
  // check left or right rim fission
  CrossPoint* cp_start = vf->GetStartCrossPoint();
  if (cp_start->kind == LStart) 
  {
    // left rim fission
    BorderLine* bl = BorderLine::Get(index);
    if (bl->GetVAlign() != BorderLine::AlignRight) 
    {
      BorderLine bl_fix(*bl);
      bl_fix.SetAlign(BorderLine::AlignRight);
      return bl_fix.Enter();
    }
  }
  else if (cp_start->kind == RStart) 
  {
    // right rim fission
    BorderLine* bl = BorderLine::Get(index);
    if (bl->GetVAlign() != BorderLine::AlignLeft) 
    {
      BorderLine bl_fix(*bl);
      bl_fix.SetAlign(BorderLine::AlignLeft);
      return bl_fix.Enter();
    }
  }
  return index;
}

// proper horizontal border
uword 
CrossPoint::GetProperHBorder(uword index)
{
  if (index == 0) 
  {
    return 0;
  }
  // check top or bottom rim fission
  CrossPoint* cp_start = hf->GetStartCrossPoint();
  if (cp_start->kind == TStart) 
  {
    // top rim fission
    BorderLine* bl = BorderLine::Get(index);
    if (bl->GetVAlign() != BorderLine::AlignBelow) 
    {
      BorderLine bl_fix(*bl);
      bl_fix.SetAlign(BorderLine::AlignBelow);
      return bl_fix.Enter();
    }
  }
  else if (cp_start->kind == BStart) 
  {
    // bottom rim fission
    BorderLine* bl = BorderLine::Get(index);
    if (bl->GetVAlign() != BorderLine::AlignAbove) 
    {
      BorderLine bl_fix(*bl);
      bl_fix.SetAlign(BorderLine::AlignAbove);
      return bl_fix.Enter();
    }
  }
  
  return index;
}
#endif

// set vertical border line
void 
CrossPoint::SetVBorder(uword index)
{
  v_border = (GetFissionFrame()->
	      GetFrameOwner()->
	      GetProperVBorder(this, index));
  UpdateVertical();
}

// set horizontal border line
void 
CrossPoint::SetHBorder(uword index)
{
  h_border = (GetFissionFrame()->
	      GetFrameOwner()->
	      GetProperHBorder(this, index));
  UpdateHorizontal();
}

// reset frame-object
void 
CrossPoint::ResetFrameObject()
{
  fr = 0;
}

// create frame-object
void 
CrossPoint::CreateFrameObject(FrameObject* fr)
{
  CrossPoint::fr = fr;
}

// exchange frame-object
void 
CrossPoint::ExchangeFrameObject(CrossPoint* cp)
{
  FrameObject* tmp = fr;
  fr = cp->fr;
  cp->fr = tmp;
  if (fr != 0)
  {
    fr->SetCrossPoint(this);
  }
  if (cp->fr != 0)
  {
    cp->fr->SetCrossPoint(cp);
  }
}

// cell bottom fission
HFission* 
CrossPoint::GetCellBottomFission()
{
  return GetCellBottomCrossPoint()->hf;
}

// cell right fission
VFission* 
CrossPoint::GetCellRightFission()
{
  return GetCellRightCrossPoint()->vf;
}

// get cell fission
void 
CrossPoint::GetCellFission(Fission*& top, Fission*& bottom, 
				 Fission*& left, Fission*& right)
{
  top = hf;
  left = vf;
  bottom = 0;
  right = 0;
  for (CrossPoint* cp = GetHNext();
       cp != 0;
       cp = cp->GetHNext()) 
  {
    if (cp->kind & (VStart | VCross)) 
    {
      right = cp->vf;
      break;
    }
  }
  for (cp = GetVNext();
       cp != 0;
       cp = cp->GetVNext()) 
  {
    if (cp->kind & (HStart | HCross)) 
    {
      bottom = cp->hf;
      break;
    }
  }
}

// check vanish this crosspoint
bool 
CrossPoint::CheckVanish()
{
  if (fr != 0) 
  {
    return fr->CheckVanish();
  }
  return True;
}

// vertical reposition
// correct vertical crosspoint link position
void 
CrossPoint::RepositionV()
{
  Iunit y = Y();
  if ((GetVPrev()->Y() <= y) &&
      (GetVNext()->Y() >= y)) 
  {
    // no need
    return;
  }
  Fission* v_f = vf;
  UnlinkV();
  CrossPoint* prev = v_f->GetStartCrossPoint();
  for (CrossPoint* cp = prev->GetVNext();
       cp != 0;
       cp = cp->GetVNext()) 
  {
    if (cp->Y() > y) 
    {
      SetVLink(prev, cp);
      SetVBorder(prev->GetVBorder());
      return;
    }
    prev = cp;
  }
  prev = v_f->GetEndCrossPoint()->GetVPrev();
  SetVLink(prev, v_f->GetEndCrossPoint());
  SetVBorder(prev->GetVBorder());
}

// horizontal reposition
// correct horizontal crosspoint link position
void 
CrossPoint::RepositionH()
{
  Iunit x = X();
  if ((GetHPrev()->X() <= x) &&
      (GetHNext()->X() >= x)) 
  {
    // no need
    return;
  }
  Fission* h_f = hf;
  UnlinkH();
  CrossPoint* prev = h_f->GetStartCrossPoint();
  for (CrossPoint* cp = prev->GetHNext();
       cp != 0;
       cp = cp->GetHNext()) 
  {
    if (cp->X() > x) 
    {
      SetHLink(prev, cp);
      SetHBorder(prev->GetHBorder());
      return;
    }
    prev = cp;
  }
  prev = h_f->GetEndCrossPoint()->GetHPrev();
  SetHLink(prev, h_f->GetEndCrossPoint());
  SetHBorder(prev->GetHBorder());
}

// reorder same position
void 
CrossPoint::ReorderVSamePosition(CrossPoint* to)
{
  if (GetVNext() != to) 
  {
    to->UnlinkV();
    to->SetVLink(this, GetVNext());
  }
}

// reorder same position
void 
CrossPoint::ReorderHSamePosition(CrossPoint* to)
{
  if (GetHNext() != to) 
  {
    to->UnlinkH();
    to->SetHLink(this, GetHNext());
  }
}

void 
CrossPoint::SetVAdaptingSelection()
{
  if (!v_single_sel && !v_double_sel) 
  {
    v_single_sel = 1;
  }
  else 
  {
    v_single_sel = 0;
    v_double_sel = 1;
  }
}

void 
CrossPoint::SetHAdaptingSelection()
{
  if (!h_single_sel && !h_double_sel) 
  {
    h_single_sel = 1;
  }
  else 
  {
    h_single_sel = 0;
    h_double_sel = 1;
  }
}

void 
CrossPoint::VMergeCrossPoint(CrossPoint* removed)
{
  assert(kind & VEnd);
  assert(removed->kind & VStart);
  
  v_border = removed->v_border;
  v_next = removed->v_next;
  v_next->v_prev = this;
  if (h_next == removed) 
  {
    h_next = removed->h_next;
    h_next->h_prev = this;
  }
  else 
  {
    h_prev = removed->h_prev;
    h_prev->h_next = this;
  }
  fr = removed->fr;
  if (fr != 0) 
  {
    fr->SetCrossPoint(this);
  }
  
  kind = (CrossKind) ((kind & HMask) | VCross);
  
  removed->fr = 0;
  removed->v_next = 0;
  removed->v_prev = 0;
  removed->h_next = 0;
  removed->h_prev = 0;
}

void
CrossPoint::HMergeCrossPoint(CrossPoint* removed)
{
  assert(kind & HEnd);
  assert(removed->kind & HStart);
  
  h_border = removed->h_border;
  h_next = removed->h_next;
  h_next->h_prev = this;
  if (v_next == removed) 
  {
    v_next = removed->v_next;
    v_next->v_prev = this;
  }
  else 
  {
    v_prev = removed->v_prev;
    v_prev->v_next = this;
  }
  fr = removed->fr;
  if (fr != 0) 
  {
    fr->SetCrossPoint(this);
  }
  
  kind = (CrossKind) ((kind & VMask) | HCross);
  
  removed->fr = 0;
  removed->v_next = 0;
  removed->v_prev = 0;
  removed->h_next = 0;
  removed->h_prev = 0;
}


