// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: fission.h,v 3.2 1999/05/12 00:22:16 kudou Exp $
// CrossPoint, Fission, HFission, VFission, FissionFrame classes
// These classes are basic elements of fission representation.
//
// +------+------+
// |      |      |
// +------+      |
// |      |      |
// +------+------+
//
// FissionFrame NX͏L̂悤ȃt[\NXłB
// ̂̐HFissionNX̃IuWFNgŁAc̐VFission
// NX̃IuWFNgɂĕ\B܂cƉ̐̌_A
// Ȃ킿'+'̕ CrossPoint NX̃IuWFNgłB
// ĐɂĈ͂܂łlp̈ɁAeLXg
// hԂA}`Ȃǂt邪ǍЂ̖ڂʂ̂
// lp̗̈̍̌_CrossPointłBCrossPoint́A
// eLXgȂǂ̊{NXłFrameObjectւ̃|C^ێ
// B

#ifndef FISSION_H
#define FISSION_H

#include "rect.h"

class FrameObject;
class FrameOwner;
class Fission;
class HFission;
class VFission;
class FissionFrame;

// ------------------------------------------------------------
// fission cross point							

class CrossPoint
{
  friend class Fission;
  friend class VFission;
  friend class HFission;
  
public:
  // kind of fission crosses
  enum CrossKind 
  {
    VCross = 0x01,			// Bit 1 : vertical fission cross
    VStart = 0x02,			// Bit 2 : vertical fission start
    VEnd   = 0x04,			// Bit 3 : vertical fission end
    VError = 0x08,			// Bit 4 : vertical fission error
    HCross = 0x10,			// Bit 5 : horizontal fission cross
    HStart = 0x20,			// Bit 6 : horizontal fission start
    HEnd   = 0x40,			// Bit 7 : horizontal fission end
    HError = 0x80,			// Bit 8 : horizontal fission error
    
    // for bit check
    Start      = VStart | HStart,	// start bit check
    End        = VEnd   | HEnd,		// end bit check
    Crossroads = VCross | HCross,	// crossroads of fission
    Error      = VError | HError,	// error
    Setting    = 0x00,			// now setting
    VMask      = 0x0f,			// vertical mask
    HMask      = 0xf0,			// horizontal mask
    
    // for rim fission check
    TStart = VStart | HStart,		// top fission start
    BStart = VEnd | HStart,		// bottom fission start
    LStart = TStart,			// left fission start
    RStart = VStart | HEnd,		// right fission start
    TEnd   = RStart,			// top fission end
    BEnd   = VEnd | HEnd,		// bottom fission end
    LEnd   = BStart,			// left fission end
    REnd   = BEnd,			// reft fission end
  };
  
private:
  // this represents the status of fission crossing
  CrossKind kind;
  
  // vertical fission
  VFission* vf;
  
  // horizontal fission
  HFission* hf;
  
  // vertical direction border index
  uword v_border;
  
  // horizontal direction border index
  uword h_border;
  
  // previous vertical cross point link
  CrossPoint* v_prev;
  
  // vertical cross point link
  CrossPoint* v_next;
  
  // previous vertical cross point link
  CrossPoint* h_prev;
  
  // horizontal cross point link
  CrossPoint* h_next;
  
  // frame object
  FrameObject* fr;
  
  // selection bits
  // horizontal and vertical red selection
  int h_red_sel : 1;
  int v_red_sel : 1;
  
  // for adapting line attribute by multi frame selection
  int v_single_sel : 1;
  int h_single_sel : 1;
  int v_double_sel : 1;
  int h_double_sel : 1;
  
  // special purpos check flag
  // this flag is used for many porpos
  int special_check_p : 1;
  
private:
  // set vertical crosspoint link
  void SetVLink(CrossPoint* prev, CrossPoint* next);
  
  // set horizontal crosspoint link
  void SetHLink(CrossPoint* prev, CrossPoint* next);
  
  // unlink vertical link
  void UnlinkV();
  
  // unlink horizontal link
  void UnlinkH();
  
  // set up frame object
  void SetupFrameObject(CrossPoint* src);
  
public:
  // constructor, for horizontal end crosspoint
  CrossPoint(HFission* f, CrossPoint* hprev, VFission* end);
  
  // constructor, for vertical end crosspoint
  CrossPoint(VFission* f, CrossPoint* vprev, HFission* end);
  
  // constructor for special purpos
  CrossPoint(Fission* f);
  
private:
  // constructor, for middle crosspoint
  CrossPoint(HFission* h, VFission* v);
  
  // get internal left position
  Iunit GetInternalX1();
  
  // get internal top position
  Iunit GetInternalY1();
  
  // get internal right position
  Iunit GetInternalX2();
  
  // get internal bottom position
  Iunit GetInternalY2();
  
  // get upper cell crosspoint
  CrossPoint* GetUpperCell();
  
  // get downward cell crosspoint
  CrossPoint* GetDownwardCell();
  
  // get left cell crosspoint
  CrossPoint* GetLeftCell();
  
  // get right cell crosspoint
  CrossPoint* GetRightCell();
  
  // proper virtical border
  uword GetProperVBorder(uword index);
  
  // proper horizontal border
  uword GetProperHBorder(uword index);
  
  // get vertical prev crosspoint
  CrossPoint* FindVPrev();
  
  // get horizontal prev crosspoint
  CrossPoint* FindHPrev();
  
public:
  // horizontal start constructor
  // generate start and end crosspoint
  CrossPoint(HFission* f, VFission* start, VFission* end, 
	      uword h_border_index);
  
  // vertical start constructor
  // generate start and end crosspoint
  CrossPoint(VFission* f, HFission* start, HFission* end, 
              uword v_border_index);
  
  // for first FissionFrame
  CrossPoint(HFission* h, VFission* v, CrossKind k);
  
  // destructor
  ~CrossPoint();
  
  // set frame-object
  FrameObject* SetFrameObject(CrossPoint* prv);
  
  // set frame-object
  FrameObject* SetFrameObject();
  
  // set frame-object
  FrameObject* SetFrameObject(FrameObject* from);
  
  // reset frame-object
  void ResetFrameObject();
  
  // create frame-object
  void CreateFrameObject(FrameObject* fr);
  
  // get frame-object
  FrameObject* GetFrameObject() 
  {
    return fr; 
  }
  
  // get fission frame
  FissionFrame* GetFissionFrame();
  
  // update frame-object
  void UpdateFrameObject();
  
  // get horizontal prev cell
  CrossPoint* GetHPrevCell();
  
  // get vertical prev cell
  CrossPoint* GetVPrevCell();
  
  // get horizontal next cell
  CrossPoint* GetHNextCell();
  
  // get vertical next cell
  CrossPoint* GetVNextCell();
  
  // set vertical border line index
  void SetVBorder(uword index);
  
  // set horizontal border line index
  void SetHBorder(uword index);
  
  // get vertical border line index
  uword GetVBorder() 
  { 
    return v_border; 
  }
  
  // get horizontal border line index
  uword GetHBorder() 
  { 
    return h_border; 
  }
  
  // have cell
  bool HaveCell() 
  {
    return ! (kind & End); 
  }
  
  // get vertical fission
  VFission* GetVFission() 
  {
    return vf; 
  }
  
  // get horizontal fission
  HFission* GetHFission() 
  {
    return hf; 
  }
  
  // get vertical next crosspoint
  CrossPoint* GetVNext() 
  {
    return v_next; 
  }
  
  // get vertical prev crosspoint
  CrossPoint* GetVPrev() 
  {
    return v_prev; 
  }
  
  // get horizontal next crosspoint
  CrossPoint* GetHNext() 
  {
    return h_next; 
  }
  
  // get horizontal prev crosspoint
  CrossPoint* GetHPrev() 
  {
    return h_prev; 
  }
  
  // get cross kind
  CrossKind GetKind() 
  { 
    return kind;
  }
  
  // x
  Iunit X();
  
  // y
  Iunit Y();
  
  // get cell rect
  Rect GetRect();
  
  // get own rect
  Rect GetOwnRect();
  
  // get point coordinates
  Vector GetPoint();
  
  // get internal rect
  Rect GetInternalRect();
  
  // get internal position
  Vector GetInternalPosition();
  
  // have this point
  bool HaveThisPoint(Vector& pnt);
  
  // fission destructor use only!
  void SetVDeletedMark();
  
  // fission destructor use only!
  void SetHDeletedMark();
  
  // make middle crosspoint
  void MakeMiddleCrossPoint(HFission* f);
  
  // make middle crosspoint
  void MakeMiddleCrossPoint(VFission* f);
  
  // get cell right crosspoint
  CrossPoint* GetCellRightCrossPoint();
  
  // get cell bottom crosspoint
  CrossPoint* GetCellBottomCrossPoint();
  
  // update vertical position or width
  void UpdateVertical();
  
  // update horizontal position or width
  void UpdateHorizontal();
  
  // set virtical start crosspoint
  void SetVStart();
  
  // set virtical end crosspoint
  void SetVEnd();
  
  // set horizontal start crosspoint
  void SetHStart();
  
  // set horizontal end crosspoint
  void SetHEnd();
  
  // set virtical fission
  void ChangeOwner(VFission* vf) 
  { 
    CrossPoint::vf = vf; 
  }
  
  // set horizontal fission
  void ChangeOwner(HFission* hf) 
  { 
    CrossPoint::hf = hf; 
  }
  
  // exchange frame-object
  void ExchangeFrameObject(CrossPoint* cp);
  
  // cell top fission
  HFission* GetCellTopFission() 
  {
    return hf; 
  }
  
  // cell bottom fission
  HFission* GetCellBottomFission();
  
  // cell left fission
  VFission* GetCellLeftFission() 
  {
    return vf; 
  }
  
  // cell right fission
  VFission* GetCellRightFission();
  
  // get cell fission
  void GetCellFission(Fission*& top, Fission*& bottom, 
                       Fission*& left, Fission*& right);
  
  // check vanish this crosspoint
  bool CheckVanish();
  
  // vertical reposition
  // correct vertical crosspoint link position
  void RepositionV();
  
  // horizontal reposition
  // correct horizontal crosspoint link position
  void RepositionH();
  
  // reorder same position
  void ReorderVSamePosition(CrossPoint* to);
  void ReorderHSamePosition(CrossPoint* to);
  
  // line selection
  
  // bit check
  int GetHRedSelection()	
  {
    return h_red_sel; 
  }
  
  int GetVRedSelection()	
  { 
    return v_red_sel; 
  }
  
  // bit set
  void SetHRedSelection()	
  { 
    h_red_sel = 1; 
  }
  
  void SetVRedSelection()	
  {
    v_red_sel = 1; 
  }
  
  // bit clear
  void ClearHRedSelection()	
  { 
    h_red_sel = 0; 
  }
  
  void ClearVRedSelection()	
  { 
    v_red_sel = 0; 
  }
  
  void ClearSelection()
  {
    h_red_sel = 0;
    v_red_sel = 0;
  }
  
  // for target line
  void SetVAdaptingSelection();
  void SetHAdaptingSelection();
  void ClearAdaptingSelection()
  {
    v_single_sel = 0;
    h_single_sel = 0;
    v_double_sel = 0;
    h_double_sel = 0;
  }
  
  bool IsSingleVAdaptingSelection() 
  {
    return v_single_sel != 0; 
  }
  
  bool IsDoubleVAdaptingSelection() 
  { 
    return v_double_sel != 0; 
  }
  
  bool IsSingleHAdaptingSelection() 
  {
    return h_single_sel != 0; 
  }
  bool IsDoubleHAdaptingSelection() 
  {
    return h_double_sel != 0; 
  }
  
  // merge two crosspoint
  void VMergeCrossPoint(CrossPoint* removed);
  void HMergeCrossPoint(CrossPoint* removed);
  
private:
  // set this represents the status of fission crossing
  void SetKind(CrossKind kind)
  {
    CrossPoint::kind = kind;
  }
  
  // set vertical fission
  void SetVFission(VFission* vf)
  {
    CrossPoint::vf = vf;
  }
  
  // set horizontal fission
  void SetHFission(HFission* hf)
  {
    CrossPoint::hf = hf;
  }
  
  // set previous vertical cross point link
  void SetVPrev(CrossPoint* v_prev)
  {
    CrossPoint::v_prev = v_prev;
  }
  
  // set vertical cross point link
  void SetVNext(CrossPoint* v_next)
  {
    CrossPoint::v_next = v_next;
  }
  
  // set previous vertical cross point link
  void SetHPrev(CrossPoint* h_prev)
  {
    CrossPoint::h_prev = h_prev;
  }
  
  // set horizontal cross point link
  void SetHNext(CrossPoint* h_next)
  {
    CrossPoint::h_next = h_next;
  } 
  
  // set vertical border line
  void SetVBorderX(uword index)
  {
    CrossPoint::v_border = index;
  }
  
  // set horizontal border line
  void SetHBorderX(uword index)
  {
    CrossPoint::h_border = index;
  }
  
public:
  // set special purpos check flag
  void SetSpecialCheckP(bool s)
  {
    special_check_p = s;
  }
  
  // get special purpos check flag
  bool SpecialCheckP()
  {
    return special_check_p != 0;
  }
};

// ------------------------------------------------------------
// fission

class Fission 
{
public:
  // direction of fission
  enum Direction 
  {
    Vertical,
    Horizontal
  };

protected:
  // direction of fission
  Direction dir;
  
  // coordinates
  Iunit point;
  
  // cross point list start
  CrossPoint* start;
  
  // cross point list end
  CrossPoint* end;
  
  // fission frame
  FissionFrame* ff;
  
  // Fission link
  // null terminate list
  Fission* next;
  
public:
  Fission* GetNext() 
  {
    return next;
  }
  
  void SetNext(Fission* n) 
  {
    next = n;
  }
  
private:
  // constructor
  Fission(FissionFrame* ff);
  
  // set frame-object on this fission
  void SetHFrameObject();
  
  // set frame-object on this fission
  void SetVFrameObject();
  
protected:
  // constructor
  Fission(Iunit point, Fission* start, Fission* end, Direction dir, 
           FissionFrame* ff, uword border_index);
  
  // constructor for special purpos
  Fission(Iunit point, Direction dir, FissionFrame* ff);
  
public:
  // constructor for FissionFrame
  Fission(Iunit x, Iunit y, FissionFrame* ff);
  
  // destructor
  ~Fission();
  
  // get start point
  Vector GetStartPoint() 
  {
    return start->GetPoint(); 
  }
  
  // get end point
  Vector GetEndPoint() 
  {
    return end->GetPoint(); 
  }
  
  // get start cross point
  CrossPoint* GetStartCrossPoint() 
  {
    return start; 
  }
  
  // get end cross point
  CrossPoint* GetEndCrossPoint() 
  {
    return end; 
  }
  
  // get point
  Iunit GetPoint() 
  {
    return point; 
  }
  
  // get direction
  Direction GetDir() 
  {
    return dir; 
  }
  
  // get start fission
  Fission* GetStartFission() 
  {
    return(dir == Vertical) ?
    (Fission*)start->GetHFission() :
    (Fission*)start->GetVFission();
  }
  
  // get end fission
  Fission* GetEndFission() 
  {
    return(dir == Vertical) ? 
    (Fission*)end->GetHFission() :
    (Fission*)end->GetVFission();
  }
  
  // get start position
  Iunit GetStartPos()
  {
    return(dir == Vertical) ? start->Y() : start->X();
  }
  
  // get end position
  Iunit GetEndPos()
  {
    return(dir == Vertical) ? end->Y() : end->X();
  }
  
  // get fission frame
  FissionFrame* GetFissionFrame() 
  {
    return ff; 
  }
  
  // set border line index
  void SetBorder(uword index);
  
  // set border line index
  void SetBorder(CrossPoint* from, CrossPoint* to, uword index);
  
  // move position
  // this method generate update request
  void MovePosition(Iunit p);
  
  // set fission point
  // this method never generate update request
  void SetPoint(Iunit p) 
  {
    point = p; 
  }
  
  // shorten fission start
  void ShortenStart(CrossPoint* to);
  
  // shorten fission end
  void ShortenEnd(CrossPoint* from);
  
  // cut fission middle
  Fission* CutMiddle(CrossPoint* from, CrossPoint* to);
  
  // disconnect fission at the cross-point position
  Fission* Disconnect(CrossPoint* pos);
  
  // find cross-point
  CrossPoint* FindCrossPoint(Fission* f);
  
  // check vanish fission
  bool CheckVanish();
  
  // check shorten start
  bool CheckShortenStart(CrossPoint* to);
  
  // check shorten fission end
  bool CheckShortenEnd(CrossPoint* from);
  
  // check cut fission middle
  bool CheckCutMiddle(CrossPoint* from, CrossPoint* to);
  
  // reorder same position
  void ReorderSamePosition(CrossPoint* from, CrossPoint* to);
  
  // expand x minimum direction and create new frame
  VFission* PrevXInsertSpecial(Iunit size, uword border_index);
  
  // expand y minimun direction and create new frame
  HFission* PrevYInsertSpecial(Iunit size, uword border_index);
};

// ------------------------------------------------------------
// vertical fission

class VFission
  : public Fission
{
public:
  // constructor
  VFission(Iunit point, HFission* start, HFission* end, 
            FissionFrame* ff, uword border_index);
  
  // destructor
  ~VFission() 
  { 
  }
  
  // get start fission
  HFission* GetStartFission() 
  { 
    return(HFission*)Fission::GetStartFission(); 
  }
  
  // get end fission
  HFission* GetEndFission() 
  {
    return(HFission*)Fission::GetEndFission();
  }
  
  // get point
  Iunit X() 
  {
    return GetPoint(); 
  }
  
  // get pointed cell
  CrossPoint* GetCell(Vector& pnt);
  
  // merge
  void MergeFission(VFission* f);
  
  // stretch start
  void StretchStart(HFission* f);
  
  // stretch end
  void StretchEnd(HFission* f);
};

// ------------------------------------------------------------
// horisontal fission

class HFission
  : public Fission
{
public:
  // constructor
  HFission(Iunit point, VFission* start, VFission* end, 
            FissionFrame* ff, uword border_index);
  
  // destructor
  ~HFission() 
  { 
  }
  
  // get start fission
  VFission* GetStartFission() 
  {
    return(VFission*)Fission::GetStartFission();
  }
  
  // get end fission
  VFission* GetEndFission() 
  {
    return(VFission*)Fission::GetEndFission();
  }
  
  // get point
  Iunit Y() 
  {
    return GetPoint(); 
  }
  
  // get pointed cell
  CrossPoint* GetCell(Vector& pnt);
  
  // merge
  void MergeFission(HFission* f);
  
  // stretch start
  void StretchStart(VFission* f);
  
  // stretch end
  void StretchEnd(VFission* f);
};

inline Iunit
CrossPoint::Y()
{
  return hf->GetPoint();
}

inline Iunit
CrossPoint::X()
{
  return vf->GetPoint();
}

inline 
VFission::VFission(Iunit point, HFission* start, HFission* end, 
		    FissionFrame* ff, uword border_index)
  : Fission(point, (Fission*)start, (Fission*)end, 
	   Fission::Vertical, ff, border_index)
{
}

inline 
HFission::HFission(Iunit point, VFission* start, VFission* end, 
		    FissionFrame* ff, uword border_index)
  : Fission(point, (Fission*)start, (Fission*)end, 
	   Fission::Horizontal, ff, border_index)
{
}

// ------------------------------------------------------------
// fission frame

class FissionFrame
{
private:
  
  // top fission
  HFission* top;
  
  // bottom fission
  HFission* bottom;
  
  // left fission
  VFission* left;
  
  // right fission
  VFission* right;
  
  // frame owner
  FrameOwner* fo;
  
  // get near fission error
  static Iunit get_near_fission_error;
  
  // fission-list
  // first -> [fission] -> [fission] -> [fission]->null
  //                                        ^
  //                                        |
  //                                       last
  Fission* first;
  Fission* last;
  
public:
  // add fission to fission-list
  void AddFission(Fission* f);
  
  // remove fission from fisison-list
  void RemoveFission(Fission* f);
  
  // get first fission
  Fission* GetFirstFission() 
  {
    return first; 
  }
  
public:
  // constructor
  FissionFrame(Iunit x, Iunit y, FrameOwner* fo);
  
  // destructor
  ~FissionFrame();
  
  // get top fission
  HFission* GetTop() 
  {
    return top; 
  }
  
  // get bottom fission
  HFission* GetBottom() 
  {
    return bottom; 
  }
  
  // get left fission
  VFission* GetLeft() 
  {
    return left; 
  }
  
  // get right fission
  VFission* GetRight() 
  {
    return right; 
  }
  
  // get cell crosspoint
  CrossPoint* GetCell(Vector& pnt);
  
  // get near fission
  Fission* GetNearFission(Vector& pnt);
  
  // set get-near-fission-error
  static void SetNearFissionError(Iunit e);
  
  // get get-near-fission-error
  static Iunit GetNearFissionError();
  
  // initial flow set.
  FrameObject* CreateInitialFlow(CrossPoint* cp);
  
  // get size
  Vector GetSize();
  
public:
  // divide fission
  VFission* Divide(HFission* start, HFission* end, Iunit point, 
		    uword border_index);
  
  // divide fission
  HFission* Divide(VFission* start, VFission* end, Iunit point, 
		    uword border_index = 0);
  
  // divide fission
  Fission* Divide(Fission* start, Fission* end, Iunit point, 
		   uword border_index);
  
  // vanish fission
  void Vanish(Fission* v);
  
  // move fission
  void Move(Fission* f, Iunit point);
  
  // change border line index
  void ChangeBorder(Fission* f, uword index);
  
  // change border line index
  void ChangeBorder(Fission* f, CrossPoint* from, CrossPoint* to, 
		     uword index);
  
  // set flow
  FrameObject* FlowTo(CrossPoint* from, CrossPoint* to);
  
  // cut fission part
  Fission* VanishPart(Fission* f, Fission* cut_start, Fission* cut_end);
  
  // exchange frame-object
  void ExchangeFrameObject(CrossPoint* cp1, CrossPoint* cp2);
  
  // change fission frame size with scalling
  void ChangeSize(Iunit width, Iunit height);
  
  // sift fisison x position 
  void ShiftXPosition(Iunit l_lim, Iunit offset);

  // sift fisison x position 
  void ShiftXPositionX(Iunit offset);
  
  
  // shift fisison y position
  void ShiftYPosition(Iunit l_lim, Iunit offset);
  
  // shift fisison y position
  void ShiftYPositionX(Iunit offset);

  // change fission frame size with no scalling
  void StretchSize(Iunit top_diff, Iunit bottom_diff,
                    Iunit left_diff, Iunit right_diff);
  
  // change size and margin
  // *_m	: new margin
  void ChangeSizeAndMargin(Iunit width, Iunit height,
                            Iunit top_m, Iunit bottom_m,
                            Iunit left_m, Iunit right_m,
                            Fission* top, Fission* bottom,
                            Fission* left, Fission* right);
  
  // segment move
  Fission* SegMove(Fission* f, Fission* seg_start, 
                    Fission* seg_end, Iunit point);
  
  // check vanish fission
  bool CheckVanish(Fission* v);
  
  // check cut fission part
  bool CheckVanishPart(Fission* f, Fission* cut_start, Fission* cut_end);
  
  // expand x minimum direction and create new frame
  void PrevXInsertSpecial(Iunit size, uword border_index);
  
  // expand y minimun direction and create new frame
  void PrevYInsertSpecial(Iunit size, uword border_index);
  
  // disconnect lines which overlapped rectanglar region
  void DisconnectOverlappedLine(FrameRectRgn& rgn);

  // clear rectanglar region
  void ClearSingleFrameRegion(FrameRectRgn& rgn);
  
  // for insert frame
  void MakeDuplicateRegion(FrameRectRgn& rgn,
			    bool post_insert,
			    bool insert_line);
  
  // for combine frame
  void DeleteRegion(FrameRectRgn& rgn, bool delete_line);
  
  // combine frame
  void CombineFrameRegion(FrameRectRgn& rgn,
			   bool combine_vertical,
			   bool combine_horizontal);
  
  // align region
  void AlignFrameRegion(FrameRectRgn& rgn, 
			 bool align_vertical,
			 bool align_horizontal);

public:
  // ------------------------------
  // frame-owner interface
  
  // these functions are called by CrossPoint.
  // you never call these functions from out side of this class.
  FrameOwner* GetFrameOwner() 
  {
    return fo;
  }
  
  // update frame
  void UpDate();
  
  // object create
  FrameObject* CreateFrameObject(FrameObject* src, CrossPoint* cp);
  
  // object change
  void ChangeFrameObject(FrameObject* fr);
  
  // object vanished
  void VanishFrameObject(FrameObject* fr);
  
  // create new flow gate
  FrameObject* CreateNewGate(CrossPoint* cp);
  
  // change flow
  // return		: new next
  FrameObject* ChangeFlow(FrameObject* from, FrameObject* to);
  
  // change fission frame size
  void UpdateSize();
  
  // check rim fission
  bool IsRimFission(Fission* f);
  
  // find move limit
  void FindMoveLimit(Fission* f, Iunit& l_move_lim, Iunit& u_move_lim);
  
  // update all frame
  void UpdateAllframe();
  
private:
  // merge
  bool MergeOneStep();
  
public:
  void MergeFission();
  
  // set special purpos check flag in all frames
  void SetSpecialCheckP(bool s);
};

#ifdef FRAMEOWN_H
// if you use these functions, include "frameown.h" before this header file.

inline void
FissionFrame::UpDate()
{
  fo->UpDate();
}

inline FrameObject*
FissionFrame::CreateFrameObject(FrameObject* src , CrossPoint* cp)
{
  return fo->CreateFrameObject(src, cp);
}

inline void
FissionFrame::ChangeFrameObject(FrameObject* fr)
{
  fr->UpdateInternalRect();
  fo->ChangeFrameObject(fr);
}

inline void
FissionFrame::VanishFrameObject(FrameObject* fr)
{
  fo->VanishFrameObject(fr);
}

inline FrameObject*
FissionFrame::CreateNewGate(CrossPoint* cp)
{
  return fo->CreateNewGate(cp);
}

inline FrameObject*
FissionFrame::ChangeFlow(FrameObject* from, FrameObject* to)
{
  return fo->ChangeFlow(from, to);
}

inline void
FissionFrame::UpdateSize()
{
  fo->UpdateSize();
}

#endif /* FRAMEOWN_H */

// ------------------------------------------------------------
// frame rect region

class FrameRectRgn
{
  HFission* top;
  HFission* bot;
  VFission* lft;
  VFission* rgh;
  
public:
  // valid rect region check
  bool ValidP();
    
  // accessors
  HFission* GetTop()    { return top; }
  HFission* GetBottom() { return bot; }
  VFission* GetLeft()   { return lft; }
  VFission* GetRight()  { return rgh; }

  // mutators
  void Set() { top = bot = 0; lft = rgh = 0; }
  
  void Set(FrameRectRgn& frr)
  {
    top = frr.GetTop();
    bot = frr.GetBottom();
    lft = frr.GetLeft();
    rgh = frr.GetRight();
  }

  void Set(FissionFrame* ff)
  {
    top = ff->GetTop();
    bot = ff->GetBottom();
    lft = ff->GetLeft();
    rgh = ff->GetRight();
  }
  
  void Set(HFission* top, HFission* bot,
	    VFission* lft, VFission* rgh)
  {
    FrameRectRgn::top = top;
    FrameRectRgn::bot = bot;
    FrameRectRgn::lft = lft;
    FrameRectRgn::rgh = rgh;
  }
  
  void Set(CrossPoint* cp);
  void Set(CrossPoint* cp1, CrossPoint* cp2);
  void SetTop(HFission* top)    { FrameRectRgn::top = top; }
  void SetBottom(HFission* bot) { FrameRectRgn::bot = bot; }
  void SetLeft(VFission* lft)   { FrameRectRgn::lft = lft; }
  void SetRight(VFission* rgh)  { FrameRectRgn::rgh = rgh; }
  
  // constructor
  FrameRectRgn()                  { Set(); }
  FrameRectRgn(FrameRectRgn& frr) { Set(frr); }
  FrameRectRgn(FissionFrame* ff)  { Set(ff); }
  FrameRectRgn(HFission* top, HFission* bot,
		VFission* lft, VFission* rgh)     { Set(top, bot, lft, rgh); }
  FrameRectRgn(CrossPoint* cp)                   { Set(cp); }
  FrameRectRgn(CrossPoint* cp1, CrossPoint* cp2) { Set(cp1, cp2); }
  
  // get position
  Iunit X1() { return lft->X(); }
  Iunit X2() { return rgh->X(); }
  Iunit Y1() { return top->Y(); }
  Iunit Y2() { return bot->Y(); }
  
  // cornner
  CrossPoint* GetTopLeft();
  CrossPoint* GetTopRight();
  CrossPoint* GetBottomLeft();
  CrossPoint* GetBottomRight();
};

#endif /* FISSION_H */
