// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: fissionf.cpp,v 3.2 1999/05/12 00:22:16 kudou Exp $
// class FissionFrame
// t[\ŜێNX

#include "pword.h"
#include "frameown.h"
#include "fission.h"
#include "fissioni.h"
#include "foiterat.h"
#include "flist.h"

// constructor
FissionFrame::FissionFrame(Iunit x, Iunit y, FrameOwner* fo)
{
  first  = 0;
  last   = 0;
  // special fission constructor
  top    = (HFission*)new Fission(x, y, this);
  left   = top->GetStartFission();
  right  = top->GetEndFission();
  bottom = left->GetEndFission();
  FissionFrame::fo = fo;
}

// destructor
FissionFrame::~FissionFrame()
{
  // delete top decendant.
  CrossPoint* cp = top->GetStartCrossPoint();
  CrossPoint* end = top->GetEndCrossPoint();
  cp = cp->GetHNext();
  while (cp != end) 
  {
    VFission* vf = cp->GetVFission();
    delete vf;
    cp = cp->GetHNext();
  }
  
  // delete left decendant.
  cp = left->GetStartCrossPoint();
  end = left->GetEndCrossPoint();
  cp = cp->GetVNext();
  while (cp != end) 
  {
    HFission* hf = cp->GetHFission();
    delete hf;
    cp = cp->GetVNext();
  }
  
  delete top;
  delete left;
  delete right;
  delete bottom;
}

// add fission to fission-list
void FissionFrame::AddFission(Fission* f)
{
  if (last == 0) 
  {
    first = last = f;
  }
  else 
  {
    last->SetNext(f);
    f->SetNext(0);
    last = f;
  }
}

// remove fission from fisison-list
void 
FissionFrame::RemoveFission(Fission* rm_f)
{
  Fission* f = first;
  Fission* prev = 0;
  while (f != 0) 
  {
    if (f == rm_f) 
    {
      if (prev != 0) 
      {
	prev->SetNext(f->GetNext());
      }
      else 
      {
	first = f->GetNext();
      }
      f->SetNext(0);
      if (rm_f == last) 
      {
	last = prev;
      }
      return;
    }
    prev = f;
    f = f->GetNext();
  }
}

// divide fission, for Horizontal line
HFission* 
FissionFrame::Divide(VFission* start, VFission* end, 
		      Iunit point, uword border_index)
{
  return new HFission(point, start, end, this, border_index);
}

// divide fission, for Virtical line
VFission* 
FissionFrame::Divide(HFission* start, HFission* end, 
		      Iunit point, uword border_index)
{
  return new VFission(point, start, end, this, border_index);
}

// divide fission
Fission* 
FissionFrame::Divide(Fission* start, Fission* end, 
		      Iunit point, uword border)
{
  if (start->GetDir() == Fission::Vertical) 
  {
    return Divide((VFission*)start, (VFission*)end, point, border);
  }
  else 
  {
    return Divide((HFission*)start, (HFission*)end, point, border);
  }
}

// vanish fission
void 
FissionFrame::Vanish(Fission* v)
{
  delete v;
}

// move fission
void 
FissionFrame::Move(Fission* f, Iunit point)
{
  f->MovePosition(point);
  MergeFission();
}

// change border
void 
FissionFrame::ChangeBorder(Fission* f, uword index)
{
  f->SetBorder(index);
}

// cange border
void 
FissionFrame::ChangeBorder(Fission* f, CrossPoint* from, 
			    CrossPoint* to, uword index)
{
  f->SetBorder(from, to, index);
}

// create initial flow
FrameObject* 
FissionFrame::CreateInitialFlow(CrossPoint* cp)
{
  return cp->SetFrameObject();
}

// get cell crosspoint
CrossPoint* 
FissionFrame::GetCell(Vector& pnt)
{
  return top->GetCell(pnt);
}

// get-near-fission-error
Iunit FissionFrame::get_near_fission_error = 100;	// default value

// set get-near-fission-error
void 
FissionFrame::SetNearFissionError(Iunit e)
{
  get_near_fission_error = e;
}

// get get-near-fission-error
Iunit 
FissionFrame::GetNearFissionError()
{
  return get_near_fission_error;
}

// get near fission
Fission* 
FissionFrame::GetNearFission(Vector& pnt)
{
  CrossPoint* cp = GetCell(pnt);
  if (cp == 0) 
  {
    // out of fission-frame
    return 0;
  }
  Rect r = cp->GetInternalRect();
  Iunit top_diff	= pnt.y - r.y1;
  Iunit bottom_diff 	= r.y2 - pnt.y;
  Iunit left_diff	= pnt.x - r.x1;
  Iunit right_diff	= r.x2 - pnt.x;
  Iunit min = ((((top_diff < bottom_diff) ? top_diff : bottom_diff) <
		((left_diff < right_diff) ? left_diff : right_diff)) ?
	       ((top_diff < bottom_diff) ? top_diff : bottom_diff) :
	       ((left_diff < right_diff) ? left_diff : right_diff));
  if (min > get_near_fission_error) 
  {
    return 0;		// can not find, error is too big.
  }
  if (min == top_diff) 
  {
    return cp->GetHFission();
  }
  if (min == left_diff) 
  {
    return cp->GetVFission();
  }
  if (min == bottom_diff) 
  {
    return cp->GetCellBottomCrossPoint()->GetHFission();
  }
  // min == right_diff
  return cp->GetCellRightCrossPoint()->GetVFission();
}

// get size
Vector 
FissionFrame::GetSize()
{
  return Vector(top->GetEndCrossPoint()->X(), 
		 right->GetEndCrossPoint()->Y());
}

// set flow
// FrameObject* FissionFrame::FlowTo(CrossPoint* from, CrossPoint* to)
// from		: source crosspoint
// to		: destination crosspoint
// return	: destination frameobject
FrameObject* 
FissionFrame::FlowTo(CrossPoint* from, CrossPoint* to)
{
  if (from == 0) 
  {
    // create new flow
    return CreateInitialFlow(to);
  }
  FrameObject* src_fr = from->GetFrameObject();
  if (src_fr == 0) 
  {
    // source crosspoint error.
    return 0;
  }
  FrameObject* dest_fr = to->GetFrameObject();
  if (dest_fr) 
  {
    // change flow stream
    return ChangeFlow(src_fr, dest_fr);
  }
  else 
  {
    // create destination flow
    return to->SetFrameObject(src_fr);
  }
}

// cut fission part
Fission* 
FissionFrame::VanishPart(Fission* f, 
			  Fission* cut_start, Fission* cut_end)
{
  if ((f->GetStartFission() == cut_start) &&
      (f->GetEndFission() == cut_end)) 
  {
    // cut all
    delete f;
    return 0;
  }
  
  CrossPoint* start;
  CrossPoint* end;
  if ((end = f->FindCrossPoint(cut_end)) == 0) 
  {
    return 0;	// error
  }
  if ((start = f->FindCrossPoint(cut_start)) == 0) 
  {
    return 0;	// error
  }
  
  if (f->GetStartFission() == cut_start) 
  {
    // cut start part
    f->ShortenStart(end);
    return 0;
  }
  if (f->GetEndFission() == cut_end) 
  {
    // cut end part
    f->ShortenEnd(start);
    return 0;
  }
  // cut middle part
  return f->CutMiddle(start, end);
}

// segment move
Fission* 
FissionFrame::SegMove(Fission* f, Fission* seg_start, 
		       Fission* seg_end, Iunit point)
{
  if ((f->GetStartFission() == seg_start) &&
      (f->GetEndFission() == seg_end)) 
  {
    // move all
    Move(f, point);
    return f;
  }
  CrossPoint* start;
  CrossPoint* end;
  if ((end = f->FindCrossPoint(seg_end)) == 0) 
  {
    return 0;	// error
  }
  if ((start = f->FindCrossPoint(seg_start)) == 0) 
  {
    return 0;	// error
  }
  
  // store frame object
  FrameObject* fo = start->GetFrameObject();
  // null clear
  start->CreateFrameObject(0);
  
  // get border
  uword border = 0;
  if (f->GetDir() == Fission::Vertical) 
  {
    border = start->GetVBorder();
  }
  else 
  {
    border = start->GetHBorder();
  }
  
  // cut segment
  if (f->GetStartFission() == seg_start) 
  {
    // cut start part
    f->ShortenStart(end);
  }
  else if (f->GetEndFission() == seg_end) 
  {
    // cut end part
    f->ShortenEnd(start);
  }
  else 
  {
    // cut middle part
    f->CutMiddle(start, end);
  }
  
  // make segment fission
  Fission* seg = Divide(seg_start, seg_end, point, border);
  
  if (seg->GetStartCrossPoint()->GetFrameObject() != 0) 
  {
    VanishFrameObject(seg->GetStartCrossPoint()->GetFrameObject());
  }
  
  // restore frame object
  seg->GetStartCrossPoint()->CreateFrameObject(fo);
  fo->SetCrossPoint(seg->GetStartCrossPoint());
  ChangeFrameObject(fo);
  MergeFission();
  return seg;
}

// exchange frame-object
void 
FissionFrame::ExchangeFrameObject(CrossPoint* cp1, CrossPoint* cp2)
{
  if (cp1->GetFrameObject() == 0 ||
      cp2->GetFrameObject() == 0) 
  {
    // error
    return;
  }
  cp1->ExchangeFrameObject(cp2);
  cp1->UpdateFrameObject();
  cp2->UpdateFrameObject();
}

// update all frame
void 
FissionFrame::UpdateAllframe()
{
  CellCPIterator cpit(this);
  CrossPoint* cp = 0;
  while ((cp = cpit()) != 0) 
  {
    cp->UpdateFrameObject();
  }
}

// change fission frame size
void 
FissionFrame::ChangeSize(Iunit width, Iunit height)
{
  // size check
  Vector org_size = GetSize();
  if ((org_size.x == width) && (org_size.y == height)) 
  {
    // same size
    return;
  }
  Lunit old_x_size = org_size.x;
  Lunit old_y_size = org_size.y;
  Lunit new_x_size = width;
  Lunit new_y_size = height;
  
  // scale new size
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Vertical) 
    {
      Lunit x = f->GetPoint();
      x = (new_x_size * x) / old_x_size;
      f->SetPoint((Iunit)x);
    }
    else 
    {          // Fission::Horizontal
      Lunit y = f->GetPoint();
      y = (new_y_size * y) / old_y_size;
      f->SetPoint((Iunit)y);
    }
  }
  
  UpdateSize();
  UpdateAllframe();
}

void 
FissionFrame::ShiftXPosition(Iunit l_lim, Iunit offset)
{
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Vertical) 
    {
      Iunit x = f->GetPoint();
      if ((x >= l_lim) && (f != right)) 
      {
	f->SetPoint(x + offset);
      }
    }
  }
  UpdateAllframe();
}

void 
FissionFrame::ShiftYPosition(Iunit l_lim, Iunit offset)
{
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Horizontal) 
    {
      Iunit y = f->GetPoint();
      if ((y >= l_lim) && (f != bottom))
      {
	f->SetPoint(y + offset);
      }
    }
  }
  UpdateAllframe();
}

void 
FissionFrame::ShiftXPositionX(Iunit offset)
{
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Vertical) 
    {
      Iunit x = f->GetPoint();
      f->SetPoint(x + offset);
    }
  }
  UpdateSize();
  UpdateAllframe();
}

void 
FissionFrame::ShiftYPositionX(Iunit offset)
{
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Horizontal) 
    {
      Iunit y = f->GetPoint();
      f->SetPoint(y + offset);
    }
  }
  UpdateSize();
  UpdateAllframe();
}

// change fission frame size with no scalling
void 
FissionFrame::StretchSize(Iunit top_diff, Iunit bot_diff,
			   Iunit lef_diff, Iunit rig_diff)
{
  // normalize argument
  Vector org_size = GetSize();
  top_diff = - ((org_size.y < - top_diff) ? org_size.y : - top_diff);
  bot_diff = - ((org_size.y < - bot_diff) ? org_size.y : - bot_diff);
  lef_diff = - ((org_size.x < - lef_diff) ? org_size.x : - lef_diff);
  rig_diff = - ((org_size.x < - rig_diff) ? org_size.x : - rig_diff);
  if (org_size.y < - (top_diff + bot_diff)) 
  {
    top_diff = - ((- top_diff + org_size.y + bot_diff) / 2);
    bot_diff = - (org_size.y + top_diff);
  }
  if (org_size.x < - (lef_diff + rig_diff)) 
  {
    lef_diff = - ((- lef_diff + org_size.x + rig_diff) / 2);
    rig_diff = - (org_size.x + lef_diff);
  }
  
  Fission* f = 0;
  //
  // change top
  //
  if (top_diff > 0) 
  {
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if ((f->GetDir() == Fission::Horizontal) &&
	  (f != GetTop())) 
      {
	f->SetPoint(f->GetPoint() + top_diff);
      }
    }
  }
  if (top_diff < 0) 
  {
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if (f->GetDir() == Fission::Horizontal) 
      {
	Iunit y = f->GetPoint() + top_diff;
	if (y < 0) 
	{
	  y = 0;
	}
	f->SetPoint(y);
      }
    }
  }
  
  //
  // change bottom
  //
  if (bot_diff > 0) 
  {
    f = GetBottom();
    f->SetPoint(f->GetPoint() + bot_diff);
  }
  if (bot_diff < 0) 
  {
    Iunit new_pos = org_size.y + bot_diff;
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if (f->GetDir() == Fission::Horizontal) 
      {
	Iunit y = f->GetPoint();
	if (y > new_pos) 
	{
	  f->SetPoint(new_pos);
	}
      }
    }
  }
  
  //
  // change left
  //
  if (lef_diff > 0) 
  {
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if ((f->GetDir() == Fission::Vertical) &&
	  (f != GetLeft())) 
      {
	f->SetPoint(f->GetPoint() + lef_diff);
      }
    }
  }
  if (lef_diff < 0) 
  {
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if (f->GetDir() == Fission::Vertical) 
      {
	Iunit x = f->GetPoint() + lef_diff;
	if (x < 0) 
	{
	  x = 0;
	}
	f->SetPoint(x);
      }
    }
  }
  
  //
  // change right
  //
  if (rig_diff > 0) 
  {
    f = GetRight();
    f->SetPoint(f->GetPoint() + rig_diff);
  }
  if (rig_diff < 0) 
  {
    Iunit new_pos = org_size.x + rig_diff;
    FissionIterator fit(this);
    while ((f = fit()) != 0) 
    {
      if (f->GetDir() == Fission::Vertical) 
      {
	Iunit x = f->GetPoint();
	if (x > new_pos) 
	{
	  f->SetPoint(new_pos);
	}
      }
    }
  }
  
  //
  // size change report to frame object
  //
  UpdateSize();
  UpdateAllframe();
}

// change size and margin
void 
FissionFrame::ChangeSizeAndMargin(Iunit width, Iunit height,
				   Iunit top_m, Iunit bottom_m,
				   Iunit left_m, Iunit right_m,
				   Fission* top, Fission* bottom,
				   Fission* left, Fission* right)
{
  // size check
  Vector org_size = GetSize();
  Lunit old_top_m    = top->GetPoint();
  Lunit old_bottom_m = bottom->GetPoint();
  Lunit old_left_m   = left->GetPoint();
  Lunit old_right_m  = right->GetPoint();
  Lunit old_x_size   = org_size.x;
  Lunit old_y_size   = org_size.y;
  Lunit new_x_size   = width;
  Lunit new_y_size   = height;
  Lunit new_top_m    = top_m;
  Lunit new_bottom_m = bottom_m;
  Lunit new_left_m   = left_m;
  Lunit new_right_m  = right_m;
  Lunit new_right_diff = new_x_size - new_right_m;
  Lunit old_right_diff = old_x_size - old_right_m;
  Lunit new_bottom_diff = new_y_size - new_bottom_m;
  Lunit old_bottom_diff = old_y_size - old_bottom_m;
  
  // scale new size
  FissionIterator fit(this);
  Fission* f = 0;
  while ((f = fit()) != 0) 
  {
    if (f->GetDir() == Fission::Vertical) 
    {
      Lunit x = f->GetPoint();
      if (x <= old_left_m) 
      {
	if (old_left_m != 0L) 
	{
	  x = (new_left_m * x) / old_left_m;
	}
	else if (f == left) 
	{
	  x = new_left_m;
	}
	else 
	{
	  x = 0L;
	}
      }
      else if (x >= old_right_m) 
      {
	if (old_right_diff != 0L) 
	{
	  x = new_right_diff * (x - old_x_size) / old_right_diff
	  + new_x_size;
	}
	else if (f == right) 
	{
	  x = new_right_m;
	}
	else 
	{
	  x = new_x_size;
	}
      }
      else 
      {
	x = (x - old_left_m) * (new_right_m - new_left_m)
            / (old_right_m - old_left_m) + new_left_m;
      }
      f->SetPoint((Iunit)x);
    }
    else 
    {          // Fission::Horizontal
      Lunit y = f->GetPoint();
      if (y <= old_top_m) 
      {
	if (old_top_m != 0L) 
	{
	  y = (new_top_m * y) / old_top_m;
	}
	else if (f == top) 
	{
	  y = new_top_m;
	}
	else 
	{
	  y = 0L;
	}
      }
      else if (y >= old_bottom_m) 
      {
	if (old_bottom_diff != 0L) 
	{
	  y = new_bottom_diff * (y - old_y_size) / old_bottom_diff
	  + new_y_size;
	}
	else if (f == bottom) 
	{
	  y = new_bottom_m;
	}
	else 
	{
	  y = new_y_size;
	}
      }
      else 
      {
	y = (y - old_top_m) * (new_bottom_m - new_top_m)
	    / (old_bottom_m - old_top_m) + new_top_m;
      }
      f->SetPoint((Iunit)y);
    }
  }
  
  UpdateSize();
  UpdateAllframe();
}

// check rim fission
bool 
FissionFrame::IsRimFission(Fission* f)
{
  if ((f == top) ||
      (f == bottom) ||
      (f == left) ||
      (f == right)) 
  {
    return True;
  }
  return False;
}

// check vanish fission
bool 
FissionFrame::CheckVanish(Fission* v)
{
  return v->CheckVanish();
}

// check cut fission part
bool 
FissionFrame::CheckVanishPart(Fission* f, Fission* cut_start, 
			       Fission* cut_end)
{
  if ((f->GetStartFission() == cut_start) &&
      (f->GetEndFission() == cut_end)) 
  {
    // cut all
    return CheckVanish(f);
  }
  
  CrossPoint* start;
  CrossPoint* end;
  if ((end = f->FindCrossPoint(cut_end)) == 0) 
  {
    return False;	// error
  }
  if ((start = f->FindCrossPoint(cut_start)) == 0) 
  {
    return False;	// error
  }
  
  if (f->GetStartFission() == cut_start) 
  {
    // cut start part
    return f->CheckShortenStart(end);
  }
  if (f->GetEndFission() == cut_end) 
  {
    // cut end part
    return f->CheckShortenEnd(start);
  }
  // cut middle part
  return f->CheckCutMiddle(start, end);
}

// find move limit
void 
FissionFrame::FindMoveLimit(Fission* f, Iunit& l_move_lim, 
			     Iunit& u_move_lim)
{
  CrossPoint* cp = f->GetStartCrossPoint();
  CrossPoint* end = f->GetEndCrossPoint();
  if (f->GetDir() == Fission::Vertical) 
  {
    if (cp->GetHPrevCell() != 0) 
    {
      l_move_lim = cp->GetHPrevCell()->X();
      for (cp = cp->GetVNext(); cp != end; cp = cp->GetVNext()) 
      {
	if ((cp->GetHPrev() != 0) && 
	    (cp->GetHPrev()->X() > l_move_lim)) 
	{
	  l_move_lim = cp->GetHPrev()->X();
	}
      }
    }
    else 
    {
      l_move_lim = cp->X();
    }
    cp = f->GetStartCrossPoint();
    if (cp->GetHNextCell() != 0) 
    {
      u_move_lim = cp->GetHNextCell()->X();
      for (cp = cp->GetVNext(); cp != end; cp = cp->GetVNext()) 
      {
	if ((cp->GetHNext() != 0) &&
	    (cp->GetHNext()->X() < u_move_lim)) 
	{
	  u_move_lim = cp->GetHNext()->X();
	}
      }
    }
    else 
    {
      u_move_lim = cp->X();
    }
  }
  else 
  {		// Fission::Horizontal
    if (cp->GetVPrevCell() != 0) 
    {
      l_move_lim = cp->GetVPrevCell()->Y();
      for (cp = cp->GetHNext(); cp != end; cp = cp->GetHNext()) 
      {
	if ((cp->GetVPrev() != 0) && 
	    (cp->GetVPrev()->Y() > l_move_lim)) 
	{
	  l_move_lim = cp->GetVPrev()->Y();
	}
      }
    }
    else 
    {
      l_move_lim = cp->Y();
    }
    cp = f->GetStartCrossPoint();
    if (cp->GetVNextCell() != 0) 
    {
      u_move_lim = cp->GetVNextCell()->Y();
      for (cp = cp->GetHNext(); cp != end; cp = cp->GetHNext()) 
      {
	if ((cp->GetVNext() != 0) &&
	    (cp->GetVNext()->Y() < u_move_lim)) 
	{
	  u_move_lim = cp->GetVNext()->Y();
	}
      }
    }
    else 
    {
      u_move_lim = cp->Y();
    }
  }
}

bool 
FissionFrame::MergeOneStep()
{
  FissionIterator fi(this);
  Fission* f;
  while ((f = fi()) != 0) 
  {
    CrossPoint* end = f->GetEndCrossPoint();
    Fission* f_end = f->GetEndFission();
    if ((end != f_end->GetStartCrossPoint()) &&
	(end != f_end->GetEndCrossPoint())) 
    {
      if (f_end->GetDir() == Fission::Vertical) 
      {
	CrossPoint* cp = end->GetVPrev();		 
	if ((cp != 0) &&
	    (cp->GetKind() & CrossPoint::HStart) &&
	    (cp->Y() == end->Y())) 
	{
	  // find!
	  ((HFission*)f)->MergeFission(cp->GetHFission());
	  return True;
	}
	cp = end->GetVNext();
	if ((cp != 0) &&
	    (cp->GetKind() & CrossPoint::HStart) &&
	    (cp->Y() == end->Y())) 
	{
	  // find!
	  ((HFission*)f)->MergeFission(cp->GetHFission());
	  return True;
	}
      }
      else 
      {
	CrossPoint* cp = end->GetHPrev();
	if ((cp != 0) &&
	    (cp->GetKind() & CrossPoint::VStart) &&
	    (cp->X() == end->X())) 
	{
	  // find!
	  ((VFission*)f)->MergeFission(cp->GetVFission());
	  return True;
	}
	cp = end->GetHNext();
	if ((cp != 0) &&
	    (cp->GetKind() & CrossPoint::VStart) &&
	    (cp->X() == end->X())) 
	{
	  // find!
	  ((VFission*)f)->MergeFission(cp->GetVFission());
	  return True;
	}
      }
    }
  }
  return False;
}

void 
FissionFrame::MergeFission()
{
  while (MergeOneStep())
  {
    ;// do nothing
  }
}

// expand x minimum direction and create new frame
void 
FissionFrame::PrevXInsertSpecial(Iunit size, uword border_index)
{
  // cleate new left
  left = left->PrevXInsertSpecial(size, border_index);
}

// expand y minimun direction and create new frame
void 
FissionFrame::PrevYInsertSpecial(Iunit size, uword border_index)
{
  // cleate new top
  top = top->PrevYInsertSpecial(size, border_index);
}

void 
FissionFrame::SetSpecialCheckP(bool s)
{
  CPIterator cpi(this);
  CrossPoint* cp;
  while ((cp = cpi()) != 0)
  {
    cp->SetSpecialCheckP(s);
  }
}

void 
FissionFrame::DisconnectOverlappedLine(FrameRectRgn& rgn)
{
  HFission* r_top = rgn.GetTop();
  HFission* r_bot = rgn.GetBottom();
  VFission* r_lft = rgn.GetLeft();
  VFission* r_rgh = rgn.GetRight();
  Iunit r_y1 = rgn.Y1();
  Iunit r_y2 = rgn.Y2();
  Iunit r_x1 = rgn.X1();
  Iunit r_x2 = rgn.X2();
  
  // disconnect overlapped lines
  FList fl1h;	// these h-fissions overlap the start and the end position
  FList fl2h;	// these h-fissions overlap the start position
  FList fl3h;	// these h-fissions overlap the end position
  FList fl1v;	// these v-fissions overlap the start and the end position
  FList fl2v;	// these v-fissions overlap the start position
  FList fl3v;	// these v-fissions overlap the end position
  
  FissionIterator fi(this);
  Fission* f;
  // get horizontal disconnect fissions
  while ((f = fi()) != 0)
  {
    if (f->GetDir() == Fission::Horizontal)
    {
      if ((f->GetPoint() > r_y1) &&
	  (f->GetPoint() < r_y2))
      {
	if (f->GetStartCrossPoint()->X() < r_x1)
	{
	  if (f->GetEndCrossPoint()->X() > r_x2)
	  {
	    fl1h.Inject(f);
	  }
	  else
	  {
	    fl2h.Inject(f);
	  }
	}
	else if (f->GetEndCrossPoint()->X() > r_x2)
	{
	  fl3h.Inject(f);
	}
      }
    }
    else
    {
      if ((f->GetPoint() > r_x1) &&
	  (f->GetPoint() < r_x2))
      {
	if (f->GetStartCrossPoint()->Y() < r_y1)
	{
	  if (f->GetEndCrossPoint()->Y() > r_y2)
	  {
	    fl1v.Inject(f);
	  }
	  else
	  {
	    fl2v.Inject(f);
	  }
	}
	else if (f->GetEndCrossPoint()->Y() > r_y2)
	{
	  fl3v.Inject(f);
	}
      }
    }
  }
  
  // disconnect horizontal lines
  FListIterator fli1h(fl1h);
  while ((f = fli1h()) != 0) 
  {
    CrossPoint* pos = r_lft->FindCrossPoint(f);
    if (pos != 0)
    {
      Fission* f2 = f->Disconnect(pos);
      pos = r_rgh->FindCrossPoint(f2);
      if (pos != 0)
      {
	f2->Disconnect(pos);
      }
    }
  }
  FListIterator fli2h(fl2h);
  while ((f = fli2h()) != 0)
  {
    CrossPoint* pos = r_lft->FindCrossPoint(f);
    if (pos != 0)
    {
      f->Disconnect(pos);
    }
  }
  FListIterator fli3h(fl3h);
  while ((f = fli3h()) != 0)
  {
    CrossPoint* pos = r_rgh->FindCrossPoint(f);
    if (pos != 0)
    {
      f->Disconnect(pos);
    }
  }
  
  // disconnent vertical lines
  FListIterator fli1v(fl1v);
  while ((f = fli1v()) != 0) 
  {
    CrossPoint* pos = r_top->FindCrossPoint(f);
    if (pos != 0)
    {
      Fission* f2 = f->Disconnect(pos);
      pos = r_bot->FindCrossPoint(f2);
      if (pos != 0)
      {
	f2->Disconnect(pos);
      }
    }
  }
  FListIterator fli2v(fl2v);
  while ((f = fli2v()) != 0)
  {
    CrossPoint* pos = r_top->FindCrossPoint(f);
    if (pos != 0)
    {
      f->Disconnect(pos);
    }
  }
  FListIterator fli3v(fl3v);
  while ((f = fli3v()) != 0)
  {
    CrossPoint* pos = r_bot->FindCrossPoint(f);
    if (pos != 0)
    {
      f->Disconnect(pos);
    }
  }
}

void 
FissionFrame::ClearSingleFrameRegion(FrameRectRgn& rgn)
{
  DisconnectOverlappedLine(rgn);
  Iunit x1 = rgn.X1();
  Iunit x2 = rgn.X2();
  Iunit y1 = rgn.Y1();
  Iunit y2 = rgn.Y2();
  bool done_flag = True;
  while (done_flag)
  {
    done_flag = False;
    Fission* f;
    FissionIterator fi(this);
    while ((f = fi()) != 0) 
    {
      CrossPoint* cp_s = f->GetStartCrossPoint();
      CrossPoint* cp_e = f->GetEndCrossPoint();
      if ((f != rgn.GetTop()) && 
	  (f != rgn.GetBottom()) &&
	  (f != rgn.GetLeft()) && 
	  (f != rgn.GetRight()) &&
	  (cp_s->X() >= x1) && (cp_s->X() <= x2) &&
	  (cp_e->X() >= x1) && (cp_e->X() <= x2) &&
	  (cp_s->Y() >= y1) && (cp_s->Y() <= y2) &&
	  (cp_e->Y() >= y1) && (cp_e->Y() <= y2))
      {
	delete f;
	done_flag = True;
	break;
      }
    }
  }
  MergeFission();
}

void 
FissionFrame::CombineFrameRegion(FrameRectRgn& rgn,
				  bool combine_vertical,
				  bool combine_horizontal)
{
  if (combine_vertical && combine_horizontal)
  {
    ClearSingleFrameRegion(rgn);
  }
  DisconnectOverlappedLine(rgn);
  Iunit x1 = rgn.X1();
  Iunit x2 = rgn.X2();
  Iunit y1 = rgn.Y1();
  Iunit y2 = rgn.Y2();
  bool done_flag = True;
  while (done_flag)
  {
    done_flag = False;
    Fission* f;
    FissionIterator fi(this);
    while ((f = fi()) != 0) 
    {
      if (f->GetDir() == Fission::Vertical)
      {
	if (combine_horizontal)
	{
	  CrossPoint* cp_s = f->GetStartCrossPoint();
	  CrossPoint* cp_e = f->GetEndCrossPoint();
	  if ((f != rgn.GetLeft()) && 
	      (f != rgn.GetRight()) &&
	      (cp_s->X() >= x1) && (cp_s->X() <= x2) &&
	      (cp_e->X() >= x1) && (cp_e->X() <= x2) &&
	      (cp_s->Y() >= y1) && (cp_s->Y() <= y2) &&
	      (cp_e->Y() >= y1) && (cp_e->Y() <= y2))
	  {
	    delete f;
	    done_flag = True;
	    break;
	  }
	}
      }
      else // horizontal
      {
	if (combine_vertical)
	{
	  CrossPoint* cp_s = f->GetStartCrossPoint();
	  CrossPoint* cp_e = f->GetEndCrossPoint();
	  if ((f != rgn.GetTop()) && 
	      (f != rgn.GetBottom()) &&
	      (cp_s->X() >= x1) && (cp_s->X() <= x2) &&
	      (cp_e->X() >= x1) && (cp_e->X() <= x2) &&
	      (cp_s->Y() >= y1) && (cp_s->Y() <= y2) &&
	      (cp_e->Y() >= y1) && (cp_e->Y() <= y2))
	  {
	    delete f;
	    done_flag = True;
	    break;
	  }
	}
      }
    }
  }
  MergeFission();
}

// for insert frame
void 
FissionFrame::MakeDuplicateRegion(FrameRectRgn& rgn, 
				   bool post_insert, 
				   bool insert_line)
{
  if (insert_line)
  {
    uword border_index = rgn.GetTop()->GetStartCrossPoint()->GetHBorder();
    if ((rgn.GetTop() == GetTop()) &&
	(rgn.GetBottom() != GetBottom()))
    {
      border_index = rgn.GetBottom()->GetStartCrossPoint()->GetHBorder();
    }
    Iunit stretch_size = rgn.Y2() - rgn.Y1();
    if ((GetTop() == rgn.GetTop()) && !post_insert)
    {
      PrevYInsertSpecial(stretch_size, border_index);
      rgn.SetBottom(rgn.GetTop());
      rgn.SetTop(GetTop());
    }
    else 
    {
      Iunit y = rgn.Y2();
      StretchSize(0, stretch_size, 0, 0);
      if (post_insert)
      {
	ShiftYPosition(y, stretch_size);
	HFission* hf = Divide(rgn.GetLeft(), rgn.GetRight(), 
			       y, border_index);
	rgn.SetTop(hf);
      }
      else
      {
	y = rgn.Y1();
	ShiftYPosition(y, stretch_size);
	HFission* hf = Divide(rgn.GetLeft(), rgn.GetRight(),
			       y, border_index);
	rgn.SetBottom(rgn.GetTop());
	rgn.SetTop(hf);
      }
      ClearSingleFrameRegion(rgn);
    }
  }
  else
  {
    uword border_index = rgn.GetLeft()->GetStartCrossPoint()->GetVBorder();
    if ((rgn.GetLeft() == GetLeft()) &&
	(rgn.GetRight() != GetRight()))
    {
      border_index = rgn.GetRight()->GetStartCrossPoint()->GetVBorder();
    }
    Iunit stretch_size = rgn.X2() - rgn.X1();
    if ((GetLeft() == rgn.GetLeft()) && !post_insert)
    {
      PrevXInsertSpecial(stretch_size, border_index);
      rgn.SetRight(rgn.GetLeft());
      rgn.SetLeft(GetLeft());
    }
    else 
    {
      Iunit x = rgn.X2();
      StretchSize(0, 0, 0, stretch_size);
      if (post_insert)
      {
	ShiftXPosition(x, stretch_size);
	VFission* vf = Divide(rgn.GetTop(), rgn.GetBottom(), 
			       x, border_index);
	rgn.SetLeft(vf);
      }
      else
      {
	x = rgn.X1();
	ShiftXPosition(x, stretch_size);
	VFission* vf = Divide(rgn.GetTop(), rgn.GetBottom(),
			       x, border_index);
	rgn.SetRight(rgn.GetLeft());
	rgn.SetLeft(vf);
      }
      ClearSingleFrameRegion(rgn);
    }
  }
}

void 
FissionFrame::DeleteRegion(FrameRectRgn& rgn, bool delete_line)
{
  ClearSingleFrameRegion(rgn);
  
  Fission* vanish_fission;
  if (delete_line)
  {
    if (rgn.GetBottom() != GetBottom())
    {
      // vanish rgn.GetBottom()
      vanish_fission = rgn.GetBottom();
      rgn.GetBottom()->GetStartCrossPoint()
      ->ExchangeFrameObject(rgn.GetTop()->GetStartCrossPoint());
      
      for (CrossPoint* cp = (rgn.GetBottom()->GetStartCrossPoint()
			     ->GetHNext());
	   cp != rgn.GetBottom()->GetEndCrossPoint();
	   cp = cp->GetHNext())
      {
	if (cp->GetKind() & CrossPoint::VStart)
	{
	  cp->GetVFission()->StretchStart(rgn.GetTop());
	  cp->ExchangeFrameObject(cp->GetVFission()->GetStartCrossPoint());
	}
      }
      Iunit y2 = rgn.Y2();
      Iunit stretch_size = y2 - rgn.Y1();
      delete vanish_fission;
      FissionIterator fit(this);
      Fission* f;
      while ((f = fit()) != 0) 
      {
	if (f->GetDir() == Fission::Horizontal)
	{
	  Iunit y = f->GetPoint();
	  if (y > y2)
	  {
	    f->SetPoint(y - stretch_size);
	  }
	}
      }
    }
    else
    {
      // vanish rgn.GetTop()
      vanish_fission = rgn.GetTop();
      for (CrossPoint* cp = rgn.GetTop()->GetStartCrossPoint()->GetHNext();
	   cp != rgn.GetTop()->GetEndCrossPoint();
	   cp = cp->GetHNext())
      {
	if (cp->GetKind() & CrossPoint::VEnd)
	{
	  cp->GetVFission()->StretchEnd(rgn.GetBottom());
	}
      }
      Iunit stretch_size = rgn.Y2() - rgn.Y1();
      delete vanish_fission;
      GetBottom()->SetPoint(GetBottom()->GetPoint() - stretch_size);
    }
  }
  else
  {
    if (rgn.GetRight() != GetRight())
    {
      // vanish rgn.GetRight()
      vanish_fission = rgn.GetRight();
      rgn.GetRight()->GetStartCrossPoint()
      ->ExchangeFrameObject(rgn.GetLeft()->GetStartCrossPoint());
      for (CrossPoint* cp = (rgn.GetRight()->GetStartCrossPoint()
			     ->GetVNext());
	   cp != rgn.GetRight()->GetEndCrossPoint();
	   cp = cp->GetVNext())
      {
	if (cp->GetKind() & CrossPoint::HStart)
	{
	  cp->GetHFission()->StretchStart(rgn.GetLeft());
	  cp->ExchangeFrameObject(cp->GetHFission()->GetStartCrossPoint());
	}
      }
      Iunit x2 = rgn.X2();
      Iunit stretch_size = x2 - rgn.X1();
      delete vanish_fission;
      FissionIterator fit(this);
      Fission* f;
      while ((f = fit()) != 0) 
      {
	if (f->GetDir() == Fission::Vertical)
	{
	  Iunit x = f->GetPoint();
	  if (x > x2)
	  {
	    f->SetPoint(x - stretch_size);
	  }
	}
      }
    }
    else
    {
      // vanish rgn.GetLeft()
      vanish_fission = rgn.GetLeft();
      for (CrossPoint* cp = rgn.GetLeft()->GetStartCrossPoint()->GetVNext();
	   cp != rgn.GetLeft()->GetEndCrossPoint();
	   cp = cp->GetVNext())
      {
	if (cp->GetKind() & CrossPoint::HEnd)
	{
	  cp->GetHFission()->StretchEnd(rgn.GetRight());
	}
      }
      Iunit stretch_size = rgn.X2() - rgn.X1();
      delete vanish_fission;
      GetRight()->SetPoint(GetRight()->GetPoint() - stretch_size);
    }
  }
  
  UpdateSize();
  UpdateAllframe();
  
  // connect lines 
  MergeFission();
}

#define DIVMAX 16384

void
FissionFrame::AlignFrameRegion(FrameRectRgn& rgn,
				bool align_vertical,
				bool align_horizontal)
{
  DisconnectOverlappedLine(rgn);
  
  // vertical align
  if (align_vertical)
  {
    // get divisions
    HFission* top = rgn.GetTop();
    HFission* bot = rgn.GetBottom();
    CrossPoint* end_cp = rgn.GetTopRight();
    CrossPoint* cp = rgn.GetTopLeft()->GetHNext();
    uword div = 1;
    while (cp != end_cp)
    {
      if ((cp->GetKind() & CrossPoint::VStart) &&
	  (cp->GetVFission()->GetEndCrossPoint()->GetHFission() == bot))
      {
	div ++;
      }
      cp = cp->GetHNext();
    }
    if ((div > 1) && (div < DIVMAX))
    {
      FList fl;
      {
	RectRgnVFissionIterator rrvfi(rgn);
	VFission* f;
	while ((f = rrvfi()) != 0)
	{
	  if ((f->GetStartCrossPoint()->GetHFission() != top) ||
	      (f->GetEndCrossPoint()->GetHFission() != bot))
	  {
	    fl.Inject(f);
	  }
	}
      }
      // 
      uword n = 1;
      Iunit x1 = rgn.X1();
      Iunit x2;
      Iunit unit = (rgn.X2() - rgn.X1()) / div;
      Iunit new_x1 = x1;
      Iunit new_x2 = x1 + unit;
      end_cp = rgn.GetTopRight();
      cp = rgn.GetTopLeft()->GetHNext();
      for (;;)
      {
	if ((cp->GetKind() & CrossPoint::VStart) &&
	    (cp->GetVFission()->GetEndCrossPoint()->GetHFission() == bot))
	{
	  x2 = cp->X();
	  FListIterator fli(fl);
	  FList remove;
	  Fission* f;
	  while ((f = fli()) != 0)
	  {
	    Iunit x = f->GetPoint();
	    if ((x > x1) && (x < x2))
	    {
	      dword new_pos = (((dword)(x - x1)) * unit) / (dword)(x2 - x1);
	      f->SetPoint((word)new_pos + new_x1);
	      remove.Inject(f);
	    }
	  }
	  FListIterator fliremove(remove);
	  while ((f = fliremove()) != 0)
	  {
	    fl.Delete(f);
	  }
	  if (n < div)
	  {
	    cp->GetVFission()->SetPoint(new_x2);
	  }
	  x1 = x2;
	  new_x1 += unit;
	  new_x2 += unit;
	  n++;
	}
	if (cp == end_cp)
	{
	  break;
	}
	cp = cp->GetHNext();
      }
    }
  }
  
  // horizontal align
  if (align_horizontal)
  {
    // get divisions
    VFission* lft = rgn.GetLeft();
    VFission* rgh = rgn.GetRight();
    CrossPoint* end_cp = rgn.GetBottomLeft();
    CrossPoint* cp = rgn.GetTopLeft()->GetVNext();
    uword div = 1;
    while (cp != end_cp)
    {
      if ((cp->GetKind() & CrossPoint::HStart) &&
	  (cp->GetHFission()->GetEndCrossPoint()->GetVFission() == rgh))
      {
	div ++;
      }
      cp = cp->GetVNext();
    }
    if ((div > 1) && (div < DIVMAX))
    {
      FList fl;
      {
	RectRgnHFissionIterator rrvfi(rgn);
	HFission* f;
	while ((f = rrvfi()) != 0)
	{
	  if ((f->GetStartCrossPoint()->GetVFission() != lft) ||
	      (f->GetEndCrossPoint()->GetVFission() != rgh))
	  {
	    fl.Inject(f);
	  }
	}
      }
      // 
      uword n = 1;
      Iunit y1 = rgn.Y1();
      Iunit y2;
      Iunit unit = (rgn.Y2() - rgn.Y1()) / div;
      Iunit new_y1 = y1;
      Iunit new_y2 = y1 + unit;
      end_cp = rgn.GetBottomLeft();
      cp = rgn.GetTopLeft()->GetVNext();
      for (;;)
      {
	if ((cp->GetKind() & CrossPoint::HStart) &&
	    (cp->GetHFission()->GetEndCrossPoint()->GetVFission() == rgh))
	{
	  y2 = cp->Y();
	  FListIterator fli(fl);
	  FList remove;
	  Fission* f;
	  while ((f = fli()) != 0)
	  {
	    Iunit y = f->GetPoint();
	    if ((y > y1) && (y < y2))
	    {
	      dword new_pos = (((dword)(y - y1)) * unit) / (dword)(y2 - y1);
	      f->SetPoint((word)new_pos + new_y1);
	      remove.Inject(f);
	    }
	  }
	  FListIterator fliremove(remove);
	  while ((f = fliremove()) != 0)
	  {
	    fl.Delete(f);
	  }
	  if (n < div)
	  {
	    cp->GetHFission()->SetPoint(new_y2);
	  }
	  y1 = y2;
	  new_y1 += unit;
	  new_y2 += unit;
	  n++;
	}
	if (cp == end_cp)
	{
	  break;
	}
	cp = cp->GetVNext();
      }
    }
  }
  MergeFission();
  UpdateSize();
  UpdateAllframe();
}

