// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: fonts.cpp,v 3.7 1999/07/25 07:00:01 kudou Exp $
// implement FontTable and FontEntry classes for font info

#include "pword.h"
#include "fonts.h"
#include "attribut.h"
#include "atts.h"
#include "menu.h"
#include "pmenus.h"
#include "pref.h"
#include "pstreams.h"
#include "view.h"
#include "xstr.h"
#include <math.h>
#include <memory.h>

// `FontEntry.virtual_font_type'̒lB
// ʏNO_VIRTUAL_FONTɂȂBVIRTUAL_MINCHO_FONT, VIRTUAL_GOTHIC_FONT́A
// ݂̊ōłꂢƎvuvAсuSVbNvtHg
// wB
// ̒lBeatWord`̃t@Cɏ܂B

#define NO_VIRTUAL_FONT 0
#define VIRTUAL_MINCHO_FONT 1
#define VIRTUAL_GOTHIC_FONT 2
#define VALID_VIRTUAL_FONT_TYPE_P(type) ((unsigned char) (type) < 3)

static unsigned int virtual_mincho_index = 0;
static unsigned int virtual_gothic_index = 0;

// ------------------------------------------------------------
// FontEntry class

// private method
unsigned char NEAR
FontEntry::GetHash()
{
  unsigned char* s = (unsigned char*) this->logfont.lfFaceName;
  unsigned char x = 0;
  unsigned char c;
  while ((c = *s++) != '\0')
  {
    x = x ^ (x << 3) ^ c;
  }
  return(x);
}

// equality of two font entries
int
operator == (const FontEntry& fte1, const FontEntry& fte2)
{
  return(fte1.virtual_font_type == fte2.virtual_font_type
	  && (strcmp((char*) &fte1.logfont.lfFaceName,
		      (char*) &fte2.logfont.lfFaceName)
	      == 0));
}

unsigned int
FontEntry::Enter()
{
  switch (this->virtual_font_type)
  {
   case VIRTUAL_MINCHO_FONT:
    return(virtual_mincho_index);
    
   case VIRTUAL_GOTHIC_FONT:
    return(virtual_gothic_index);
  }
  return(GlobalFontTable.MakeEntry(*this));
}

void
FontEntry::RemoveIfUnmarked(unsigned int i, void*)
{
  if (i != virtual_mincho_index
      && i != virtual_gothic_index && !GetMark() && !GetDead())
  {
    RemoveFromInterface(i);
    SetDead();
  }
}

void
FontEntry::AddToInterface(unsigned int i, void*)
{
  Menu_Type menu_type = Menu_Type_UNUSED_MAX;
  if (logfont.lfCharSet == SHIFTJIS_CHARSET)
  {
#ifdef BW3_DOLLERFONTOFF
    if ((logfont.lfFaceName[0] != '$' || Pref_show_unicode_font) &&
	(logfont.lfFaceName[0] != '@' || Pref_show_tategaki_font))
    {
      menu_type = Menu_Type_JAPANESE_FONT;
    }
#else /* BW3_DOLLERFONTOFF */
    if (logfont.lfFaceName[0] != '@' || Pref_show_tategaki_font)
    {
      menu_type = Menu_Type_JAPANESE_FONT;
    }
#endif /*BW3_DOLLERFONTOFF */ 
  }
  else
  {
    menu_type = Menu_Type_ENGLISH_FONT;
  }
  if (menu_type != Menu_Type_UNUSED_MAX)
  {
    Menu::add_menu(menu_type, i, (char*) this->logfont.lfFaceName);
  }
}

void
FontEntry::RemoveFromInterface(unsigned int i)
{
  if (!this->GetDead())
  {
    Menu::delete_menu((logfont.lfCharSet == SHIFTJIS_CHARSET
			? Menu_Type_JAPANESE_FONT : Menu_Type_ENGLISH_FONT),
		       i);
  }
}

void
FontEntry::WriteToStream(PStream* stream)
{
#ifdef _WIN32
  stream->StartBlockOut(PStream::HDR_FONTENTRY, 1);
  CHECK;
#else /* _WIN32 */
  stream->StartBlockOut(PStream::HDR_FONTENTRY);
  CHECK;
#endif /* _WIN32 */

  // It looks that Win32 only use uword part.
  // If not, I need to chage the bw file format. (kudou)
  stream->OutUWord((uword)font_type);
  CHECK;
  
  stream->Out(&logfont, sizeof(LOGFONT));
  CHECK;
  
  // BW2_FILE_FORMAT
  // New information on V2.0.4 or later.
  stream->OutByte(this->virtual_font_type);
  CHECK;
  
  stream->EndBlockOut();
}

#ifdef _WIN32
// Previous version of BW file stores LOGFONT.
struct W16LOGFONT
{
  word lfHeight;
  word lfWidth;
  word lfEscapement;
  word lfOrientation;
  word lfWeight;
  BYTE lfItalic;
  BYTE lfUnderline;
  BYTE lfStrikeOut;
  BYTE lfCharSet;
  BYTE lfOutPrecision;
  BYTE lfClipPrecision;
  BYTE lfQuality;
  BYTE lfPitchAndFamily;
  char lfFaceName[32];
public:
  void SetLOGFONT(LOGFONT& logfont)
  {
    logfont.lfHeight = lfHeight;
    logfont.lfWidth = lfWidth;
    logfont.lfEscapement = lfEscapement;
    logfont.lfOrientation = lfOrientation;
    logfont.lfWeight = lfWeight;
    logfont.lfItalic = lfItalic;
    logfont.lfUnderline = lfUnderline;
    logfont.lfStrikeOut = lfStrikeOut;
    logfont.lfCharSet = lfCharSet;
    logfont.lfOutPrecision = lfOutPrecision;
    logfont.lfClipPrecision = lfClipPrecision;
    logfont.lfQuality = lfQuality;
    logfont.lfPitchAndFamily = lfPitchAndFamily;
    assert (LF_FACESIZE == 32);
    for (int i = 0; i < 32; i++)
    {
      logfont.lfFaceName[i] = lfFaceName[i];
    }
  }
};
#else /* _WIN32 */
struct W32LOGFONT
{
  dword lfHeight;
  dword lfWidth;
  dword lfEscapement;
  dword lfOrientation;
  dword lfWeight;
  BYTE lfItalic;
  BYTE lfUnderline;
  BYTE lfStrikeOut;
  BYTE lfCharSet;
  BYTE lfOutPrecision;
  BYTE lfClipPrecision;
  BYTE lfQuality;
  BYTE lfPitchAndFamily;
  char lfFaceName[32];
public:
  void SetLOGFONT(LOGFONT& logfont)
  {
    logfont.lfHeight = (word)lfHeight;
    logfont.lfWidth = (word)lfWidth;
    logfont.lfEscapement = (word)lfEscapement;
    logfont.lfOrientation = (word)lfOrientation;
    logfont.lfWeight = (word)lfWeight;
    logfont.lfItalic = lfItalic;
    logfont.lfUnderline = lfUnderline;
    logfont.lfStrikeOut = lfStrikeOut;
    logfont.lfCharSet = lfCharSet;
    logfont.lfOutPrecision = lfOutPrecision;
    logfont.lfClipPrecision = lfClipPrecision;
    logfont.lfQuality = lfQuality;
    logfont.lfPitchAndFamily = lfPitchAndFamily;
    assert (LF_FACESIZE == 32);
    for (int i = 0; i < 32; i++)
    {
      logfont.lfFaceName[i] = lfFaceName[i];
    }
  }
};
#endif /* _WIN32 */

void
FontEntry::ReadFromStream(PStream* stream)
{
  word vers;
  
  stream->StartTypedBlockIn(PStream::HDR_FONTENTRY, vers);
  CHECK;
  
  font_type = stream->InUWord();
  CHECK;

#ifdef _WIN32
  switch (vers)
  {
  case 0:
    {
      W16LOGFONT w16logfont;
      stream->In(&w16logfont, sizeof(W16LOGFONT));
      CHECK;
      w16logfont.SetLOGFONT(logfont);
    }
    break;
    
  default:
    stream->In(&logfont, sizeof(LOGFONT));
    CHECK;
    break;
  }
#else /* _WIN32 */
  switch (vers)
  {
  case 0:
    stream->In(&logfont, sizeof(LOGFONT));
    CHECK;
    break;
    
  default:
    W32LOGFONT w32logfont;
    stream->In(&w32logfont, sizeof(W32LOGFONT));
    CHECK;
    w32logfont.SetLOGFONT(logfont);
    break;
  }
#if 0
  // original 16-bit code.   
  stream->In(&logfont, sizeof(LOGFONT));
  CHECK;
#endif
#endif /* _WIN32 */
  
  // BW2_FILE_FORMAT
  // New information on V2.0.4 or later.
  char virtual_font_type = (char) stream->InByte();
  switch (stream->GetError())
  {
   default:
    return;
    
   case SE_NOMORE:
    stream->SetError(SE_NOERROR);
    break;

   case SE_NOERROR:
    if (VALID_VIRTUAL_FONT_TYPE_P(virtual_font_type))
    {
      this->virtual_font_type = virtual_font_type;
    }
    break;
  }
  
  stream->EndBlockIn();
}

#ifndef NDEBUG
void
FontEntry::DumpHeader(FILE* f)
{
  fprintf(f, "\nFont Table\n\nNNNN Name\n");
}

void
FontEntry::Dump(unsigned int i, void* v)
{
  FILE* f = (FILE*) v;
  fprintf(f, "%4d %s\n", i, logfont.lfFaceName);
}
#endif

void
FontEntry::SetDead()
{
  if (!this->GetDead())
  {
    this->ImmediateSetDead();
    GlobalFontTable.RemoveHash(this);
  }
}

// public constructor
FontEntry::FontEntry()
{
  this->virtual_font_type = NO_VIRTUAL_FONT;
}

// construct from logfont and font type.
// public constructor
#ifdef _WIN32
FontEntry::FontEntry(LPLOGFONT logfont, DWORD font_type)
#else /* _WIN32 */
FontEntry::FontEntry(LPLOGFONT logfont, short font_type)
#endif /* _WIN32 */
{
  this->font_type = font_type;
  this->logfont = *logfont;
  this->virtual_font_type = NO_VIRTUAL_FONT;
}

// constructor for virtual font.
FontEntry::FontEntry(FontEntry& e, int virtual_font_type)
{
  this->font_type = e.font_type;
  this->logfont = e.logfont;
  this->virtual_font_type = virtual_font_type;
}

// ------------------------------------------------------------
// GLOBALS

#define T FontEntry
#define TABLE BaseFontTable
#define SEG BaseFontTableSeg
#define DEADABLE_ENTRY
#define DO_READ_WRITE
#define FORCE_MAKE_ENTRY
#include "tabletem.cpp"

FontTable STATIC_NEAR GlobalFontTable;

FontTable::FontTable()
{
  this->ansi_default = 0;
  this->kanji_default = 0;
}

// EnumAllFaces -- callback from EnumFonts; a friend of EnumFont
#ifdef _WIN32
int FAR PASCAL
EnumAllFaces_ (LPLOGFONT lplf,
	       LPTEXTMETRIC /*lptm*/, DWORD nFontType, LPARAM lp)
#else /* _WIN32 */
int FAR PASCAL
EnumAllFaces_ (LPLOGFONT lplf,
	       LPTEXTMETRIC /*lptm*/, short nFontType, FontTable* font_table)
#endif /* _WIN32 */
{
#ifdef _WIN32
  FontTable* font_table = (FontTable*)lp;
#endif /* _WIN32 */
  FontEntry ft_entry(lplf, nFontType);
  font_table->MakeEntry(ft_entry);
  return(1);
}

// from sdk31/include/windows.h
//#define RASTER_FONTTYPE     0x0001
//#define DEVICE_FONTTYPE     0X0002
#ifndef TRUETYPE_FONTTYPE
#define TRUETYPE_FONTTYPE   0x0004
#endif /* TRUETYPE_FONTTYPE */

#define ALL_FONTTYPE() \
(windows_has_true_type_p() \
 ? (RASTER_FONTTYPE | DEVICE_FONTTYPE | TRUETYPE_FONTTYPE) \
 : (RASTER_FONTTYPE | DEVICE_FONTTYPE))

struct FontValue
{
  int family;
  int type;
  int name_length;
  int name_value_0;
  char* name;
  double delta;
};

inline void NEAR
empty(FontValue* v)
{
  v->family = FF_DONTCARE;
  v->type = 0;
  v->name_length = LF_FACESIZE + 1;
  v->name_value_0 = 0;
  v->name = "\377";
  v->delta = 256.0;
}

static void NEAR
fill(FontValue* v, LOGFONT* l, int type)
{
  v->family = (l->lfPitchAndFamily & 0xf0);
  v->type = type & ALL_FONTTYPE();
  char* name = (char*) l->lfFaceName;
  v->name_length = strlen(name);
  v->name_value_0 = name[0] != '@';
  v->name = name;
  int width;
  v->delta = ((width = l->lfWidth) <= 0 ? 256.0
	      : fabs((double) l->lfHeight / width - 2));
}

static int NEAR
compare(FontValue* w0, FontValue* w1, int family)
{
  int result;
  if ((result = (w0->family == family) - (w1->family == family)) == 0
      && (result = w0->name_value_0 - w1->name_value_0) == 0
      && (result = w0->type - w1->type) == 0
      && ((result = w0->delta < w1->delta ? 1 : w1->delta < w0->delta ? -1 : 0)
	  == 0)
      && (result = w1->name_length - w0->name_length) == 0
      && (result = strcmp(w1->name, w0->name)) == 0)
  {
  }
  return(result);
}

// Fill -- go ahead and enumerate fonts
void
FontTable::Fill(HDC dc)
{
  EnumFonts(dc, 0, (FONTENUMPROC)EnumAllFaces, (LPARAM_T)this);

  // Get virtual font
  FontValue mincho_value[1];
  FontValue gothic_value[1];
  int mincho = 0;
  int gothic = 0;
  {
    empty(mincho_value);
    empty(gothic_value);
    unsigned int a = this->GetNEntries();
#ifdef BW3_DEFAULTFONT
    int def_mincho = 0;
    int def_gothic = 0;
    FontValue def_mincho_value[1];
    FontValue def_gothic_value[1];
#endif /* BW3_DEFAULTFONT */
    while (--a)
    {
      FontEntry& f = ::get_font_entry(a);
      if (!f.GetDead() && f.GetLOGFONT() .lfCharSet == SHIFTJIS_CHARSET)
      {
#ifdef BW3_DEFAULTFONT
	char* facename = f.GetLOGFONT().lfFaceName;
	if (facename[0] == '$' ||
	    (facename[0] == '@' && facename[1] == '$'))
	{
	  // for sutupid fonts
	  // JUST w$JSgothic, $JSmincho, @$JSgothic, @$JSmincho
	  continue;
	}
#endif /* BW3_DEFAULTFONT */
	FontValue value[1];
	fill(value, &f.GetLOGFONT(), f.GetFontType());
	if (compare(mincho_value, value, FF_ROMAN) <= 0)
	{
	  *mincho_value = *value;
	  mincho = a;
	}
	if (compare(gothic_value, value, FF_MODERN) <= 0)
	{
	  *gothic_value = *value;
	  gothic = a;
	}
#ifdef BW3_DEFAULTFONT
	if (Pref_default_mincho_font &&
	    strcmp(facename, Pref_default_mincho_font) == 0)
	{
	  *def_mincho_value = *value;
	  def_mincho = a;
	}
	if (Pref_default_gothic_font &&
	    strcmp(facename, Pref_default_gothic_font) == 0)
	{
	  *def_gothic_value = *value;
	  def_gothic = a;
	}
#endif /* BW3_DEFAULTFONT */
      }
    }
#ifdef BW3_DEFAULTFONT
    if (!Pref_auto_mingo_font_mechanism)
    {
      if (def_mincho)
      {
	*mincho_value = *def_mincho_value;
	mincho = def_mincho;
      }
      if (def_gothic)
      {
	*gothic_value = *def_gothic_value;
	gothic = def_gothic;
      }
    }
#endif /* BW3_DEFAULTFONT */
  }

  // register virtual fonts
  {
    FontEntry& src = ::get_font_entry(mincho);
    if (virtual_mincho_index == 0)
    {
      FontEntry e(src, VIRTUAL_MINCHO_FONT);
      // Don't use `FontEntry::Enter'.
      virtual_mincho_index = GlobalFontTable.MakeEntry(e);
    }
    else
    {
      FontEntry& e = ::get_font_entry(virtual_mincho_index);
      e.font_type = src.font_type;
      e.logfont = src.logfont;
    }
  }
  {
    FontEntry &src = ::get_font_entry(gothic);
    if (virtual_gothic_index == 0)
    {
      FontEntry e(src, VIRTUAL_GOTHIC_FONT);
      // Don't use `FontEntry::Enter'.
      virtual_gothic_index = GlobalFontTable.MakeEntry(e);
    }
    else
    {
      FontEntry& e = ::get_font_entry(virtual_gothic_index);
      e.font_type = src.font_type;
      e.logfont = src.logfont;
    }
  }
  
  // set default fonts
  int ansi_default = 0;
  int kanji_default = 0;
  int matched_ansi_default = False;
  int matched_kanji_default = False;
  if (Pref_default_ansi_font && stricmp(Pref_default_ansi_font, S_no) == 0)
  {
    // force `no romaji font'
    matched_ansi_default = True;
  }
  FontValue ansi_value[1];
  FontValue kanji_value[1];
  {
    empty(ansi_value);
    empty(kanji_value);
    unsigned int a = this->GetNEntries();
    while (--a)
    {
      FontEntry& f = ::get_font_entry(a);
      if (!f.GetDead())
      {
	LOGFONT& lf = f.GetLOGFONT();
#ifdef BW3_DEFAULTFONT
	char* facename = f.GetLOGFONT().lfFaceName;
	if (facename[0] == '$' ||
	    (facename[0] == '@' && facename[1] == '$'))
	{
	  // for sutupid fonts
	  // JUST w$JSgothic, $JSmincho, @$JSgothic, @$JSmincho
	  continue;
	}
#endif /* BW3_DEFAULTFONT */

	FontValue value[1];
	fill(value, &lf, f.GetFontType());
	switch (lf.lfCharSet)
	{
	 case ANSI_CHARSET:
	 case OEM_CHARSET:
	  if (!matched_ansi_default)
	  {
	    if (Pref_default_ansi_font
		&& Pref_default_ansi_font[0] != '\0'
		&& (stricmp((char*) lf.lfFaceName, Pref_default_ansi_font)
		    == 0))
	    {
	      matched_ansi_default = True;
	    }
	    else if (compare(ansi_value, value, FF_ROMAN) <= 0)
	    {
	      *ansi_value = *value;
	    }
	    else
	    {
	      break;
	    }
	    ansi_default = a;
	  }
	  break;
	  
	 case SHIFTJIS_CHARSET:
	  if (!matched_kanji_default)
	  {
#ifdef BW3_DEFAULTFONT
	    if (!Pref_auto_def_font_mechanism
	        && Pref_default_kanji_font
		&& Pref_default_kanji_font[0] != '\0'
		&& (stricmp((char*) lf.lfFaceName, Pref_default_kanji_font)
		    == 0))
#else /* BW3_DEFAULTFONT */
	    if (Pref_default_kanji_font
		&& Pref_default_kanji_font[0] != '\0'
		&& (stricmp((char*) lf.lfFaceName, Pref_default_kanji_font)
		    == 0))
#endif /* BW3_DEFAULTFONT */
	    {
	      matched_kanji_default = True;
	    }
	    else if (compare(kanji_value, value, FF_ROMAN) <= 0)
	    {
	      *kanji_value = *value;
	    }
	    else
	    {
	      break;
	    }
	    kanji_default = a;
	  }
	  break;
	}
      }
    }
  }
  if (!matched_kanji_default)
  {
    kanji_default = virtual_mincho_index;
  }
  AttsEntry &e = ::get_atts_entry(DefaultCharAtts);
  e.vals[CA_facename2] = this->kanji_default = kanji_default;
  e.vals[CA_facename] =
  this->ansi_default = ((kanji_default && !matched_ansi_default) ? 0
			: ansi_default);
}

// static method
void
FontTable::register_static_menus()
{
  Menu::add_static_menu(Menu_Type_ENGLISH_FONT,
			 0, S_no_romaji_font, IDM_More_NoRomajiFont);
  Menu::add_static_menu(Menu_Type_JAPANESE_FONT,
			 virtual_mincho_index,
			 S_virtual_mincho_name, IDM_More_VirtualMincho);
  Menu::add_static_menu(Menu_Type_JAPANESE_FONT,
			 virtual_gothic_index,
			 S_virtual_gothic_name, IDM_More_VirtualGothic);
}

void
FontTable::RemoveUnmarkedEntries()
{
  
  this->Apply(&FontEntry::RemoveIfUnmarked, 0);
}
