// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: interval.cpp,v 3.2 1999/05/12 00:22:16 kudou Exp $
// implement interval class

#include "pword.h"
#include "pmenus.h"
#include "attribut.h"
#include "docconte.h"
#include "docwindo.h"
#include "line.h"
#include "sjis.h"
#include "vtextpnl.h"
#include "pstreams.h"
#include "bufnew.h"
#include "textflow.h"
#include "bound.h"
#include "intlist.h"
#include "vmem.h"
#include "interval.h"

void
Interval::Init()
{
  this->BP0 = NULL;
  this->BP1 = NULL;
  this->atts = 0;
}

Interval::Interval()
{
  Init();
}

Interval::Interval(BPP s, BPP e)
{
  Init();
  Set(s, e);
}

Interval::Interval(BPP bpp)
{
  Init();
  Set(bpp);
}

Interval::Interval(Interval& i)
{
  Init();
  Set(i.BP0, i.BP1);
  this->atts = i.atts;
}

Interval::Interval(Interval* i)
{
  Init();
  Set(i->BP0, i->BP1);
  this->atts = i->atts;
}

Interval::Interval(EditText* e)
{
  Init();
  BP0 = e->GetStartBufferPointer();
  BP1 = e->GetEndBufferPointer();
}

Interval::Interval(TextFlow* tf)
{
  Init();
  BP0 = tf->GetStartBufferPointer();
  BP1 = tf->GetEndBufferPointer();
}

Interval::Interval(BoundInterval &bi, DocumentContent* dc)
{
  Init();
  TextFlow* tf = bi.GetTextFlow();
  if (tf == 0)
  {
    tf = dc->GetTextFlowBySerial(bi.GetTextFlowSerial());
  }
  BP0 = new BufferPointer(bi.GetBP0(), tf);
  BP1 = new BufferPointer(bi.GetBP1(), tf);
  if (!BP0->GetEditor())
  {
#ifndef NDEBUG
    if (!BP1->GetEditor())
    {
      syserr("Interval::Interval -- both BPs not mapped");
    }
#endif
    *BP0 = *BP1;
  }
  else if (!BP1->GetEditor())
  {
    *BP1 = *BP0;
  }
}

Interval::~Interval()
{
  if (BP1)
  {
    delete BP1;
  }
  if (BP0)
  {
    delete BP0;
  }
}

void
Interval::SetAtts(attr a)
{
  atts = a;
}

void
Interval::SetBP0(BPP bpp)
{
  BPP old = BP0;
  BP0 = new BufferPointer(bpp);
  if (old) {
    delete old;
  }
}

void
Interval::SetBP1(BPP bpp)
{
  BPP old = BP1;
  BP1 = new BufferPointer(bpp);
  if (old)
  {
    delete old;
  }
}

void
Interval::Set(BPP s, BPP e)
{
  BPP old0 = BP0, old1 = BP1;
  BP0 = new BufferPointer(s);
  BP1 = new BufferPointer(e);
  if (old0) delete old0;
  if (old1) delete old1;
}

void
Interval::Set(BPP s_and_e)
{
  Set(s_and_e, s_and_e);
}

void
Interval::Reverse()
{
  BPP old0 = BP0;
  BP0 = BP1;
  BP1 = old0;
}

inline signed int 
Interval::Sense()
{
  return ::BPcmp(*BP0, *BP1);
}

BPP
Interval::GetLeft()
{
  if (Sense() >= 0)   return BP0;
  else		       return BP1;
}

BPP
Interval::GetRight()
{
  if (Sense() < 0)  return BP0;
  else		     return BP1;
}

void
Interval::GetLeftRight(BufferPointer*& left, BufferPointer*& right)
{
  if (Sense() < 0)  
  {
    left = BP1;
    right = BP0;
  }
  else
  {
    left = BP0;
    right = BP1;
  }  
}

bool
Interval::PointP()
{
  return (bool)((*BP0 == *BP1) != 0);
}

// BPcmp tells you the relationship between a buffer pointer and "this" interval.
Interval::BPCMP Interval::BPcmp(BPP bpp)
{
  if (this->GetTextFlow() != bpp->GetTextFlow())
  {
    return BC_WRONGINT;
  }
  
  switch (Sense())
  {
   case -1:
   case 0:
         if (*bpp>*BP0)      return BC_BEHIND;
    else if (*bpp<*BP1)      return BC_BEYOND;
    else		      return BC_INSIDE;

   case 1:
         if (*bpp<*BP0)      return BC_BEHIND;
    else if (*bpp>*BP1)      return BC_BEYOND;
    else 		      return BC_INSIDE;
  }
  return BC_ERROR;  // keep compiler quiet
}

// Pointize does a low-level physical pointization of the interval.
void
Interval::GotoBeginningPoint()
{
  switch (Sense())
  {
    case -1: SetBP0(BP1); break;
    case 0:  break;
    case 1:  SetBP1(BP0); break;
  }
}

void
Interval::GotoEndPoint()
{
  switch (Sense())
  {
    case -1: SetBP1(BP0); break;
    case 0:  break;
    case +1: SetBP0(BP1); break;
  }
}

// Diff calculates the intervals that need to be added to and/or 
// subtracted from an interval to make it extend to a particular
// buffer pointer.  

void
Interval::Diff(BPP bpp, Interval*& add_me, Interval*& subtract_me)
{
  add_me = subtract_me = 0;
  
  if (*bpp==*BP1)
  {
    return;
  }
  switch (this->BPcmp(bpp))
  {
   // bpp is off in other direction from anchor point
   case BC_BEHIND: 
    subtract_me = new Interval(this);
    add_me = new Interval(bpp, BP0);
    break;
   
   // bpp is in interval    
   case BC_INSIDE:
    subtract_me = new Interval(bpp, BP1); 
    break;

   // bpp is further from anchor in same direction    
   case BC_BEYOND:
    add_me = new Interval(BP1, bpp); 
    break;
  }
}


void
Interval::DelayFormat()
{
  (this->GetTextFlow())->DelayFormat(this->BP0, this->BP1);
}

void
Interval::DelayFormatParagraphs()
{
  Interval* i = this->IncludeAllChars();
  BufferPointer* left,* right;
  i->GetLeftRight(left, right);
  left->BeginningOfBuffer();
  right->EndOfBuffer();
  i->DelayFormat();
  delete i;
}

Interval* 
Interval::IncludeAllChars()
{
  Interval* i = new Interval(this);
  BufferPointer* left,* right;
  i->GetLeftRight(left, right);
  left->BeginningOfBuffer();
  right->EndOfBuffer();
  return i;
}

// How many paragraphs or parts thereof are in the interval?
// There ALWAYS is at least one.

unsigned long
Interval::ParCount()
{
  BufferPointer* left,* right;
  this->GetLeftRight(left, right);
  Paragraph* lp = left->GetParagraph();
  Paragraph* rp = right->GetParagraph();
  return((lp == rp) ? 1 : 
	  (right->GetParagraph()->GetID() - left->GetParagraph()->GetID() + 1));
}

// are two intervals equal?
int
Interval::EQ(BaseInterval& bi)
{
  Interval& i = *(Interval*)&bi;
  BufferPointer* left1,* right1,* left2,* right2;
  this->GetLeftRight(left1, right1);
  i.GetLeftRight(left2, right2);
  return 
    (*left1 == *left2)
    &&
    (*right1 == *right2)
    ;
}

DocumentContent*
Interval::GetDocumentContent()
{
  return GetTextFlow()->GetDocContent();
}


// WriteASCIIToStream writes a single interval to an ascii stream.
void
Interval::WriteASCIIToStream(BufStream* stream)
{
  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    if (!first)
    {
      stream->NewLine();
      CHECK;
    }
    b->WriteASCIIToStream(stream,
			   (first ? left->GetByteOffset()
			    : b->GetBeginningOffset()),
			   (b == right_b ? right->GetByteOffset()
			    : b->GetEndOffset()));
    CHECK;
    if (b == right_b)
    {
      break;
    }
  }
}

void
Interval::MarkCookieConnections(PStream* stream)
{
  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    b->MarkCookieConnections(stream,
			      (first ? left->GetByteOffset()
			       : b->GetBeginningOffset()),
			      (b == right_b ? right->GetByteOffset()
			       : b->GetEndOffset()));
    if (b == right_b)
    {
      break;
    }
  }
}

void
Interval::WriteFormattedToStream(PStream* stream)
{  
  long count = ParCount();
  
  stream->StartBlockOut(PStream::HDR_PARLIST);
  CHECK;
  stream->OutULong(count);
  CHECK;

  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    b->WriteFormattedToStream(stream,
			       (first ? left->GetByteOffset()
				: b->GetBeginningOffset()),
			       (b == right_b ? right->GetByteOffset()
				: b->GetEndOffset()));
    CHECK;
    if (b == right_b)
    {
      break;
    }
  }
  
  stream->EndBlockOut();
}

// Navigate is the top level routine called from outside.
// Currently, this method is called from `DocumentEdit::Navigate' only.
void
Interval::Navigate(WORD command)
{
  WORD old_command = command;
  int horiz = 0;
  int vert = 0;
  switch (command)
  {
   case IDM_Nav_CellLeft:
    horiz = -1;
    break;
    
   case IDM_Nav_CellRight:
    horiz = 1;
    break;
    
   case IDM_Nav_CellUp:
    vert = -1;
    break;
    
   case IDM_Nav_CellDown:
    vert = 1;
    break;

   case IDM_Nav_CharLeftSelect:
    command = IDM_Nav_CharLeft;
    break;

   case IDM_Nav_CharRightSelect:
    command = IDM_Nav_CharRight;
    break;

   case IDM_Nav_LineUpSelect:
    command = IDM_Nav_LineUp;
    break;

   case IDM_Nav_LineDownSelect:
    command = IDM_Nav_LineDown;
    break;

   case IDM_Nav_WordLeftSelect:
    command = IDM_Nav_WordLeft;
    break;

   case IDM_Nav_WordRightSelect:
    command = IDM_Nav_WordRight;
    break;

   case IDM_Nav_ParStartSelect:
    command = IDM_Nav_ParStart;
    break;

   case IDM_Nav_ParEndSelect:
    command = IDM_Nav_ParEnd;
    break;
    
   case IDM_Nav_LineLeftSelect:
    command = IDM_Nav_LineLeft;
    break;

   case IDM_Nav_LineRightSelect:
    command = IDM_Nav_LineRight;
    break;

   case IDM_Nav_DocTopSelect:
    command = IDM_Nav_DocTop;
    break;
   
   case IDM_Nav_DocBottomSelect:
    command = IDM_Nav_DocBottom;
    break;
  }
  
  if (horiz | vert)
  {
    BP new_bp(this->GetBP1());
    if (new_bp.FrameMove(horiz, vert))
    {
      this->TurnOff();
      this->Set(&new_bp);
    }
  }
  else
  {
    int selectp = (command != old_command);
    
    if (this->GetTextFlow() ->tategaki_flow_p())
    {
      switch (command)
      {
       case IDM_Nav_CharLeft:
	command = IDM_Nav_LineDown;
	break;
	
       case IDM_Nav_CharRight:
	command = IDM_Nav_LineUp;
	break;
	
       case IDM_Nav_LineUp:
	command = IDM_Nav_CharLeft;
	break;
	
       case IDM_Nav_LineDown:
	command = IDM_Nav_CharRight;
	break;
	
       case IDM_Nav_WordLeft:
	command = IDM_Nav_ParEnd;
	break;
	
       case IDM_Nav_WordRight:
	command = IDM_Nav_ParStart;
	break;
	
       case IDM_Nav_ParStart:
	command = IDM_Nav_WordLeft;
	break;
	
       case IDM_Nav_ParEnd:
	command = IDM_Nav_WordRight;
	break;
      }
    }
    
    if (selectp)
    {
      // old ExtendSelection()
      // There implement shift keys used together with arrow keys to
      // extend selections.
      BP bp(this->GetBP1());
      bp.Move(command);
      this->IncDefRegion(&bp, 0);
      this->InferAttributes();
    }
    else if (!this->PointP())
    {
      switch (command)
      {
       case IDM_Nav_CharLeft:
       case IDM_Nav_LineUp:
       case IDM_Nav_WordLeft:
       case IDM_Nav_ParStart:
       case IDM_Nav_LineLeft:
       case IDM_Nav_DocTop:
	this->TurnOff();
	this->GotoBeginningPoint();
	break;
	
       case IDM_Nav_CharRight:
       case IDM_Nav_LineDown:
       case IDM_Nav_WordRight:
       case IDM_Nav_ParEnd:
       case IDM_Nav_LineRight:
       case IDM_Nav_DocBottom:
	this->TurnOff();
	this->GotoEndPoint();
	break;
      }
    }
    else // no intervals
    {
      this->GetBP0() ->Move(command);
      this->SetBP1(this->GetBP0());
    }
  }
}

// IncDefRegion -- extend an interval up to/back to a buffer pointer.
// This routine is used both with the mouse and for shift-arrow
// selections.
void
Interval::IncDefRegion(BufferPointer* bpp, bool words)
{
  if (bpp->GetTextFlow() == this->GetTextFlow())
  {
    if (words)
    {
      bool left_expand_p = (bool)((*bpp < *GetBP0()) != 0);
      bpp->ForwardWord(1);
      if (left_expand_p)
      {
	bpp->ForwardWord(-1);
      }
    }

    Interval* add_me,* subtract_me;
    Diff(bpp, add_me, subtract_me);
	
    if (add_me)
    {
      add_me->TurnOn();
      delete add_me;
    }
    if (subtract_me)
    {
      subtract_me->TurnOffMain();
      delete subtract_me;
    }
    
    SetBP1(bpp);
  }
}
		
void
Interval::InsertFromStream(PStream* stream)
{
  GetLeft()->InsertFromStream(stream);
}
	
void
Interval::InsertASCII(BufStream* stream, int break_type, bool zero_terminated)
{
  GetRight()->InsertASCII(stream, break_type, zero_terminated);
  DelayFormat();
}

void
Interval::InsertText(sjis* stuff, uword count)
{
  GetRight()->InsertText(stuff, count, atts);
  DelayFormat();
}

void
Interval::TurnOff() 
{
  TurnOnOff(IV_ALLMASK, 0);
}

void
Interval::TurnOffMain()
{
  TurnOnOff(IV_MASK, 0);
}

// "turn on" turns on the main "yellow" flag, while leaving the "blue" flag
void
Interval::TurnOn() 
{
  TurnOnOff(0, IV_MASK);
}

void
Interval::TurnOnSub() 
{
  TurnOnOff(IV_ALLMASK, IV_SUBMASK);
}

// during selections, we leave the blue flag on so we can
// restore it after the yellow selection "eats" it but then retreats.
// but this means after the selection is finalized we need to
// kill any blueness now in the yellow selection.

void
Interval::TurnOffBlueOnYellow()
{
  TurnOnOff(IV_SUBMASK, 0, False);
}

void
Interval::MakeBackSpace()
{
  if (PointP())
  {
    GetBP0()->XDecrement();
  }
  // otherwise, leave whole interval to be deleted.
}

void
Interval::MakeLine()
{
  BufferPointer* left,* right;
  this->GetLeftRight(left, right);
  
  Line* line;
  line = left->FindLine();
  if (line)
  {
    left->Set(line->GetFirstBP());
  }
  line = right->FindLine();
  if (line)
  {
    right->Set(line->GetFirstBP());
    right->Add(line->GetNumChars());
  }
  
  if (*left == *right)
  {
    if (!right->XIncrement())
    {
      right->XDecrement();
    }
  }
}

void
Interval::MakePara()
{
  BufferPointer* left,* right;
  this->GetLeftRight(left, right);
  left->BeginningOfBuffer();
  right->EndOfBuffer();
  if (*left == *right)
  {
    if (!right->XIncrement())
    {
      right->XDecrement();
    }
  }
}

void
Interval::MakeRight()
{
  if (PointP())
  {
    GetRight()->XIncrement();
  }
}

void
Interval::MakeRightWord()
{
  BPP right = GetRight();
  right->ForwardWord(+1);
  if (*right == *GetLeft())
  {
    right->ForwardWord(+1);
  }
}

void
Interval::MakeWord()
{
  BPP left = GetLeft();
  left->ForwardWord(-1);
  if (*left == *GetRight())
  {
    left->ForwardWord(-1);
  }
}

attr
Interval::InferAttributes()
{
  BufferPointer* bp = this->GetLeft();
  if (!bp)
  {
    return this->atts = 0;
  }
  return this->atts = bp->InferAttributes(!PointP());
}

void
Interval::Delete()
{
  if (!this->PointP())
  {
    BufferPointer* bp0 = this->GetBP0();
    BufferPointer* bp1 = this->GetBP1();
    Paragraph* p0 = bp0->GetParagraph();
    Paragraph* p1 = bp1->GetParagraph();
    attr atts0 = p0->GetAtts();
    attr atts1 = p1->GetAtts();
    int att_changed_p = (atts0 != atts1);
    if (att_changed_p && (bp0->BeginningOfBufferP()))
    {
      p0->SetAtts(atts1);
    }
    Buffer::BigDelete(bp0, bp1);
    Interval i(this);
    if (att_changed_p)
    {
      i.GetBP1() ->EndOfBuffer();
    }
    i.GetBP1() ->XIncrement();
    i.GetBP0() ->XDecrement();
    i.DelayFormat();
    this->GotoBeginningPoint();
  }
}

void
Interval::TurnOnOff(uword off, uword on, bool invalidate)
{
  Buffer::BigSetFlags(this->GetBP0(), this->GetBP1(), off, on);
  if (invalidate && Selected_Window != NULL)
  {
    Selected_Window->Invalidate(this->BP0, this->BP1);
  }
}

// Cut -- cut the interval, with the twist about surrounding white space
// 폜ȂTrueAłȂFalseԂB
int
Interval::Cut()
{
  // can't cut an insertion point!!
  if (PointP())
  {
    return(False);
  }
  this->Delete();
  return(True);
}

// Merge -- see if two intervals overlap.  If they do,
// change "this" if necessary to give the combined interval.

int
Interval::Merge(Interval* right)
{
  if (!this || !right || this->GetTextFlow() != right->GetTextFlow())
  {
    return 0;
  }
  
  Interval* left = this;

  // get into right order
  if (*(left->GetLeft()) > *(right->GetLeft()))
  {
    Interval* tmp = left;
    left = right;
    right = tmp;
  }
  
  BufferPointer* left_left;
  BufferPointer* right_left;
  BufferPointer* left_right;
  BufferPointer* right_right;
  left->GetLeftRight(left_left, left_right);
  right->GetLeftRight(right_left, right_right);
  
  if (*right_left < *left_right) // overlap
  {
    // this could be optimized
    SetBP0(left_left);
    SetBP1(left_right);
    GetBP1()->SetMax(right_right);
    return 1;
  }
  return 0;
}

void
Interval::Mitose()
{
  // this is not where we delete intervals prior to replacing them
  // with CR.  That happends over in doc, where we handle undo etc.
  assert(PointP());
  attr oldatt = atts;
  GetBP0()->Mitose();
  atts = oldatt;
}

int
Interval::VariableTextUpdate()
{
  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  int total_changed = 0;
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    int present;
    int changed;
    b->VariableTextUpdate((first ? left->GetByteOffset()
			    : b->GetBeginningOffset()),
			   (b == right_b ? right->GetByteOffset()
			    : b->GetEndOffset()),
			   present, changed);
    total_changed += changed;
    if (b == right_b)
    {
      break;
    }
  }
  return(total_changed);
}
  
bool
Interval::GetCommonCharAtt(attr& common_atts)
{
  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  attr attrib = 0;
  bool present = False;
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    attr cur_atts;
    if (b->GetCommonAtt(cur_atts,
			 (first ? left->GetByteOffset()
			  : b->GetBeginningOffset()),
			 (b == right_b ? right->GetByteOffset()
			  : b->GetEndOffset())))
    {
      present = True;
      if (attrib == 0)
      {
	attrib = cur_atts;
      }
      else if (attrib != cur_atts)
      {
	attrib = 0;
	break;
      }
    }
    if (b == right_b)
    {
      break;
    }
  }
  common_atts = attrib;
  return(present);
}

void
Interval::ApplyCharAtt(attr(*func) (attr, int, int),
			int attcode, int attval)
{
  this->atts = (*func) (this->atts, attcode, attval);
  if (!this->PointP())
  {
    Buffer::BigApplyCharAtt(this->GetBP0(),
			     this->GetBP1(), func, attcode, attval);
    this->DelayFormat();
  }
}

void
Interval::ApplyParAtt(attr(*func)(attr, int, int), int attcode, int attval)
{
  BufferPointer* l;
  BufferPointer* r;
  this->GetLeftRight(l, r);
  Paragraph* lp = l->GetParagraph();
  Paragraph* rp = r->GetParagraph();

  attr old = 0, neww = 0;
  
  for (Paragraph* p = lp; p ; p=p->GetNextParagraph())
  {
    attr cur = p->GetAtts();
    if (cur != old)
    {
      neww = (*func) (cur, attcode, attval);
      old = cur;
    }
    p->xSetAtts(neww);
    if (p == rp)
    {
      break;
    }
  }

  this->DelayFormatParagraphs();
}

void
Interval::ApplyParAtt(AttSpecList* attlist)
{
  BufferPointer* l;
  BufferPointer* r;
  this->GetLeftRight(l, r);
  Paragraph* lp = l->GetParagraph();
  Paragraph* rp = r->GetParagraph();

  attr old = 0, neww = 0;
  
  for (Paragraph* p = lp; p ; p=p->GetNextParagraph())
  {
    attr cur = p->GetAtts();
    if (cur != old)
    {
      neww = ::ApplyAttList(cur, attlist);
      old = cur;
    }
    p->xSetAtts(neww);
    if (p == rp)
    {
      break;
    }
  }
  this->DelayFormatParagraphs();
}

void
Interval::SetParCharAtts()
{
  BufferPointer* l;
  BufferPointer* r;
  this->GetLeftRight(l, r);

  Paragraph* lp = l->GetParagraph();
  Paragraph* rp = r->GetParagraph();

  for (Paragraph* p = lp; p ; p=p->GetNextParagraph())
  {
    p->SetParCharAtts();
    if (p == rp)
    {
      break;
    }
  }
}

IntervalList* 
Interval::FindVText()
{
  BP* left;
  BP* right;
  this->GetLeftRight(left, right);
  
  IntervalList* il = NULL;
  Buffer* right_b = right->GetBuffer();
  bool first = True;
  for (Buffer* b = left->GetBuffer(); b; b = b->GetNext(), first = False)
  {
    il = IntervalList::Concat(il,
			      b->FindVText((first ? left->GetByteOffset()
					    : b->GetBeginningOffset()),
					   (b == right_b
					    ? right->GetByteOffset()
					    : b->GetEndOffset())));
    if (b == right_b)
    {
      break;
    }
  }
  return(il);
}

char* 
Interval::GetHint()
{
  const int buflen = 32;
  char* ans = 0;
  VMem* vmem = VMem::Make(6);
  Interval i(this);
  if (i.PointP())
  {
    i.GetBP1() -> ForwardWord(1);
  }
  else if (i.GetBP0() ->GetBuffer() != i.GetBP1() ->GetBuffer())
  {
    // We don't want too large interval.
    i.GetRight() ->Set(i.GetLeft());
    i.GetBP1() ->Add(buflen);
  }
  
  BufStream stream(vmem);
  i.WriteASCIIToStream(&stream);
  if (stream.GetError() != SE_NOERROR)
  {
    stream.TermOut();
  }
  else
  {
    stream.PutByte(0);
    stream.TermOut();
    ans = new char[buflen + 1];
    ::copy_string(ans, (char*) (vmem->GetStuff()), buflen + 1);
    
    // kill CRs, other weirdities
    char ch;
    for (char* c = ans; (ch = *c) != '\0'; c = ::AnsiNext(c))
    {
      if ((unsigned char) ch < ' ')
      {
	*c = '\0';
	break;
      }
    }
  }
  delete vmem;
  return(ans);
}
