// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pref.cpp,v 3.5 1999/05/18 18:51:25 kudou Exp $
// preference

#include "pword.h"
#include "pref.h"
#include <stdarg.h>
#include <stdlib.h>
#include "ansidlg.h"
#include "boxes.h"
#include "dialogs.h"
#include "dlg.h"
#include "docwindo.h"
#include "menu.h"
#include "mwindow.h"
#include "pagesize.h"
#include "pime.h"
#include "pwordbas.h"
#include "pwordpre.h"
#include "xstr.h"

#ifdef NDEBUG
#define BEGIN_PROFILE()
#define END_PROFILE()
#define CHECK_PROFILE()
#else /* not NDEBUG */
// for debugging
int profiling_now = False;
#define BEGIN_PROFILE() \
do \
{ \
  assert(!profiling_now); \
  profiling_now = True; \
} while (0)

#define END_PROFILE() \
do \
{ \
  assert(profiling_now); \
  profiling_now = False; \
} while (0)

#define CHECK_PROFILE() \
do \
{ \
  assert(profiling_now); \
} while (0)
#endif /* not NDEBUG */

// for c++ indentation
#define LPAREN '('
#define RPAREN ')'
#define DQUOTE '"'

// check token separator
#define SPACEP(c) \
((c) == ' ' \
 || (c) == '\t' || (c) == '\f' || (c) == '\n' || (c) == '\r' || (c) == '\v')

// ------------------------------------------------------------
// simple SetPref functions

void
SetPref_int(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
}

void
SetPref_int_load_menu(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  Menu::load();
}

void
SetPref_int_control_scroll_bar(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PWordBase::control_scroll_bar();
}

void
SetPref_int_touch_caret(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  DocumentWindow::touch_caret();
}

void
SetPref_int_button_size(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  Boxes::fit_button_size();
}

void
SetPref_int_cache_buttons(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  Boxes::clear_bitmap_cache();
}

void
SetPref_ansi_var_font(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PWordPresentation::GetAnsiDialog() ->fit();
}

void
SetPref_int_show_shortcut(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  Menu::control_shortcuts();
}

void
SetPref_int_show_hidden(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PWordBase::RedisplayHidden();
}

// update Pref_keep_ime_line
void
SetPref_ime_keep_lines(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  main_window->PositionWindows();
}

void
SetPref_ime_change(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PIME::SetIgnoreFont(Pref_system_ime_font);
  DocumentWindow::control_ime();
}

static int page_setup_changed;

void
SetPref_page_setup(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  page_setup_changed = True;
}

// update integer and delay-format all documents
void
SetPref_int_delay_format(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PWordBase::DelayFormat();
}

// update integer and redisplay all documents
void
SetPref_int_redisplay(void* dest, void* src)
{
  *((int*) dest) = *((int*) src);
  PWordPresentation::redraw_all_windows();
}

// Preference type, See map_prefs().
enum Pref_Type
{
  Pref_Type_BOOLEAN,
  Pref_Type_ENUM,
  Pref_Type_INTEGER,
  Pref_Type_MM0,
  Pref_Type_STRING,
  Pref_Type_STRING10,
};

// Customizing types.

#define NO_CUSTOM 0

// Basic type.  There mean dialog control types.
// To get basic type, use `TYPE & CUSTOM_BASIC_MASK'.
#define CUSTOM_BASIC_BOOLEAN	0x01
#define CUSTOM_BASIC_ENUM	0x02
#define CUSTOM_BASIC_INTEGER	0x03
#define CUSTOM_BASIC_STRING	0x04
#define CUSTOM_BASIC_MASK	0x07
#define CUSTOM_BASIC_TYPE(type) ((type) & CUSTOM_BASIC_MASK)

#define VARIATION(basic, n) ((basic) + ((n) << 3))

// Variation types.

#define CUSTOM_BOOLEAN	VARIATION(CUSTOM_BASIC_BOOLEAN, 0)

#define CUSTOM_ENUM	VARIATION(CUSTOM_BASIC_ENUM, 0)

#define CUSTOM_MM0	VARIATION(CUSTOM_BASIC_INTEGER, 0)
#define CUSTOM_LINES	VARIATION(CUSTOM_BASIC_INTEGER, 1)
#define CUSTOM_COLUMNS	VARIATION(CUSTOM_BASIC_INTEGER, 2)
#define CUSTOM_DIVS	VARIATION(CUSTOM_BASIC_INTEGER, 3)
#define CUSTOM_DOTS	VARIATION(CUSTOM_BASIC_INTEGER, 4)
#define CUSTOM_POINTS	VARIATION(CUSTOM_BASIC_INTEGER, 5)

// Generate external variables
#define DEF(type, name, init) \
type STATIC_NEAR name = init; \
extern char STATIC_NEAR SKEY_ ## name[]; \
extern char STATIC_NEAR SDOC_ ## name[];

#define PREF_BOOLEAN(name, key, def, cust, fn, doc) \
DEF(int, name, 0)

#define PREF_ENUM(name, key, desc, cust, fn, doc) \
DEF(int, name, 0)

#define PREF_INTEGER(name, key, min, max, def, cust, fn, doc) \
DEF(int, name, def)

#define PREF_MM0(name, key, min, max, def, cust, fn, doc) \
DEF(int, name, def)

#define PREF_STRING(name, key, def, cust, fn, doc) \
DEF(char* , name, NULL)

#define PREF_STRING10(name, key, cust, fn, doc) \
String10 STATIC_NEAR name; \
extern char STATIC_NEAR SKEY_ ## name[]; \
extern char STATIC_NEAR SDOC_ ## name[];

#include "pref_m.h"
#undef DEF

#define PREF_ENUM(name, key, desc, cust, fn, doc) extern Enum_Desc desc[];
#include "pref_m.h"

static Enum_Desc boolean_desc[] =
{
  { "No",	0, },
  { "Yes",	1, },
  { "False",	0, },
  { "True",	1, },
  { "0",	0, },
  { "1",	1, },
  { "nil",	0, },
  { "t",	1, },
  { "Cancel",	0, },
  { "Ok",	1, },
  { "null",	0, },
  { "nul",	0, },
  { NULL,	0, },
};
#define SET_BOOLEAN_DESC_DEFAULT(a) \
do \
{ \
  boolean_desc[NUMBER_OF(boolean_desc) - 1].value = (a); \
} while (0)

struct Core_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  void* place;
};

#define DEF(name, key, cust, fn) cust, fn, SDOC_ ## name, SKEY_ ## name, &name

struct Boolean_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  int* place;
  int initial;
  int def;
  int custom;
};

static Boolean_Pref STATIC_NEAR boolean_pref[] =
{
#define PREF_BOOLEAN(name, key, def, cust, fn, doc) \
  { DEF(name, key, cust, fn), def, def },
#include "pref_m.h"
};

struct Enum_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  int* place;
  int initial;
  int def;
  Enum_Desc* desc;
  int custom;
};

static Enum_Pref STATIC_NEAR enum_pref[] =
{
#define PREF_ENUM(name, key, desc, cust, fn, doc) \
  { DEF(name, key, cust, fn), 0, 0, desc, },
#include "pref_m.h"
};

struct Integer_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  int* place;
  int initial;
  int def;
  int min_value;
  int max_value;
  int custom;
};

static Integer_Pref STATIC_NEAR integer_pref[] =
{
#define PREF_INTEGER(name, key, min, max, def, cust, fn, doc) \
  { DEF(name, key, cust, fn), def, def, min, max, },
#include "pref_m.h"
};

static Integer_Pref STATIC_NEAR mm0_pref[] =
{
#define PREF_MM0(name, key, min, max, def, cust, fn, doc) \
  { DEF(name, key, cust, fn), def, def, min, max, },
#include "pref_m.h"
};

struct String_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  char** place;
  char* initial;
  char* def;
  char* custom;
};

static String_Pref STATIC_NEAR string_pref[] =
{
#define PREF_STRING(name, key, def, cust, fn, doc) \
  { DEF(name, key, cust, fn), def, def, },
#include "pref_m.h"
};

struct String10_Pref
{
  int custom_type;
  void(*SetPref_fn) (void* , void*);
  char* doc;
  char* key;
  String10* place;
};

static String10_Pref STATIC_NEAR string10_pref[] =
{
#define PREF_STRING10(name, key, cust, fn, doc) \
  { DEF(name, key, cust, fn) },
#include "pref_m.h"
};

#undef DEF

// Don't change PREF_BUFSIZE to over 32KB, because Windows requests buffer
// size by "int".
#define PREF_BUFSIZE	(1024 * 31)

// Memory handle and base pointer,  there are allocated by GlobalAlloc()
HANDLE pref_buffer_handle = 0;
char* pref_buffer = NULL;

// If ALLOCATION_FAILURE is True,  do not write profile informations.
static int allocation_failure = False;

// ------------------------------------------------------------
// String10

inline int
String10::changed()
{
  return(this->changed_);
}

inline void
String10::set_changed(int changed)
{
  this->changed_ = changed;
}

// public constructor
String10::String10()
{
  this->changed_ = False;
  for (int i = 0; i != 10; ++i)
  {
    this->string[i] = NULL;
  }
}

char* 
String10::get_string(int n)
{
  SET_BOUND(n, 0, 9);
  char* s = this->string[n];
  return(s == NULL ? S_NIL : s);
}
  
void
String10::add_string(char* s)
{
  if (s != NULL)
  {
    char* t;
    int i;
    for (i = 0; i != 9; ++i)
    {
      t = this->string[i];
      if (t == NULL)
      {
	break;
      }
      if (strcmp(t, s) == 0)
      {
	if (i == 0)
	{
	  // unchanged
	  return;
	}
	break;
      }
    }
    this->changed_ = True;
    t = this->string[i];
    if (t != NULL)
    {
      xfree(t);
    }
    for (; i != 0; --i)
    {
      this->string[i] = this->string[i - 1];
    }
    this->string[0] = dup_string(s);
  }
}

static void NEAR
global_alloc()
{
//  pref_buffer_handle = GlobalAlloc((GMEM_FIXED
//				     | GMEM_NOCOMPACT | GMEM_NODISCARD),
//				    PREF_BUFSIZE);
  pref_buffer_handle = GlobalAlloc(GMEM_MOVEABLE, PREF_BUFSIZE);
}

int
Pref::begin_profile()
{
  BEGIN_PROFILE();
  global_alloc();
  if (pref_buffer_handle == 0)
  {
    GlobalCompact(PREF_BUFSIZE);
    global_alloc();
  }
  if (pref_buffer_handle != 0)
  {
    pref_buffer = (char*) GlobalLock(pref_buffer_handle);
  }
  allocation_failure = (pref_buffer == NULL);
  return(!allocation_failure);
}

void
Pref::end_profile()
{
  END_PROFILE();
  if (pref_buffer_handle != 0)
  {
    if (pref_buffer != NULL)
    {
      pref_buffer = NULL;
      GlobalUnlock(pref_buffer_handle);
    }
    GlobalFree(pref_buffer_handle);
    pref_buffer_handle = 0;
  }
}

void
map_prefs(void(*fn) (Pref_Type, Core_Pref*))
{
  {
    int i = 0;
    Boolean_Pref* pref = boolean_pref;
    for (; i != NUMBER_OF(boolean_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_BOOLEAN, (Core_Pref*) pref);
    }
  }
  {
    int i = 0;
    Enum_Pref* pref = enum_pref;
    for (; i != NUMBER_OF(enum_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_ENUM, (Core_Pref*) pref);
    }
  }
  {
    int i = 0;
    Integer_Pref* pref = integer_pref;
    for (; i != NUMBER_OF(integer_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_INTEGER, (Core_Pref*) pref);
    }
  }
  {
    int i = 0;
    Integer_Pref* pref = mm0_pref;
    for (; i != NUMBER_OF(mm0_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_MM0, (Core_Pref*) pref);
    }
  }
  {
    int i = 0;
    String_Pref* pref = string_pref;
    for (; i != NUMBER_OF(string_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_STRING, (Core_Pref*) pref);
    }
  }
  {
    int i = 0;
    String10_Pref* pref = string10_pref;
    for (; i != NUMBER_OF(string10_pref); ++i, ++pref)
    {
      (*fn) (Pref_Type_STRING10, (Core_Pref*) pref);
    }
  }
}

static Core_Pref* found_core_pref;
static Pref_Type found_pref_type;
static char* key_of_find_pref;

void
find_pref_0(Pref_Type type, Core_Pref* core)
{
  if (strcmp(core->key, key_of_find_pref) == 0)
  {
    found_core_pref = core;
    found_pref_type = type;
  }
}

static Core_Pref* NEAR
find_pref(char* key, Pref_Type* type_ret)
{
  found_core_pref = NULL;
  key_of_find_pref = key;
  map_prefs(find_pref_0);
  *type_ret = found_pref_type;
  return(found_core_pref);
}

// public interface
char* 
Pref::enum_to_string(Enum_Desc* desc, int value)
{
  char* name;
  for (;; ++desc)
  {
    name = desc->name;
    if (name == NULL || desc->value == value)
    {
      break;
    }
  }
  return(name);
}

// ------------------------------------------------------------
// reading

// public interface
char* 
Pref::read_string(char* key, char* def)
{
  CHECK_PROFILE();
  if (pref_buffer == NULL)
  {
    return(def);
  }
  ::GetPrivateProfileString(S_ini_section_name,
			     key,
			     def, pref_buffer, PREF_BUFSIZE, S_ini_file_name);
  return(pref_buffer);
}

// public interface
char* 
Pref::read_string(char* key)
{
  // 2 characters required, because "\0\0" indicates end of string list.
  static char empty_buffer[2];
  empty_buffer[0] = '\0';
  empty_buffer[1] = '\0';
  return(Pref::read_string(key, empty_buffer));
}

static Enum_Desc* NEAR
enum_entry(Enum_Desc* desc, char* s)
{
  for (;; ++desc)
  {
    char* name = desc->name;
    if (name == NULL || (s != NULL && stricmp(name, s) == 0))
    {
      return(desc);
    }
  }
}

// public interface
int
Pref::enum_value(Enum_Desc* desc, char* s)
{
  return(enum_entry(desc, s) ->value);
}

static int NEAR
enum_default_value(Enum_Desc* desc)
{
  return(Pref::enum_value(desc, NULL));
}

static int NEAR
read_enum(char* key, Enum_Desc* desc)
{
  int c;
  char* buffer = Pref::read_string(key);
  for (;;)
  {
    c = *buffer;
    if (!SPACEP(c))
    {
      break;
    }
    c = *buffer;
  }
  char* tail = buffer;
  for (;;)
  {
    if (c == '\0' || SPACEP(c))
    {
      break;
    }
    ++tail;
    c = *tail;
  }
  *tail = '\0';
  return(Pref::enum_value(desc, buffer));
}

static void
read_pref(Pref_Type type, Core_Pref* core)
{
  switch (type)
  {
   case Pref_Type_BOOLEAN:
    {
      Boolean_Pref* pref = (Boolean_Pref*) core;
      SET_BOOLEAN_DESC_DEFAULT(pref->def);
      int value = read_enum(pref->key, boolean_desc);
      pref->initial = value;
      *pref->place = value;
    }
    break;
    
   case Pref_Type_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) core;
      pref->def = enum_default_value(pref->desc);
      int value = read_enum(pref->key, pref->desc);
      pref->initial = value;
      *pref->place = value;
    }
    break;
    
   case Pref_Type_INTEGER:
   case Pref_Type_MM0:
    {
      Integer_Pref* pref = (Integer_Pref*) core;
      int value;
      if (!(type == Pref_Type_MM0
	    ? string_to_mm0(Pref::read_string(pref->key), &value)
	    : string_to_int(Pref::read_string(pref->key), &value)))
      {
	value = pref->def;
      }
      SET_BOUND(value, pref->min_value, pref->max_value);
      pref->initial = value;
      *pref->place = value;
    }
    break;
    
   case Pref_Type_STRING:
    {
      String_Pref* pref = (String_Pref*) core;
      char* s = Pref::read_string(pref->key, pref->def);
      pref->initial = dup_string(s);
      *pref->place = dup_string(s);
    }
    break;
    
   case Pref_Type_STRING10:
    {
      String10_Pref* pref = (String10_Pref*) core;
      String10* string10 = pref->place;
      char* key = pref->key;
      int key_length = strlen(key);
      char* key_buffer = (char*) xmalloc(key_length + 2);
      strcpy(key_buffer, key);
      key_buffer[key_length + 1] = '\0';
      for (int n = 9; 0 <= n; --n)
      {
	key_buffer[key_length] = '0' + n;
	string10->add_string(Pref::read_string(key_buffer));
      }
      xfree(key_buffer);
      string10->set_changed(False);
    }
    break;
  }
}

// public interface
void
Pref::read_profile()
{
  Pref::begin_profile();
  if (allocation_failure)
  {
    initialization_fail();
  }
  map_prefs(read_pref);
  PageSize::read_profile();
  Boxes::read_profile();
  Pref::end_profile();
}

// ------------------------------------------------------------
// writing

static void NEAR
write_string(char* key, char* s)
{
  // STUPID STUPID STUPID STUPID STUPID
  // On Windows 3.0, writing null string cause system error.  (konno)
  if (s[0] == '\0')
  {
    s = " ";
  }
  ::WritePrivateProfileString(S_ini_section_name, key, s, S_ini_file_name);
}

static void
write_pref(Pref_Type type, Core_Pref* core)
{
  switch (type)
  {
   case Pref_Type_BOOLEAN:
    {
      Boolean_Pref* pref = (Boolean_Pref*) core;
      // normalize Boolean
      int value =* pref->place ? 1 : 0;
      if (value != pref->initial)
      {
	write_string(pref->key, Pref::enum_to_string(boolean_desc, value));
      }
    }
    break;
    
   case Pref_Type_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) core;
      int value = *pref->place;
      if (value != pref->initial)
      {
	write_string(pref->key, Pref::enum_to_string(pref->desc, value));
      }
    }
    break;

   case Pref_Type_INTEGER:
   case Pref_Type_MM0:
    {
      Integer_Pref* pref = (Integer_Pref*) core;
      int value = *pref->place;
      if (value != pref->initial)
      {
	char* s = (type == Pref_Type_MM0 ? mm0_to_string(value)
		   : int_to_string(value));
	write_string(pref->key, s);
	delete s;
      }
    }
    break;
    
   case Pref_Type_STRING:
    {
      String_Pref* pref = (String_Pref*) core;
      char* s = *pref->place;
      if (s == NULL)
      {
	s = "";
      }
      if (strcmp(s, pref->initial) != 0)
      {
	write_string(pref->key, s);
      }
    }
    break;
    
   case Pref_Type_STRING10:
    {
      String10_Pref* pref = (String10_Pref*) core;
      String10* string10 = pref->place;
      if (string10->changed())
      {
	char* key = pref->key;
	int key_length = strlen(key);
	char* key_buffer = (char*) xmalloc(key_length + 2);
	strcpy(key_buffer, key);
	key_buffer[key_length + 1] = '\0';
	for (int n = 0; n != 10; ++n)
	{
	  key_buffer[key_length] = '0' + n;
	  write_string(key_buffer, string10->get_string(n));
	}
	xfree(key_buffer);
	string10->set_changed(False);
      }
    }
    break;
  }
}

// public interface
void
Pref::write_profile()
{
  Pref::begin_profile();
  if (!allocation_failure)
  {
    Boxes::write_profile();
    PageSize::write_profile();
    map_prefs(write_pref);
  }
  Pref::end_profile();
}

// ------------------------------------------------------------
// expr interfaces

static char* expr_ptr = NULL;

void
Pref::flush_expr(char* key)
{
  CHECK_PROFILE();
  if (expr_ptr != NULL && pref_buffer != NULL)
  {
    write_string(key, pref_buffer);
  }
  expr_ptr = NULL;
}

static void CDECL NEAR
print(char* format, ...)
{
  va_list ap;
  va_start(ap, format);
  wvsprintf(expr_ptr, format, (VA_LIST) ap);
  va_end(ap);
  char* s = expr_ptr;
  while (*s != '\0')
  {
    ++s;
  }
  expr_ptr = s;
}

void
Pref::write_expr(Pref_Expr* expr)
{
  CHECK_PROFILE();
  if (expr_ptr != NULL)
  {
    print(" %s", expr->op);
    int n = expr->num_args;
    if (n != 0)
    {
      print("(");
      Pref_Value* value = expr->args;
      for (; n != 0; --n, ++value)
      {
	switch (value->type)
	{
	 case Pref_Value_STRING:
	  {
	    char* s = value->u.s;
	    char* t = expr_ptr;
	    *t++ = DQUOTE;
	    for (;;)
	    {
	      int c = *s++;
	      if (c == '\0')
	      {
		break;
	      }
	      if (c == DQUOTE || c == '\\')
	      {
		*t++ = '\\';
	      }
	      *t++ = c;
	    }
	    *t++ = DQUOTE;
	    *t = '\0';
	    expr_ptr = t;
	  }
	  break;
	  
	 default:
	  print("%d", value->u.i);
	  break;
	}
	print(n == 1 ? ")" : ", ");
      }
    }
  }
}

void
Pref::start_write_expr()
{
  CHECK_PROFILE();
  expr_ptr = pref_buffer;
  if (expr_ptr != NULL)
  {
    *expr_ptr = '\0';
  }
}

// public interface
int
Pref::read_expr(Pref_Expr* expr)
{
  char* s = expr_ptr;
  int c = *s;
  while (SPACEP(c))
  {
    c = *++s;
  }
  if (c == '\0')
  {
    return(False);
  }
  expr->op = s;
  while (c != '\0' && c != LPAREN && !SPACEP(c))
  {
    c = *++s;
  }
  *s = '\0';
  int num_args = 0;
  while (SPACEP(c))
  {
    c = *++s;
  }
  if (c == LPAREN)
  {
    c = *++s;
    for (;;)
    {
      while (SPACEP(c))
      {
	c = *++s;
      }
      if (c == '\0' || c == RPAREN)
      {
	break;
      }
      char* top = s;
      Pref_Value value;
      if (c == DQUOTE)
      {
	value.type = Pref_Value_STRING;
	value.u.s = top;
	for (;;)
	{
	  c = *++s;
	  if (c == DQUOTE)
	  {
	    c = *++s;
	    break;
	  }
	  if (c == '\\')
	  {
	    c = *++s;
	  }
	  if (c == '\0')
	  {
	    break;
	  }
	  *top++ = c;
	}
	*top = '\0';
      }
      else
      {
	while (c != '\0' && c != RPAREN && c != ',' && !SPACEP(c))
	{
	  c = *++s;
	}
	value.type = Pref_Value_INTEGER;
	value.u.i = atoi(top);
      }
      if (num_args < PREF_MAX_EXPR_ARGS)
      {
	expr->args[num_args] = value;
	++num_args;
      }
      if (c != '\0' && c != RPAREN)
      {
	c = *++s;
      }
    }
    if (c == RPAREN)
    {
      ++s;
    }
  }
  expr_ptr = s;
  expr->num_args = num_args;
  return(True);
}

// public interface
void
Pref::set_expr(char* s)
{
  CHECK_PROFILE();
  if (pref_buffer != NULL)
  {
    strcpy(pref_buffer, s);
    expr_ptr = pref_buffer;
  }
}

// ------------------------------------------------------------
// Windows related

static LRESULT_T NEAR
send_message(HWND dialog, int id, MSG_T msg, WPARAM_T wParam, LPARAM_T lParam)
{
  return(::SendDlgItemMessage(dialog, id, msg, wParam, lParam));
}

static void NEAR
clear_combobox(HWND dialog, int id)
{
  while (0 < (int) send_message(dialog, id, CB_GETCOUNT, 0, 0))
  {
    send_message(dialog, id, CB_DELETESTRING, 0, 0);
  }
}

void
Pref::set_item_string10(HWND dialog, int item_id, String10* string10)
{
  clear_combobox(dialog, item_id);
  for (int i = 0; i != 10; ++i)
  {
    char* s = string10->get_string(i);
    if (s[0] != '\0')
    {
      send_message(dialog, item_id, CB_INSERTSTRING, -1, Ptr2Long(s));
    }
  }
}

// ------------------------------------------------------------
// customize

class CustomDialog
  : public NoKanjiModalDialog
{
private:
  int busy;
  
  // constructor, destructor
public:
  CustomDialog();
  
  // virtual method
public:
  LRESULT_T CustomDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam);
};

#undef SUPER_CLASS
#define SUPER_CLASS NoKanjiModalDialog

static int last_listbox_index = 0;
static Core_Pref* last_core_pref;
static Pref_Type last_pref_type;
static int last_custom_type;
static HWND custom_dialog;

static LONG NEAR
send_message(int id, MSG_T msg, WPARAM_T wParam, LPARAM_T lParam)
{
  return(send_message(custom_dialog, id, msg, wParam, lParam));
}

static LONG NEAR
send_message(int id, MSG_T msg, WPARAM_T wParam)
{
  return(send_message(id, msg, wParam, 0));
}

static LONG NEAR
send_message(int id, MSG_T msg)
{
  return(send_message(id, msg, 0));
}

static HWND NEAR
get_item(int id)
{
  return(::GetDlgItem(custom_dialog, id));
}

static void NEAR
show_item(int id, int flag)
{
  ::ShowWindow(get_item(id), flag);
}

static void NEAR
show_item(int id)
{
  show_item(id, SW_SHOW);
}

#define TEXT_BUFSIZE 101

static void NEAR
get_item_text(int id, char* s)
{
  ::GetDlgItemText(custom_dialog, id, s, TEXT_BUFSIZE - 1);
}

static void NEAR
set_item_text(int id, char* s)
{
  ::SetDlgItemText(custom_dialog, id, s);
}

static void NEAR
custom_dialog_check_values()
{
  char buf[TEXT_BUFSIZE];
  Core_Pref* core = last_core_pref;
  int ok = False;
  switch (last_custom_type)
  {
   case CUSTOM_BOOLEAN:
    {
      Boolean_Pref* pref = (Boolean_Pref*) core;
      pref->custom = (int) send_message(IDD_CUSTOM_YES, BM_GETCHECK) ? 1 : 0;
      ok = True;
    }
    break;

   case CUSTOM_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) core;
      get_item_text(IDD_CUSTOM_COMBOBOX, buf);
      Enum_Desc* desc = enum_entry(pref->desc, buf);
      if (desc->name != NULL)
      {
	pref->custom = desc->value;
	ok = True;
      }
    }
    break;
    
   case CUSTOM_MM0:
   case CUSTOM_LINES:
   case CUSTOM_COLUMNS:
   case CUSTOM_DIVS:
   case CUSTOM_DOTS:
   case CUSTOM_POINTS:
    {
      Integer_Pref* pref = (Integer_Pref*) core;
      int value;
      int stat = get_item_int_or_mm0(custom_dialog,
				      IDD_CUSTOM_VALUE,
				      &value, last_pref_type == Pref_Type_MM0);
      if (stat && pref->min_value <= value && value <= pref->max_value)
      {
	pref->custom = value;
	ok = True;
      }
    }
    break;
  }
  ::EnableWindow(get_item(IDOK), ok);
}

static void NEAR
custom_dialog_update_value()
{
  switch (last_custom_type)
  {
   case CUSTOM_BOOLEAN:
    {
      Boolean_Pref* pref = (Boolean_Pref*) last_core_pref;
      int yes = pref->custom;
      send_message(IDD_CUSTOM_YES, BM_SETCHECK, yes);
      send_message(IDD_CUSTOM_NO, BM_SETCHECK, !yes);
    }
    break;

   case CUSTOM_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) last_core_pref;
      char* s = Pref::enum_to_string(pref->desc, pref->custom);
      set_item_text(IDD_CUSTOM_COMBOBOX, s == NULL ? S_NIL : s);
    }
    break;

   case CUSTOM_MM0:
   case CUSTOM_LINES:
   case CUSTOM_COLUMNS:
   case CUSTOM_DIVS:
   case CUSTOM_DOTS:
   case CUSTOM_POINTS:
    {
      Integer_Pref* pref = (Integer_Pref*) last_core_pref;
      set_item_int_or_mm0(custom_dialog,
			   IDD_CUSTOM_VALUE,
			   pref->custom, last_pref_type == Pref_Type_MM0);
    }
    break;
  }
  custom_dialog_check_values();
}

static void NEAR
custom_dialog_set_value(int command)
{
  switch (CUSTOM_BASIC_TYPE(last_custom_type))
  {
   case CUSTOM_BASIC_BOOLEAN:
    {
      Boolean_Pref* pref = (Boolean_Pref*) last_core_pref;
      pref->custom = (command == IDD_CUSTOM_STANDARD ? pref->def
		      : command == IDD_CUSTOM_STARTUP ? pref->initial
		      : *pref->place);
    }
    break;

   case CUSTOM_BASIC_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) last_core_pref;
      pref->custom = (command == IDD_CUSTOM_STANDARD ? pref->def
		      : command == IDD_CUSTOM_STARTUP ? pref->initial
		      : *pref->place);
    }
    break;

   case CUSTOM_BASIC_INTEGER:
    {
      Integer_Pref* pref = (Integer_Pref*) last_core_pref;
      pref->custom = (command == IDD_CUSTOM_STANDARD ? pref->def
		      : command == IDD_CUSTOM_STARTUP ? pref->initial
		      : *pref->place);
    }
    break;
  }
  custom_dialog_update_value();
}

static void NEAR
custom_dialog_setup()
{
  char* unit = NULL;
  switch (last_custom_type)
  {
   case CUSTOM_ENUM:
    {
      Enum_Pref* pref = (Enum_Pref*) last_core_pref;
      send_message(IDD_CUSTOM_COMBOBOX, CB_RESETCONTENT);
      clear_combobox(custom_dialog, IDD_CUSTOM_COMBOBOX);
      Enum_Desc* desc = pref->desc;
      for (; desc->name != NULL; ++desc)
      {
	Enum_Desc* match = pref->desc;
	for (; match != desc && match->value != desc->value; ++match)
	{
	}
	if (match == desc)
	{
	  send_message(IDD_CUSTOM_COMBOBOX,
			CB_INSERTSTRING, -1, Ptr2Long(desc->name));
	}
      }
    }
    break;
    
   case CUSTOM_MM0:
    unit = S_mm;
    break;
    
   case CUSTOM_LINES:
    unit = S_lines;
    break;
    
   case CUSTOM_COLUMNS:
    unit = S_columns;
    break;
    
   case CUSTOM_DIVS:
    unit = S_divs;
    break;

   case CUSTOM_DOTS:
    unit = S_dots;
    break;
    
   case CUSTOM_POINTS:
    unit = S_points;
    break;
  }

  if (unit != NULL)
  {
    set_item_text(IDD_CUSTOM_UNITS, unit);
  }
}

static void NEAR
custom_dialog_new_pref()
{
  int new_index = (int) send_message(IDD_CUSTOM_LISTBOX, LB_GETCURSEL);
  if (last_listbox_index != new_index)
  {
    last_listbox_index = new_index;
    int old_custom_type = last_custom_type;
    int old_basic_type = CUSTOM_BASIC_TYPE(old_custom_type);
    if (0 <= new_index)
    {
      char buf[50];
      send_message(IDD_CUSTOM_LISTBOX, LB_GETTEXT, new_index, Ptr2Long(buf));
      Pref_Type type;
      Core_Pref* core = find_pref(buf, &type);
      if (core != NULL)
      {
	last_core_pref = core;
	last_pref_type = type;
	set_item_text(IDD_CUSTOM_DESCRIPTION, core->doc);
	int new_custom_type = core->custom_type;
	int new_basic_type = CUSTOM_BASIC_TYPE(new_custom_type);
	last_custom_type = new_custom_type;

	custom_dialog_setup();
	custom_dialog_update_value();
	
	if (old_basic_type != new_basic_type)
	{
	  for (int i = IDD_CUSTOM_INPUT_BEGIN; i <= IDD_CUSTOM_INPUT_END; ++i)
	  {
	    show_item(i, SW_HIDE);
	  }
	  switch (new_basic_type)
	  {
	   case CUSTOM_BASIC_BOOLEAN:
	    show_item(IDD_CUSTOM_YES);
	    show_item(IDD_CUSTOM_NO);
	    break;
	    
	   case CUSTOM_BASIC_ENUM:
	    show_item(IDD_CUSTOM_COMBOBOX);
	    break;
	    
	   case CUSTOM_BASIC_INTEGER:
	    show_item(IDD_CUSTOM_VALUE);
	    show_item(IDD_CUSTOM_UNITS);
	    break;
	    
	   case CUSTOM_BASIC_STRING:
	    show_item(IDD_CUSTOM_STRING);
	    break;
	  }
	}
      }
    }
  }
}

static void
register_key_to_listbox(Pref_Type type, Core_Pref* core)
{
  if (core->custom_type != NO_CUSTOM)
  {
    switch (type)
    {
     case Pref_Type_BOOLEAN:
     case Pref_Type_ENUM:
     case Pref_Type_INTEGER:
     case Pref_Type_MM0:
     case Pref_Type_STRING:
      send_message(IDD_CUSTOM_LISTBOX,
		    LB_ADDSTRING, -1, Ptr2Long(core->key));
      break;
    }
  }
}

static void NEAR
custom_dialog_init()
{
  map_prefs(register_key_to_listbox);
  send_message(IDD_CUSTOM_LISTBOX, LB_SETCURSEL, last_listbox_index);
  last_listbox_index = -1;
  last_custom_type = NO_CUSTOM;
  custom_dialog_new_pref();
}

LRESULT_T
CustomDialog::WndProc(MSG_T iMessage, WPARAM_T wParam, LPARAM_T lParam)
{
  custom_dialog = this->GetHandle();
  switch (iMessage)
  {
   case WM_INITDIALOG:
    this->busy = True;
    custom_dialog_init();
    this->busy = False;
    return(1);
    
   case WM_COMMAND:
    if (!this->busy)
    {
      this->busy = True;
#ifdef _WIN32
      switch (LOWORD(wParam))
#else /* _WIN32 */
      switch (wParam)
#endif /* _WIN32 */
      {
       default:
	custom_dialog_check_values();
	break;

       case IDD_CUSTOM_STANDARD:
       case IDD_CUSTOM_STARTUP:
       case IDD_CUSTOM_CURRENT:
#ifdef _WIN32
	custom_dialog_set_value(LOWORD(wParam));
#else /* _WIN32 */
	custom_dialog_set_value(wParam);
#endif /* _WIN32 */
	break;
	
       case IDD_CUSTOM_LISTBOX:
#ifdef _WIN32
	switch (HIWORD(wParam))
#else /* _WIN32 */
	switch (HIWORD(lParam))
#endif /* _WIN32 */
	{
	 case LBN_SELCHANGE:
	  custom_dialog_new_pref();
	  break;
	}
	break;
      }
      this->busy = False;
    }
    break;
  }
  return(SUPER_CLASS::WndProc(iMessage, wParam, lParam));
}

// public constructor
CustomDialog::CustomDialog()
{
  this->class_name = "CustomDialog";
  this->busy = False;
}

static void
custom_set(Pref_Type type, Core_Pref* core)
{
  void(*fn) (void* , void*) = core->SetPref_fn;
  if (fn != NULL && core->custom_type != NO_CUSTOM)
  {
    void* custom = NULL;
    switch (type)
    {
     case Pref_Type_BOOLEAN:
      {
	Boolean_Pref* pref = (Boolean_Pref*) core;
	if (pref->custom == *pref->place)
	{
	  return;
	}
	custom = (void*) &pref->custom;
      }
      break;
      
     case Pref_Type_ENUM:
      {
	Enum_Pref* pref = (Enum_Pref*) core;
	if (pref->custom == *pref->place)
	{
	  return;
	}
	custom = (void*) &pref->custom;
      }
      break;
      
     case Pref_Type_INTEGER:
     case Pref_Type_MM0:
      {
	Integer_Pref* pref = (Integer_Pref*) core;
	if (pref->custom == *pref->place)
	{
	  return;
	}
	custom = (void*) &pref->custom;
      }
      break;
      
     case Pref_Type_STRING:
      {
	String_Pref* pref = (String_Pref*) core;
	if (strcmp(pref->custom, *pref->place) == 0)
	{
	  xfree(pref->custom);
	  return;
	}
	custom = (void*) &pref->custom;
      }
      break;
    }
    if (custom != NULL)
    {
      (*core->SetPref_fn) (core->place, custom);
    }
  }
}

static void
custom_cancel(Pref_Type type, Core_Pref* core)
{
  if (core->custom_type != NO_CUSTOM)
  {
    switch (type)
    {
     case Pref_Type_STRING:
      xfree(((String_Pref*) core)->custom);
      break;
    }
  }
}

static void
custom_init(Pref_Type type, Core_Pref* core)
{
  if (core->custom_type != NO_CUSTOM)
  {
    switch (type)
    {
     case Pref_Type_BOOLEAN:
      ((Boolean_Pref*) core)->custom = *((Boolean_Pref*) core)->place;
      break;
      
     case Pref_Type_ENUM:
      ((Enum_Pref*) core)->custom = *((Enum_Pref*) core)->place;
      break;
      
     case Pref_Type_INTEGER:
     case Pref_Type_MM0:
      ((Integer_Pref*) core)->custom = *((Integer_Pref*) core)->place;
      break;
      
     case Pref_Type_STRING:
      ((String_Pref*) core)->custom
      = dup_string(*((String_Pref*) core)->place);
      break;
    }
  }
}

// public interface
void
Pref::custom()
{
  map_prefs(custom_init);
  int status;
  {
    CustomDialog dialog;
    status = dialog.Go();
  }
  page_setup_changed = False;
  map_prefs(status ? custom_set : custom_cancel);
  if (page_setup_changed)
  {
    PageSize::read_profile();
  }
}
