// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pstreams.h,v 3.4 1999/05/12 00:22:16 kudou Exp $
// define PStream class
// RawStream is a pure virtual class to serve as the superclass for raw
// memory and file streams(MemoryStream and FileStream).
//
// BufStream contains a RawStream and supports buffered reading and
// writing, including the two fast GetByte and PutByte routines.
//  
// BlockStream adds the concept of hierarchical blocks and self-sizing
// numbers.  A file written using BlockStream can be deciphered
// independent of its semantics, since its blocks and values are self
// identifying.
// 
// PStream is basically just a BlockStream, with an extra field for
// BeatWord-specific information about what to write and read.

#ifndef PSTREAMS_H
#define PSTREAMS_H

#ifndef NDEBUG
extern char* stream_errmsg[];
#endif

#define hdr(a,b) (((a) << 8) | (b))

enum StreamError
{
  SE_NOERROR = 0,
  SE_BADHEADER = 1,
  SE_DISKWRITEERROR = 2,
  SE_DISKREADERROR = 3,
  SE_DISKSEEKERROR = 4,
  SE_BADVERSION = 5,
  SE_TEXTFORMAT = 6,
  SE_DISKOPENERROR = 7,
  SE_MEMORYERROR = 8,
  SE_NOTENOUGHERROR = 9,
  SE_STACKOVERFLOW = 10,
  SE_STACKUNDERFLOW = 11,
  SE_NOMORE = 12,
  SE_BROKENFLAGS = 13,
  SE_NOTBLOCK = 14,
  SE_UNEXPECTEDENDBLOCK = 15,
  SE_UNEXPECTEDSTARTBLOCK = 15,
  SE_TOOMUCH = 16,
  SE_FILEENDERROR = 17,
  SE_BADVAL = 18,
  SE_UNEXPECTEDBINARY = 19,
  SE_OVERFLOW = 20,
  SE_TOOBIG = 21,
};

// ------------------------------------------------------------
// RawStream class

// RawStream is the pure virtual superclass for raw memory and file
// streams.  It is owned by BufStream.

class RawStream
{
protected:
  StreamError error;
  udword pos;
  
private:
  // Following slots are used for maximum management.
  udword mx;
  bool(*max_func) ();
  int max_set;
  int max_reached;
  
protected:
  RawStream();
public:
  virtual ~RawStream();
  
public:
  virtual udword Get(void* buf, udword count) = 0;
  virtual void Put(void* buf, udword count) = 0;
  virtual void SkipIn(udword count) = 0;
  
public:
  StreamError GetError();
  void CheckMax(udword count);
  void SetError(StreamError e);
  void SetMax(udword mx, bool(*) ());
};

inline StreamError
RawStream::GetError()
{
  return(this->error);
}

inline void
RawStream::SetError(StreamError e)
{
//#ifdef BW3_DEBUG
//  if (e != SE_NOERROR)
//  {
//    assert(False);
//  }
//#endif /* BW3_DEBUG */
  this->error = e;
}

// ------------------------------------------------------------
// MemoryStream class

class MemoryStream
  : public RawStream
{
private:
  VMem*& area;

public:
  MemoryStream(VMem*& area);
  virtual ~MemoryStream();

public:
  virtual udword Get(void* buf, udword count);
  virtual void Put(void* buf, udword count);
  virtual void SkipIn(udword count);
};

// ------------------------------------------------------------
// FileStream

class FileStream
  : public RawStream
{
private:
  FILE* file;

public:
  FileStream(FILE*);
  virtual ~FileStream();

  virtual udword Get(void* buf, udword count);
  virtual void Put(void* buf, udword count);
  virtual void SkipIn(udword count);
};

// ------------------------------------------------------------
// BufStream class

class BufStream 
{
  enum
  {
    BUFSIZE_LOG2 = LOG2_OF_BEST_BUFSIZE,
    BUFSIZE = 1 << (BUFSIZE_LOG2-1),
  };

private:
  ubyte* buf;
  int ipos;
  ubyte* cpos;
  int imax;
  bool last;		// recent "fill" returned "last"
  StreamError error;
  bool own_raw;
  
protected:
  RawStream* raw;
  bool termed;

public:
  BufStream(FILE* f);
  BufStream(RawStream*);
  BufStream(VMem*& vmem);
  ~BufStream();

private:
  void NEAR Fill();
  void NEAR Flush();
public:
  StreamError GetError();
  ubyte GetByte();
  udword GetHuge(void* buf, udword count);
  udword GetLong();
  uword Get(void* buf, uword count);
  uword GetWord();
  void Init();
  void NewLine();
  void Put(void* buf, uword count);
  void PutByte(ubyte);
  void PutHuge(void* buf, udword count);
  void PutLong(udword);
  void PutWord(uword);
  void SetError(StreamError e);
  void SetMax(udword mx, bool(*fnc) ());
  void SkipIn(udword count);
  void TermIn();
  void TermOut();
};
  
inline StreamError
BufStream::GetError()
{
  return(this->error);
}

inline void
BufStream::SetError(StreamError e)
{
//#ifdef BW3_DEBUG
//  if (!((e == SE_NOERROR) || (e == SE_NOMORE)))
//  {
//    assert(False);
//  }
//#endif /* BW3_DEBUG */
  this->error = e;
}

// ------------------------------------------------------------
// BlockStream

class BlockStream
  : public BufStream
{
  enum BHFlags
  {
    // top three bits give type
    BHFType = 0xe0,
    
    BHFReservedType0 = 0x00,
    BHFHeader = 0x20,
    BHFVal = 0x40,
    BHFImm = 0x60,
    BHFReservedType1 = 0x80,
    BHFReservedType2 = 0xa0,
    BHFReservedType3 = 0xc0,
    BHFReservedType4 = 0xe0,
    
    // low-order bit in upper nibble advised that binary data follows
    BHFBinary = 0x10,
    
    // if header
    BHFEndBlock = 0x01,
    BHFVers = 0x02,
    BHFName = 0x04,
    
    // if val
    // two bits give length
    BHFLen = 0x03,
    BHFLenReserved = 0x00,
    BHFLenByte = 0x01,
    BHFLenWord = 0x02,
    BHFLenLong = 0x03,
    
    // if immediate
    BHFImmVal = 0x0f,
  };
  
  enum
  {
    MAXLEV = 256,
    BUFSIZE_LOG2 = LOG2_OF_BEST_BUFSIZE,
    BUFSIZE = 1 << (BUFSIZE_LOG2-1),
    
    HDR_PSTREAM = hdr('p','s'), // pstream version
  };
  
private:
  static const int PSTREAM_VERSION;
  
  int hit_endblock;
  
  // remember state of StartTypedBlockIn
  word wanted;
  word got;

#ifndef NDEBUG
  // stack management for debugging
  word* types;
  int curlev;
#endif

public:
  BlockStream(FILE* f);
  BlockStream(RawStream* raw);
  BlockStream(VMem*& vmem);
  ~BlockStream();

private:
  dword NEAR xValIn(byte flags, int& binary);
  udword NEAR xUValIn(byte flags, int& binary);
  void NEAR DownLevel();
  void NEAR GotoEndBlock();
  void NEAR Init();
  void NEAR UpLevel(word type);

public:
  char* InString();
  byte InByte();
  ubyte InUByte();
  word InWord();
  uword InUWord(int&);
  uword InUWord();
  dword InLong();
  udword InULong();
  udword InUDWord() { return InULong(); }
  udword UValIn(int& binary);
  dword ValIn(int& binary);
  void EndBlockIn();
  void EndBlockOut();
  void In(void* buf, uword len);
  void InHuge(void* buf, udword len);
  void InitIn();
  void InitOut();
  void Out(void* buf, uword len);
  void OutByte(byte);
  void OutHuge(void* buf, udword len);
  void OutLong(dword);
  void OutString(char*);
  void OutUByte(ubyte);
  void OutUDWord(udword d) { OutULong(d); }
  void OutULong(udword);
  void OutUWord(uword);
  void OutWord(word);
  void StartBlockOut();
  void xStartBlockIn(byte flags, word& type, word& vers);
  void StartBlockIn(word& type, word& vers);
  void StartBlockOut(word type);
  void StartBlockOut(word type, word vers);
  void StartTypedBlockIn(word type, word& vers);
  void TermIn();
  void TermOut();
  void UValOut(udword);
  void UValOut(udword, int binary);
  void ValOut(dword);
  void ValOut(dword, int binary);
#ifndef NDEBUG
public:
  char* MakeErrorString();
  void Dump(FILE* f);
  void Dump1(FILE* f, int lev);
#endif
};

inline void 
BlockStream::OutUByte(ubyte b)
{
  UValOut(b);
}

inline void 
BlockStream::OutByte(byte b)
{
  ValOut(b);
}

inline void 
BlockStream::OutWord(word w)
{
  ValOut(w);
}

inline void 
BlockStream::OutUWord(uword w)
{
  UValOut(w);
}

inline void 
BlockStream::OutLong(dword l)
{
  ValOut(l);
}

inline void 
BlockStream::OutULong(udword l)
{
  UValOut(l);
}

// private lexical method
inline void NEAR
BlockStream::DownLevel()
{
#ifndef NDEBUG
  if (this->curlev <= 0) 
  {
    this->SetError(SE_STACKUNDERFLOW);
    return;
  }
  --curlev;
#endif
  this->hit_endblock = 0;
}

inline void NEAR
BlockStream::UpLevel(word type)
{
#ifndef NDEBUG
  if (this->curlev >= MAXLEV)
  {
    this->SetError(SE_STACKOVERFLOW);
    return;
  }
  this->types[curlev++] = type;
#endif
}

// ------------------------------------------------------------
// PStream

class PStream
  : public BlockStream
{
public:
  enum WriteWhat
  {
    WW_NONE             = 0,
    WW_TEXT 		= 1<<0,    // write out text
    WW_CHARATTS 	= 1<<1,    // write out attribute values
    WW_ATTCONTENT 	= 1<<2,    // implies attributes have to be mapped
    
    WW_PARATTS 		= 1<<3,    // write out attribute values
    
    WW_CLIP             = 1<<4,    // going to clipboard, special EOL handling
    WW_UNDO             = 1<<5,    // undo
    
    // combinations for various purposes
    WW_ATTS = WW_CHARATTS | WW_PARATTS,
    WW_UNDOTEXT = WW_TEXT | WW_ATTS | WW_UNDO,
    WW_CLIPBOARD = WW_TEXT | WW_ATTS | WW_ATTCONTENT | WW_CLIP,
    WW_FILE = WW_TEXT | WW_ATTS | WW_ATTCONTENT,
    WW_MARK = WW_ATTCONTENT,
  };
  
  enum HeaderType
  {
    // pstreams themselves
    HDR_PWORD	    =hdr('p','w'),	// file header
    
    HDR_DOCCONT     =hdr('d','c'),	// doc content, inc. textflows
    HDR_MEMO        =hdr('m','e'),	// document memo
    
    // presentation related
    HDR_DOCPRES     =hdr('d','p'),	// doc presentation
    HDR_DOCWIN	    =hdr('d','w'),	// document window 
    HDR_DOCEDIT     =hdr('d','e'),	// document editor, selections
    HDR_DOCSEL      =hdr('d','s'),	// document selection
    
    // attribute related
    HDR_ATTTABLES   =hdr('a','s'),      // all tables
    
    HDR_ATTTABLE    =hdr('a','t'),	// attribute table
    HDR_ATTSENTRY   =hdr('a','e'),	// attribute entry
    
    HDR_ADTABLE     =hdr('a','d'),	// attr deriv table
    HDR_DERIVENTRY  =hdr('d','e'),	// attr deriv entry
    
    HDR_STABLE      =hdr('s','t'),	// style table
    HDR_STYLEENTRY  =hdr('s','e'),	// style entry
    
    HDR_FTABLE      =hdr('f','t'),	// font table
    HDR_FONTENTRY   =hdr('f','e'),	// font entry
    
    HDR_MTABLE      =hdr('m','t'),	// mark table
    HDR_MARKENTRY   =hdr('m','e'),	// mark entry
    
    HDR_BLTABLE	    =hdr('b','l'),	// borderline table
    HDR_BORDERENTRY =hdr('b','e'),	// borderline entry
    
#ifdef BW2_ABBREV
    HDR_ABBREVTABLE =hdr('v','t'),      // abbreviation table
    HDR_ABBREVENTRY =hdr('v','e'),      // abbreviation entry
#endif /* BW2_ABBREV */
    
    HDR_TEXTFLOWS   =hdr('t','s'),	// several textflows
    HDR_TEXTFLOW    =hdr('t','f'),	// one text flow
    HDR_PARAGRAPH   =hdr('p','a'),	// paragraph
    HDR_BUFFER      =hdr('b','u'),	// buffer
    
    HDR_FAT	    =hdr('f','a'),	// 2-byte char list
    HDR_ATTLIST     =hdr('a','l'),	// list of atts on text
    HDR_PARLIST     =hdr('p','l'),	// bunch of paragraphs
    HDR_INTLIST     =hdr('i','l'),      // interval list
    
    // page info
    HDR_PAGEINFO    =hdr('p','i'),	// document page
    HDR_LAYOUT      =hdr('l','y'),	// layout template
    HDR_PAGEMAP     =hdr('p','m'),	// pagemap
    HDR_PAGESIZE    =hdr('p','s'),	// page size
    HDR_BLUEPRINT   =hdr('b','p'),	// frame blueprint
    HDR_SEED        =hdr('s','d'),	// seed data
    HDR_STAGE       =hdr('s','g'),	// stage data
    
    // panel
    HDR_PANEL       =hdr('p','n'),	// panel infomation
    HDR_TABLEPANEL  =hdr('t','p'),	// table panel
    HDR_VTEXTPANEL  =hdr('v','p'),	// variable text panel
    HDR_VTEXT       =hdr('v','t'),	// variable text content
  };
  WriteWhat what;
  
  PStream(FILE* f, WriteWhat);
  PStream(VMem*& , WriteWhat);
  ~PStream();
  
  WriteWhat GetWhat() { return what; }
  void SetWhat(WriteWhat new_what)
  {
    what = new_what;
  }
};

#define CHECK if (stream->GetError() != SE_NOERROR) return

#endif /* PSTREAMS_H */
