// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pundo.h,v 3.3 1999/05/12 00:22:16 kudou Exp $
// define PUndo class to handle undoing things
//
// The basic plan is to have a list of undo items of type PUndoAtom.  Each
// contains a linked list of undo actions in the atom, of type PUndoItem.
// 
// The main methods on PUndoitem are Undo() and Redo().
// 
// The list of undo atoms is managed by a class called *PUndo*.  This is
// pointed to by DocumentContent.  In other words, undo information is
// maintained separately for each document.
// 
// The main methods on PUndo are:
// Undo() and Redo() to undo and redo a single action
// UndoRun() and RedoRun() to undo a "run"
// StartAtom() and EndAtom(), to mark the beginning of an atomic undaoble
// action.
// Register(), to register a particular function within an atom.
// management routines such as Clear() which should be called when the
// user has done something which will invalidate all undo info
// 
// A run is determined by the whether or not a "move" occurred between
// actions, and by whether or not the actions are the same type.  In
// particular, sequences of insertions and deletions form runs.  Character
// and paragraph formatting form separate runs.  Even in the middle of
// character formatting, a run stops when the user changes the selection.

#ifndef PUNDO_H
#define PUNDO_H

#include "smartnew.h"
#include "vmem.h"

class PUndoAtom;

// make something PUndo owns, but cannot access directly

class PUndoPrivate
{
protected:
  enum
  {
    MAXUNDO = 100
  };
  
private:
  PUndoAtom* undobuf[ MAXUNDO ];
  
protected:
  PUndoAtom* & GetAtom(int i)
  {
    return undobuf[i%MAXUNDO];
  }
  PUndoPrivate();
};


class PUndo : private PUndoPrivate
{
public:
  
  // a description is provided to StartAtom().  It has NO
  // semantics associated with it.  It simply controls what the
  // user will be told about what can be/was undone.
  // Note: these values are closely tied to strings defined in
  // pword.rc.  THINK before you rearrange them.
  enum UndoDesc
  {
#define DEFUNDO(desc, j, e) desc,
#include "undo_m.h"
    UD_UNUSED_MAX,
  };
  
  // a function is provided to Register.  It indicates the type
  // of undo processing involved.  If you add an item here, you MUST
  // add an entry to func_data!
  enum UndoFunc
  {
    UF_Insert,
    UF_Delete,
    UF_ParAtt,
    UF_CharAtt,
    UF_StyleDefine,
    UF_StyleUndefine,
    UF_StyleRedefine,
  };
  
  static int func_data[];
  
private:
  
  int cur_item;		// action which the user has just undone.
  // If == next_item,
  // the user has not started undoing/redoing.
  // if == first_item, the user has undone
  // as far as he can go.
  
  int next_item;        // place where NEXT action will go
  // beginning sequence of undos
  
  int first_item;	// action done least recently
  
  DocumentContent* content;   // who we belong to
  
  bool moved;            // used to mark runs
  bool dead_atom;
  
public:
  static bool big_undo_msg_issued;
  static bool big_undo_result;
  static bool ConfirmBigUndo();
  
private:
  // add an atom to the stack
  void NEAR Add(PUndoAtom* atom);
  
  // do two items form a run?
  bool NEAR Run(int i1, int i2);
  
  // find next item which can be undone or redone
  bool NEAR NextUndoableAtom(int& i);
  bool NEAR NextRedoableAtom(int& i);
  
  // internal routines to undo and redo either single or run
  bool NEAR xRedo(int single);
  bool NEAR xUndo(int single);
  long NEAR PUndo::menu_state(int undop);
public:
  PUndo(DocumentContent*);
  ~PUndo();
  
  long PUndo::undo(int menu_state_p, int singlep);
  long PUndo::redo(int menu_state_p, int singlep);
  
  // kill all undo info
  void Clear();
  
  // throw away redoable items
  void Resync();
  
  // start and end atomic groupings of actions
  void StartAtom(IntervalList* before, UndoDesc desc);
  void EndAtom(IntervalList* after);
  
  // register a particular action
  void Register(Interval* target, UndoFunc func, int misc=0);
  
  // moved management
  void SetMoved();
  bool GetMovedAndClear();
  
  void ReportDeadTextFlow(TextFlow*);
  
  DocumentContent* GetDocumentContent()
  {
    return content;
  }
};


// ------------------------------------------------------------
// PUndoAtom holds information on groups of actions.

class PUndoAtom
{
private:
  PUndo* undo;            // to which i belong
  PUndoItem* head;            // start of list of items
  PUndoItem* tail;            // and its end
  BoundIntervalList* before;  // interval list before this atom
  BoundIntervalList* after;   // and after it
  int count;
  // saving memory
  PUndo::UndoDesc desc : 8;   // description of atom
  int moved : 8;	      // first action since user moved?
  
  void Restore(BoundIntervalList*);
  
public:
  PUndoAtom(PUndo* undo, IntervalList* before, PUndo::UndoDesc desc);
  ~PUndoAtom();
  REFER_SMARTNEW(PUndoAtom)
  
  // add an item to the atom
  bool Add(Interval* target, PUndo::UndoFunc func, int misc);
  
  // undo or redo the atom
  bool Undo();
  bool Redo();
  
  // tell the user what happened
  void NEAR Announce(char* fmt);
  
  // restore the interval list as of before or after the atom
  void RestoreBefore();
  void RestoreAfter();
  
  // miscellaneous
  bool ReportDeadTextFlow(TextFlow*);
  char*  NEAR PUndoAtom::get_string();
  void SetAfter(IntervalList* il);
  bool BigP();
  
  // inline routines
  bool GetMoved()
  {
    return moved != 0;
  }
  PUndo::UndoDesc GetUndoDesc()
  {
    return desc;
  }
  PUndo* GetUndo()
  {
    return undo;
  }
  int GetCount()
  {
    return count;
  }
};

// ------------------------------------------------------------
// PUndoItem holds information  on individual actions.

class PUndoItem 
{
  friend class PUndo;
  friend class PUndoAtom;
  
private: // members
  PUndoAtom* atom;		// to which I belong
  VMem* stuff;                  // place to store what might need to be undone
  PUndoItem* next;              // next item in this atom
  PUndoItem* prev;              // prev item in this atom
  BoundInterval* target;	// selection as of time when action was taken
  PUndo::UndoFunc func;         // the function involved
  int misc;                     // addt'l info, depends on func
  
  
private: // functions
  VMem* Gather(bool& success);
  bool InitialGather();
  void PutBack();
  
  int GetWhat();
  
public:
  PUndoItem(PUndoAtom* atom, Interval* i, PUndo::UndoFunc func, int stuff);
  ~PUndoItem();
  REFER_SMARTNEW(PUndoItem)
  bool Undo();
  bool Redo();
  
  // inline functions
  VMem* GetStuff()
  {
    return this->stuff;
  }
  PUndo::UndoFunc GetUndoFunc() 
  { 
    return this->func; 
  }
  BoundInterval* GetTarget()
  {
    return this->target;
  }
};

#endif /* PUNDO_H */
