// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: seeds.cpp,v 3.5 1999/06/23 06:43:49 kudou Exp $
// Shrinked FrameObject for file save/load.
// These classes are used in BluePrint. 

#include "pword.h"
#include "frameown.h"
#include "fission.h"
#include "frametem.h"
#include "fillobjc.h"
#include "grphobjc.h"
#include "picture.h"
#include "objectid.h"
#include "metafile.h"
#include "mfobject.h"
#include "pstreams.h"
#include "vmem.h"
#include "seeds.h"
#include "blueprin.h"


// ------------------------------------------------------------
// class Seed

// constructor
Seed::Seed(FrameObject* fo)
  : margin(fo->GetMargin())
{
  id = (ObjectID)fo->GetObjectID();
}

// constructor for make template
Seed::Seed(ObjectID id, Margin& m)
  : margin(m)
{
  Seed::id = id;
}

// generate seed
Seed* 
Seed::GenerateSeed(FrameObject* fo)
{
  switch (fo->GetObjectID()) 
  {
   case IDTextObject:
    return new TextSeed(fo);
   case IDFillObject:
    return new FillSeed(fo);
   case IDGraphicObject:
    return new GraphicSeed(fo);
   case IDAllLayoutObject:
    return new AllLayoutSeed(fo);
   case IDMetafileObject:
    return new MetafileSeed(fo);
   case IDUnknownObject:
   default:
    return new UnknownSeed(fo);
  }
}

// seed data save
void 
Seed::Save(PStream* stream)
{
  // header
  stream->StartBlockOut(PStream::HDR_SEED, save_version);
  CHECK;
  
  // save object ID
  stream->OutUWord(id);
  CHECK;
  
  // save margin
  stream->OutWord(margin.GetTop());
  CHECK;
  stream->OutWord(margin.GetBottom());
  CHECK;
  stream->OutWord(margin.GetLeft());
  CHECK;
  stream->OutWord(margin.GetRight());
  CHECK;
  
  switch (id) 
  {
   case IDTextObject:
    ((TextSeed*)this)->TextSeed::Save(stream);
    CHECK;
    break;
   case IDFillObject:
    ((FillSeed*)this)->FillSeed::Save(stream);
    CHECK;
    break;
   case IDGraphicObject:
    ((GraphicSeed*)this)->GraphicSeed::Save(stream);
    CHECK;
    break;
   case IDAllLayoutObject:
    ((AllLayoutSeed*)this)->AllLayoutSeed::Save(stream);
    CHECK;
    break;
   case IDMetafileObject:
    ((MetafileSeed*)this)->MetafileSeed::Save(stream);
    CHECK;
    break;
   case IDUnknownObject:
   default:
    ((UnknownSeed*)this)->UnknownSeed::Save(stream);
    CHECK;
    break;
  }
  
  stream->EndBlockOut();
}

Seed* 
Seed::Load(PStream* stream, int bp_version)
{
  word seed_version = Seed::sv_luna;
  if (bp_version >= BluePrint::sv_snow) 
  {
    stream->StartTypedBlockIn(PStream::HDR_SEED, seed_version);
    CHECK 0;
  }
  
  ObjectID object_id;
  Margin m;
  if (seed_version >= Seed::sv_bw1) 
  {
    // load object ID
    object_id = (ObjectID)stream->InUWord();
    CHECK 0;
    
    // load margin
    m.SetTop(stream->InWord());
    CHECK 0;
    m.SetBottom(stream->InWord());
    CHECK 0;
    m.SetLeft(stream->InWord());
    CHECK 0;
    m.SetRight(stream->InWord());
    CHECK 0;
  }
  else 
  {
    class SV2_SeedHeader
    {
    public:
#ifdef _WIN32
      word id;
      word size;
#else /* _WIN32 */
      ObjectID id;
      size_t size;
#endif /* _WIN32 */
    } sv2_sh;
    
    int size = sizeof(SV2_SeedHeader);
    stream->In(&sv2_sh, size);
    CHECK 0;

#ifdef _WIN32
    object_id = (enum ObjectID)(int)sv2_sh.id;
#else /* _WIN32 */
    object_id = sv2_sh.id;
#endif /* _WIN32 */    
  }
  
  Seed* s = 0;
  switch (object_id) 
  {
   case IDTextObject:
    s = TextSeed::Load(stream, m, seed_version);
    break;
   case IDFillObject:
    s = FillSeed::Load(stream, m, seed_version);
    break;
   case IDGraphicObject:
    s = GraphicSeed::Load(stream, m, seed_version);
    break;
   case IDAllLayoutObject:
    s = AllLayoutSeed::Load(stream, m, seed_version);
    break;
   case IDMetafileObject:
    s = MetafileSeed::Load(stream, m, seed_version);
    break;
//   case IDUnknownObject:
//      s = UnknownSeed::Load(stream, m, seed_version);
//      break;
   default:
    s = new UnknownSeed(m);
    break;
  }
  if (!s) 
  {
    // less memory 
    s = new UnknownSeed(m, True);
  }
  
  if (bp_version >= BluePrint::sv_snow) 
  {
    stream->EndBlockIn();
  }
  
  CHECK 0;
  return s;	// success
}

// ------------------------------------------------------------
// class TextSeed

// constructor
TextSeed::TextSeed(FrameObject* fo)
  : Seed(fo)
{
  FrameTemplate* ft = (FrameTemplate*)fo;
  frame_align = ft->frame_align();
}

TextSeed::TextSeed(Margin& m)
  : Seed(IDTextObject, m)
{
  frame_align = AL_vertical_adjust;
}

void 
TextSeed::Save(PStream* stream)
{
  stream->OutWord((word)frame_align);
  CHECK;
}

Seed* 
TextSeed::Load(PStream* stream, Margin& m, int version)
{
  TextSeed* ts = new TextSeed(m);
  
  switch (version)
  {
   case sv_luna:
   case sv_snow:
    {
      ts->frame_align = AL_vertical_zero_reserved;
      int size = sizeof(OLDTextSeed);
      stream->In(ts, size);
      CHECK 0;
    }
    break;
    
   case sv_bw1:
    ts->frame_align = AL_vertical_zero_reserved;
    break;
    
   case sv_bw2:
   case sv_bw3:
    {
      ts->frame_align = (VerticalAlignment)stream->InWord();
      CHECK 0;
    }
    break;
  }
  
  return ts;
}

// ------------------------------------------------------------
// class FillSeed

// constructor
FillSeed::FillSeed(FrameObject* fo)
  : Seed(fo)
{
  FillObject* fi = (FillObject*)fo;
  typ = fi->GetFillType();
}

// constructor for make template
FillSeed::FillSeed(FillObject::FillType t)
  : Seed(IDFillObject, Margin(0, 0, 0, 0))
{
  typ = t;
}

// save seed data
void 
FillSeed::Save(PStream* stream)
{
  stream->OutUWord(typ);
  CHECK;
}

// load seed data
Seed* 
FillSeed::Load(PStream* stream, Margin& m, int version)
{
  FillSeed* fs = new FillSeed(m);
  if (version >= Seed::sv_bw1) 
  {
    fs->typ = (FillObject::FillType)stream->InUWord();
    CHECK 0;
  }
  else 
  {
    int size = sizeof(FillSeed);
    stream->In(fs, size);
    CHECK 0;
  }
  return fs;
}

// ------------------------------------------------------------
// class GraphicSeed

// constructor
GraphicSeed::GraphicSeed(FrameObject* fo)
  : Seed(fo)
{
  GraphicObject* go = (GraphicObject*)fo;
  Picture* pic = go->GetPicture();
  vm_dib = pic->GetVMDIB();
  dib_size = pic->GetSize();
  ref_flag = True;
}

// constructor for file read
GraphicSeed::GraphicSeed(Margin& m)
  : Seed(IDGraphicObject, m) 
{
  vm_dib = 0;
  dib_size = 0; 
  ref_flag = False;
}

// desturctor
GraphicSeed::~GraphicSeed()
{
  if (!ref_flag) 
  {
    delete vm_dib;
  }
}

// save seed data
void 
GraphicSeed::Save(PStream* stream)
{
  stream->OutUDWord(dib_size);
  CHECK;
  
  stream->OutHuge((void*)*vm_dib, dib_size);
  CHECK;
}

class SV2_GraphicSeed 
  : public Seed
{
public:
#ifdef _WIN32
  word dib_data;
  dword dib_size;
#else /* _WIN32 */
  HANDLE dib_data;
  DWORD dib_size;
#endif /* _WIN32 */
  SV2_GraphicSeed()
  : Seed(IDGraphicObject, Margin(0, 0, 0, 0)) 
  { ; }
} STATIC_NEAR sv2_gs;

// load seed data
Seed* 
GraphicSeed::Load(PStream* stream, Margin& m, int version)
{
  GraphicSeed* gs = new GraphicSeed(m);
  
  if (version >= sv_bw1) 
  {
    gs->dib_size = stream->InUDWord();
    CHECK 0;
  }
  else 
  {
    int size = sizeof(SV2_GraphicSeed);
    stream->In(&sv2_gs, size);
    CHECK 0;
    gs->dib_size = sv2_gs.dib_size;
  }
  
  VMem* vm_dib = VMem::Make(gs->dib_size);
  if (((void*)*vm_dib) == 0) 
  {
    delete gs;
    return 0;
  }
  
  gs->SetVMDIB(vm_dib);
  stream->InHuge((void*)*vm_dib, gs->dib_size);
  CHECK 0;
  
  return gs;
}

// ------------------------------------------------------------
// all layout seed

// constructor
AllLayoutSeed::AllLayoutSeed(FrameObject* fo)
  : Seed(fo)
  
{
  ; // do nothing
}

// constructor for make template
AllLayoutSeed::AllLayoutSeed(uword)
  : Seed(IDAllLayoutObject, Margin(0, 0, 0, 0))
{
  ; // do nothing
}

// save seed data
void AllLayoutSeed::Save(PStream*)
{
  ; // do nothing
}
			  
// load seed data
Seed* 
AllLayoutSeed::Load(PStream* stream, Margin& m, int version)
{
  AllLayoutSeed* as = new AllLayoutSeed(m);
  
  if (version <= sv_snow) 
  {
    int size = sizeof(AllLayoutSeed);
    stream->In(as, size);
    CHECK 0;
  }
  return as;	// success
}

// ------------------------------------------------------------
// metafile seed

// constructor
MetafileSeed::MetafileSeed(FrameObject* fo)
  : Seed(fo)
{
  MetafileObject* mfo = (MetafileObject*)fo;
  Metafile* mf = mfo->GetMetafile();
  mp = * (mf->GetMetafilePict());
  metafile_size = ::GlobalSize(mp.hMF);
  ref_flag = True;
}

// constructor for file load
MetafileSeed::MetafileSeed(Margin& m)
  : Seed(IDMetafileObject, m)
{
  ref_flag = False; 
}

// desturctor
MetafileSeed::~MetafileSeed()
{
  if (!ref_flag && (mp.hMF != 0)) 
  {
    ::GlobalFree(mp.hMF);
  }
}

#ifdef _WIN32
struct W16METAFILEPICT
{
  word mm;
  word xExt;
  word yExt;
  word hMF;
};
#else /* _WIN32 */
struct W32METAFILEPICT
{
  dword mm;
  dword xExt;
  dword yExt;
  dword hMF; // 32 bit ptr.
};
#endif /* _WIN32 */

class SV2_MetafileSeed 
  : public Seed 
{
public:
#ifdef _WIN32
  W16METAFILEPICT mp;
  dword metafile_size;
#else /* _WIN32 */
  METAFILEPICT mp;
  DWORD metafile_size;
#endif /* _WIN32 */
  SV2_MetafileSeed()
  : Seed(IDMetafileObject, Margin(0, 0, 0, 0)) 
  { ; }
} STATIC_NEAR sv2_ms;

// save seed data
void 
MetafileSeed::Save(PStream* stream)
{
#ifdef _WIN32
  stream->Out(&mp, sizeof(METAFILEPICT));
  CHECK;
#else /* _WIN32 */
  W32METAFILEPICT w32mp;
  w32mp.mm = mp.mm;
  w32mp.xExt = mp.xExt;
  w32mp.yExt = mp.yExt;
  w32mp.hMF = 0;
  stream->Out(&w32mp, sizeof(W32METAFILEPICT));
  CHECK;
#endif /* _WIN32 */
  
  stream->OutUDWord(metafile_size);
  CHECK;
  
  void* p = ::GlobalLock(mp.hMF);
  stream->OutHuge(p, metafile_size);
  CHECK;
  ::GlobalUnlock(mp.hMF);
}

// load seed data
Seed* 
MetafileSeed::Load(PStream* stream, Margin& m, int version)
{
  MetafileSeed* ms = new MetafileSeed(m);

  switch (version)
  {
  case sv_luna:
  case sv_snow:
    {
      int size = sizeof(SV2_MetafileSeed);
      stream->In(&sv2_ms, size);
      CHECK 0;
#ifdef _WIN32
      ms->mp.mm = sv2_ms.mp.mm;
      ms->mp.xExt = sv2_ms.mp.xExt;
      ms->mp.yExt = sv2_ms.mp.yExt;
      ms->mp.hMF = 0;
#else /* _WIN32 */
      ms->mp = sv2_ms.mp;
#endif /* _WIN32 */
      ms->metafile_size = sv2_ms.metafile_size;
    }
    break;

  case sv_bw1:
  case sv_bw2:
  case sv_bw3:
#ifdef _WIN32
    {
      int size = sizeof(W16METAFILEPICT);
      W16METAFILEPICT w16mp;
      stream->In(&w16mp, size);
      CHECK 0;
    
      ms->mp.mm = w16mp.mm;
      ms->mp.xExt = w16mp.xExt;
      ms->mp.yExt = w16mp.yExt;
      ms->mp.hMF = 0;

      ms->metafile_size = stream->InUDWord();
      CHECK 0;
    }
#else /* _WIN32 */
    {
      int size = sizeof(METAFILEPICT);
      stream->In(&ms->mp, size);
      CHECK 0;
    
      ms->metafile_size = stream->InUDWord();
      CHECK 0;
    }
#endif /* _WIN32 */
    break;

  default:
#ifdef _WIN32
    {
      int size = sizeof(METAFILEPICT);
      stream->In(&ms->mp, size);
      CHECK 0;
    
      ms->metafile_size = stream->InUDWord();
      CHECK 0;
    }
#else /* _WIN32 */
    {
      int size = sizeof(W32METAFILEPICT);
      W16METAFILEPICT w32mp;
      stream->In(&w32mp, size);
      CHECK 0;
    
      ms->mp.mm = (int)w32mp.mm;
      ms->mp.xExt = (int)w32mp.xExt;
      ms->mp.yExt = (int)w32mp.yExt;
      ms->mp.hMF = 0;

      ms->metafile_size = stream->InUDWord();
      CHECK 0;
    }
#endif /* _WIN32 */
    break;
  }
  
#ifdef _WIN32
  ms->mp.hMF = (HMETAFILE)::GlobalAlloc(GHND, ms->metafile_size);
#else /* _WIN32 */
  ms->mp.hMF = ::GlobalAlloc(GHND, ms->metafile_size);
#endif /* _WIN32 */
  if (ms->mp.hMF == 0) 
  {
    delete ms;
    return 0;
  }
  
  void* p = ::GlobalLock(ms->mp.hMF);
  udword dsize = ms->metafile_size;
  stream->InHuge(p, dsize);
  ::GlobalUnlock(ms->mp.hMF);
  
  return ms;	// success
}

// copy metafilepict
METAFILEPICT* 
MetafileSeed::CopyMETAFILEPICT()
{
  METAFILEPICT* ret = new METAFILEPICT;
  *ret = mp;
  return ret;
}

// ------------------------------------------------------------
// unknown seed

// constructor
UnknownSeed::UnknownSeed(FrameObject* fo)
  : Seed(fo)
{
  err = False;
}

// constructor for make template
UnknownSeed::UnknownSeed(uword)
  : Seed(IDUnknownObject, Margin(0, 0, 0, 0))
{
  err = False;
}

// constructor for file load
UnknownSeed::UnknownSeed(Margin& m, bool err) 
  : Seed(IDUnknownObject, m)
{
  UnknownSeed::err = err;
}

// save seed data
void 
UnknownSeed::Save(PStream*)
{
  ; // do nothing
}
			
// load seed data
Seed* 
UnknownSeed::Load(PStream* stream, Margin& m, int version)
{
  UnknownSeed* ms = new UnknownSeed(m);
  if (version <= Seed::sv_snow) 
  {
    int size = sizeof(UnknownSeed);
    stream->In(ms, size);
    CHECK 0;
  }
  return ms;	// success
}
