// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: atts.cpp,v 3.5 1999/07/24 15:38:09 kudou Exp $
// manage attribute table

#include "pword.h"
#include "pstreams.h"
#include "attribut.h"
#include "derivs.h"
#include "styles.h"
#include "fonts.h"
#include "borders.h"
#include "rats.h"
#include "atts.h"

#define T AttsEntry
#define TABLE BaseAttsTable
#define SEG BaseAttsTableSeg
#define DO_READ_WRITE
#define NO_ENTRY_ADD_TO_INTERFACE
#include "tabletem.cpp"

AttsTable STATIC_NEAR GlobalAttsTable;

unsigned int DefaultCharAtts;
unsigned int DefaultParAtts;
unsigned int UnknownParAtts;
unsigned int UnknownCharAtts;

// here are default attribute values which are placed in the table
// by its constructor

static int default_char_atts[] =
{
  0,			// CA_zero_reserved
  0,			// CA_facename
  0,			// CA_facename2
  DEFAULT_POINT,	// CA_size
  0,			// CA_delta_size
  0,			// CA_bold
  0,			// CA_italic
  0,			// CA_subscript
  0,			// CA_compressed
  0,			// CA_unbreakable
  0,			// CA_hidden
  0,			// CA_lines
  0,			// CA_reserved_12
  0,			// CA_strikeout
  TAB_left,		// CA_tab
  0,			// CA_inverted
  AttsEntry::UNKNOWN,
};

static int default_par_atts[] =
{
  0,					// PA_zero_reserved
  0,					// PA_firstleftmargin
  0,					// PA_secondleftmargin
  0,					// PA_rightmargin
  AL_flushright,			// PA_alignment
  0,					// PA_parspace
  0,					// PA_linespace
  0,					// PA_lines
  0,					// PA_nobreak
  PL_none,				// PA_placement
  0,					// PA_charatt
  AL_vertical_adjust,			// PA_vertical_alignment
  mm_to_iu(15),			// PA_tabsize
  AttsEntry::UNKNOWN,
};

void
AttsTable::Init()
{
  default_char_atts[CA_facename] = GlobalFontTable.ansi_default;
  default_char_atts[CA_facename2] = GlobalFontTable.kanji_default;
  
  AttsEntry def_char(AttsEntry::CHARSTYLE, default_char_atts);
  AttsEntry def_par(AttsEntry::PARSTYLE, default_par_atts);
  
  DefaultCharAtts = GlobalAttsTable.MakeEntry(def_char);
  DefaultParAtts  = GlobalAttsTable.MakeEntry(def_par);
  
  int unknown_vals[AttsEntry::NUMATTS], n=AttsEntry::NUMATTS, *v=unknown_vals;
  while (n--)
  {
    *v++ = AttsEntry::UNKNOWN;
  }
  AttsEntry unknown_char(AttsEntry::CHARSTYLE, unknown_vals);
  UnknownCharAtts = GlobalAttsTable.MakeEntry(unknown_char);
  AttsEntry unknown_par(AttsEntry::PARSTYLE, unknown_vals);
  UnknownParAtts = GlobalAttsTable.MakeEntry(unknown_par);
}

void
AttsTable::SetDefaultSize(int size)
{
  if (size == 0)
  {
    size = 12;
  }
  SET_BOUND(size, 6, 72);
  ::get_atts_entry(DefaultCharAtts) .vals[CA_size] = size;
}

// ------------------------------------------------------------
// ATTSENTRY

// Init -- used from various constructors
// private method
void NEAR
AttsEntry::Init()
{
  for (int i = 0; i < NUMATTS; ++i)
  {
    this->vals[i] = UNKNOWN;
  }
  this->rat_entry = NULL;
};

// public constructor
AttsEntry::AttsEntry()
{
  this->Init();
}

// public constructor
AttsEntry::AttsEntry(int type)
{
  this->Init();
  this->type = type;
}

// public copy constructor
AttsEntry::AttsEntry(AttsEntry& entry)
{
  this->Init();
  this->type = entry.type;
  this->CopyVals(entry);
}

// construct an attribute table entry from derivation information,
// that is an entry in the attribute derivation table
AttsEntry::AttsEntry(DerivEntry& adt_entry)
{
  this->Init();
  this->Plug(adt_entry);
}

// construct from a list of values
AttsEntry::AttsEntry(int type,  int* inits)
{
  Init();
  AttsEntry::type = type;
  
  for (int i=0; i<NUMATTS && *inits != UNKNOWN; i++, inits++)
  {
    vals[i] = *inits;
  }
}

// utility used in the constructor above -- copy information from
// style entries
void
AttsEntry::Plug(DerivEntry& adt_entry)
{
  int base = adt_entry.GetBase();
  
  // this code is why styles work!!
  if (base != 0)
  {
    CopyVals(::get_style_entry(base) .GetAttsEntry());
  }
  AttsEntry& at_entry = adt_entry.GetAttsEntry();
  CopyVals(at_entry);
  type = at_entry.type;
}

// copy utility used in copy constructor and Plug routine above
// This unilaterally copies all unknwon values from the source,
// overwriting any existing values
void
AttsEntry::CopyVals(AttsEntry& entry)
{
  int val;
  for (int i=0; i<NUMATTS; i++)
  {
    if ((val=entry.vals[i]) != UNKNOWN)
    {
      vals[i] = val;
    }
  }
}

// another sort of copy -- this one is called "fillin" because
// it copies from the source only when the target is unknown.
// it is used when filling in defaults, for example
void
AttsEntry::FillIn(AttsEntry& entry)
{
  for (int i=0; i<NUMATTS; i++)
  {
    if (vals[i]==UNKNOWN)
    {
      vals[i] = entry.vals[i];
    }
  }
}

// Subtract -- mark atts with identical value in one entry to
// UNKNOWN in this one.
// This is used to wipe away locally applied attributes when the user
// applies an attribute which is the same as the style or default would
// give the entry anyway.
void
AttsEntry::Subtract(AttsEntry& back)
{
  for (int i=0; i<NUMATTS; i++)
  {
    if (back.vals[i]==vals[i])
    {
      vals[i] = UNKNOWN;
    }
  }
}

// SubtractAny -- change atts with any value in one entry to UNKNOWN in this.
// This is used to wipe away locally applied attributes when the user
// applies a style which defines the same attribute.
void
AttsEntry::SubtractAny(AttsEntry& back)
{
  for (int i=0; i<NUMATTS; i++)
  {
    if (back.vals[i]!=UNKNOWN)
    {
      vals[i] = UNKNOWN;
    }
  }
}

// private method
unsigned char NEAR
AttsEntry::GetHash()
{
  unsigned char x = (unsigned char) this->type;
  int i = AttsEntry::NUMATTS;
  do
  {
    --i;
    x = x ^ (x << 1) ^ this->vals[i];
  } while (i);
  return(x);
}

int
operator == (AttsEntry& e1, AttsEntry& e2)
{
  int i = AttsEntry::NUMATTS;
  do
  {
    --i;
    if (e1.vals[i] != e2.vals[i])
    {
      return(False);
    }
  } while (i);
  return(e1.type == e2.type);
}

// routines related to marking entries in the attribute table and
// writing them to streams

// mark an entry and all other entries in other tables affected
void
AttsEntry::Mark()
{
  if (!this->GetMark())
  {
    this->SetMark();
    
    // any additional attributes whose values point into other tables
    // have to be added here
    
    unsigned int a;
    if (type == CHARSTYLE)
    {
      if (VALID_ATTR_INDEX_P(a = vals[CA_facename]))
      {
	::get_font_entry(a) .Mark();
      }
      if (VALID_ATTR_INDEX_P(a = vals[CA_facename2]))
      {
	::get_font_entry(a) .Mark();
      }
      if (VALID_ATTR_INDEX_P(a = vals[CA_lines]))
      {
	::get_border_entry(a) .Mark();
      }
    }
    else if (this->type == PARSTYLE)
    {
      if (VALID_ATTR_INDEX_P(a = vals[PA_charatt]))
      {
	::get_deriv_entry(a) .Mark();
      }
      if (VALID_ATTR_INDEX_P(a = vals[PA_lines]))
      {
	::get_border_entry(a) .Mark();
      }
    }
  }
}

// AllUnknown -- check if all entries are unknown
int
AttsEntry::AllUnknown()
{
  for (int j=1; j<NUMATTS; j++)
  {
    if (vals[j]!=UNKNOWN)
    {
      return 0;
    }
  }
  return 1;
}

unsigned int
AttsEntry::Enter()
{
  return GlobalAttsTable.MakeEntry(*this);
}

void
AttsEntry::ReadFromStream(PStream* stream)
{
  word vers;
  stream->StartTypedBlockIn(PStream::HDR_ATTSENTRY, vers);
  CHECK;
  
  type = stream->InUWord(); CHECK;

  int v;
  if (type == CHARSTYLE)
  {
    v = vals[CA_facename] = GlobalFontTable.ReadEntry(stream);
    v = vals[CA_facename2] = GlobalFontTable.ReadEntry(stream);
    v = vals[CA_size] = stream->InWord(); CHECK;
    v = vals[CA_delta_size] = stream->InWord(); CHECK;
    v = vals[CA_bold] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_italic] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_subscript] = stream->InWord(); CHECK;
    v = vals[CA_compressed] = stream->InWord(); CHECK;
    if (v < 0 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_unbreakable] = stream->InWord() ; CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_hidden] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_lines] = GlobalBorderTable.ReadEntry(stream); CHECK;
    v = vals[CA_strikeout] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_tab] = stream->InWord(); CHECK;
    if ((!(v >= TAB_FIRST && v <= TAB_LAST)) && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[CA_inverted] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
  }
  else
  {
    v = vals[PA_firstleftmargin] = stream->InWord(); CHECK;
    v = vals[PA_secondleftmargin] = stream->InWord(); CHECK;
    v = vals[PA_rightmargin] = stream->InWord(); CHECK;
    v = vals[PA_alignment] = stream->InWord(); CHECK;
    if ((!(v >= AL_FIRST && v <= AL_LAST)) && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_parspace] = stream->InWord(); CHECK;
    if (v < 0 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_linespace] = stream->InWord(); CHECK;
    if (v < 0 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_lines] = GlobalBorderTable.ReadEntry(stream); CHECK;
    v = vals[PA_nobreak] = stream->InWord(); CHECK;
    if (v != 0 && v != 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_placement] = stream->InWord(); CHECK;
    if ((!(v >= PL_FIRST && v <= PL_LAST)) && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_charatt] = GlobalDerivTable.ReadEntry(stream); CHECK;
    v = vals[PA_vertical_alignment] = stream->InWord(); CHECK;
    if ((!(v >= AL_vertical_FIRST && v <= AL_vertical_LAST)) && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
    v = vals[PA_tabsize] = stream->InWord(); CHECK;
    if (v < 1 && v != AttsEntry::UNKNOWN)
    {
      stream->SetError(SE_BADVAL);
      return;
    }
  }
  stream->EndBlockIn();
}

void
AttsEntry::WriteToStream(PStream* stream)
{
  stream->StartBlockOut(PStream::HDR_ATTSENTRY);
  CHECK;
  
  stream->OutUWord(type);
  CHECK;
  
  if (type == CHARSTYLE)
  {
    GlobalFontTable.WriteEntry(stream, vals[CA_facename]);
    GlobalFontTable.WriteEntry(stream, vals[CA_facename2]);
    stream->OutWord(vals[CA_size]);
    stream->OutWord(vals[CA_delta_size]);
    stream->OutWord(vals[CA_bold]);
    stream->OutWord(vals[CA_italic]);
    stream->OutWord(vals[CA_subscript]);
    stream->OutWord(vals[CA_compressed]);
    stream->OutWord(vals[CA_unbreakable]);
    stream->OutWord(vals[CA_hidden]);
    GlobalBorderTable.WriteEntry(stream, vals[CA_lines]);
    stream->OutWord(vals[CA_strikeout]);
    stream->OutWord(vals[CA_tab]);
    stream->OutWord(vals[CA_inverted]);
  }
  else
  {
    stream->OutWord(vals[PA_firstleftmargin]);
    stream->OutWord(vals[PA_secondleftmargin]);
    stream->OutWord(vals[PA_rightmargin]);
    stream->OutWord(vals[PA_alignment]);
    stream->OutWord(vals[PA_parspace]);
    stream->OutWord(vals[PA_linespace]);
    GlobalBorderTable.WriteEntry(stream, vals[PA_lines]);
    stream->OutWord(vals[PA_nobreak]) ;
    stream->OutWord(vals[PA_placement]);
    GlobalDerivTable.WriteEntry(stream, vals[PA_charatt]);
    stream->OutWord(vals[PA_vertical_alignment]);
    stream->OutWord(vals[PA_tabsize]);
  }
  stream->EndBlockOut();
}

#ifndef NDEBUG

void
AttsEntry::DumpHeader(FILE* f)
{
  fprintf(f,
	   "\nAttribute Table\n\n"
	   "NNNN T  01  02  03  04  05  06  07  08  09  10  11  12  13  14  15  16\n"
	 );
}

void
AttsEntry::Dump(unsigned int i, void* v)
{
  FILE* f = (FILE*) v;
  fprintf(f, "%4d %1d", i, type);
  for (int j=1; j<NUMATTS; j++)
  {
    if (vals[j]==UNKNOWN)
    {
      fprintf(f, "  --");
    }
    else
    {
      fprintf(f, "%4d", vals[j]);
    }
  }
  fprintf(f, "\n");
}
#endif // NDEBUG

