// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: borderli.cpp,v 3.2 1999/05/12 00:22:16 kudou Exp $
// class BorderLine
// \NX

#include "pword.h"
#include "borderli.h"
#include "attribut.h"
#include "borders.h"
#include "rect.h"
#include "screen.h"
#include "view.h"

#define index_top index_x
#define index_bot index_y
#define index_horizontal index_x
#define index_vertical index_y

#define TOP_X(r) RECT_UNIT(r, INDEX_TOP, index_x)
#define BOT_X(r) RECT_UNIT(r, INDEX_BOT, index_x)
#define TOP_Y(r) RECT_UNIT(r, index_top, index_y)
#define BOT_Y(r) RECT_UNIT(r, index_bot, index_y)

#define SIGN(value) (index_x == INDEX_X ? (value) : - (value))

const uword BorderLine::line_width_max  = 256;	// line width max
const uword BorderLine::default_width   = 2;

uword BorderLine::def_ch_index = 1;	// default character border index
uword BorderLine::def_par_index = 1;	// default paragraph border index
uword BorderLine::def_fr_index = 1;	// default frame border index
uword BorderLine::def_table_fr_index = 1;// default frame border index for tables
uword BorderLine::zero_index = 0;	// zero width border

void
BorderLine::Init()
{
  this->single = True;
  this->type = Solid;
  this->width = 2;
  this->algn = 0;
  this->bisai = False;
}

BorderLine::BorderLine(BLLineType mt,
		       uword mw, unsigned int algn, bool single)
{
  this->Init();
  this->single = single;
  this->type = mt;
  this->width = mw;
  this->algn = algn;
}

// null border constructor
BorderLine::BorderLine()
{
  this->Init();
}

// ------------------------------------------------------------
// basic methods

// This method returns width of border line.
// `this->single ? 2 : 4' means:
// Width of single-BISAI line is 1 dot on printer, width of double-BISAI
// line is 3 dot dot on printer.
// `2' and `4' are `1 + 1' and `3 + 1'.  I use `+ 1' to keep drawing area.
// INDEX_HV is a INDEX_HORIZONTAL or INDEX_VERTICAL.  If you want width of
// horizontal border line, use INDEX_HORIZONTAL, otherwise INDEX_VERTICAL.
Iunit
BorderLine::strict_width(int index_hv)
{
  // We use `REVERSED_INDEX(index_hv)' to calculate line width, because
  // INDEX_HORIZONTAL is defined equal to INDEX_X.
  Iunit width = (this->bisai
		 ? (Iunit) Printer_View->du_to_lu(this->single ? 2 : 4,
						   REVERSED_INDEX(index_hv))
		 : mm0_to_iu(this->width));
  return(MAX(width, 1));
}

// public lexical method
// This method returns pixel based width of border line.  In case of
// double fine line, we use 4 dot width, because 3 dot is too fine to see
// gap space.
int
BorderLine::strict_pixel_width(View* view, int index_hv)
{
  int width = this->single ? 1 : 4;
  if (!this->bisai)
  {
    int pixel_width = view->lu_to_du(this->strict_width(index_hv),
				      REVERSED_INDEX(index_hv));
    SET_MAX(width, pixel_width);
  }
  return(width);
}

Iunit
BorderLine::GetWidth()
{
  return(this->bisai ? 0 : mm0_to_iu(this->width));
}

// public lexical method
// This method returns brush handle to draw border line.  INDEX_HV choose
// direction of line. INDEX_HORIZONTAL means horizontal line,
// INDEX_VERTICAL means vertical line.
HBRUSH
BorderLine::get_brush(int index_hv)
{
  switch (this->type)
  {
   case Broken:
    return(Screen::simple_broken_brush[index_hv]);
    
   case Dot:
    return(Screen::simple_dotted_brush[index_hv]);
    
   case Invisible:
    return(Screen::white_brush);
    
   default:
    return(Screen::black_brush);
  }
}

static void NEAR
fill_rect(HDC dc, Drect* dr, HBRUSH brush)
{
  FillRect(dc, CastDrectToWindowsRECT(dr), brush);
}

// public lexical method
// This method draws border line into rectangle DR.  INDEX_HV choose
// direction of line. INDEX_HORIZONTAL means horizontal line,
// INDEX_VERTICAL means vertical line.
void
BorderLine::draw_border_in_drect(View* view, Drect* r, int index_hv)
{
  Drect dr[1];
  *dr = *r;
  int reversed_index_hv = REVERSED_INDEX(index_hv);
  
  // We design code based on horizontal line.
#define index_x index_hv
#define index_y reversed_index_hv
  
  if (!RECT_IS_EMPTY_P(dr) && this->type != Invisible)
  {
    int width = (RECT_UNIT(dr, INDEX_BOT, index_y)
		 - RECT_UNIT(dr, INDEX_TOP, index_y));
    HBRUSH brush = this->get_brush(index_hv);
    view->set_brush_org(dr, brush);
    HDC dc = view->GetDC();
    if (!this->single && 3 <= width)
    {
      int gap = view->lu_to_du(DOUBLE_LINE_GAP, index_y);
      SET_BOUND(gap, 1, width - 2);
      int line_width = (width - gap) / 2;
      int old_bot = RECT_UNIT(dr, INDEX_BOT, index_y);
      RECT_UNIT(dr, INDEX_BOT, index_y) = (RECT_UNIT(dr, INDEX_TOP, index_y)
					    + line_width);
      fill_rect(dc, dr, brush);
      RECT_UNIT(dr, INDEX_TOP, index_y) = old_bot - line_width;
      RECT_UNIT(dr, INDEX_BOT, index_y) = old_bot;
    }
    fill_rect(dc, dr, brush);
  }
#undef index_x
#undef index_y
}

// static method
BorderLine* 
BorderLine::Get(int index)
{
  return(index == 0 ? NULL : &(::get_border_entry(index) .GetBorderLine()));
}

// ------------------------------------------------------------
// paragraph border control

void
BorderLine::draw_para_border(View* view, Lrect* guide_lrect, int index_x)
{
  int index_y = REVERSED_INDEX(index_x);
  
  Drect rect[1];
  view->lrect_to_drect(guide_lrect, rect);
  
  Drect h0[1];
  Drect h1[1];
  Drect v0[1];
  Drect v1[1];
  *h0 = *rect;
  *h1 = *rect;
  *v0 = *rect;
  *v1 = *rect;
  
  int strict_size[2];
  strict_size[INDEX_X] = this->strict_pixel_width(view, INDEX_VERTICAL);
  strict_size[INDEX_Y] = this->strict_pixel_width(view, INDEX_HORIZONTAL);
  int strict_width_x = strict_size[index_x];
  int strict_width_y = strict_size[index_y];
  
  int align = (int) this->algn;
  
  BOT_X(v0) = TOP_X(v0);
  if ((align & AlignLeft) != 0)
  {
    TOP_X(v0) -= strict_width_x;
  }
  
  TOP_X(v1) = BOT_X(v1);
  if ((align & AlignRight) != 0)
  {
    BOT_X(v1) += strict_width_x;
  }
  
  BOT_Y(h0) = TOP_Y(h0);
  if ((align & AlignAbove) != 0)
  {
    BOT_Y(h0) += SIGN(strict_width_y);
  }
  
  TOP_Y(h1) = BOT_Y(h1);
  if ((align & AlignBelow) != 0)
  {
    TOP_Y(h1) -= SIGN(strict_width_y);
  }
  
  // adjusting corner positions
  TOP_X(h0) = TOP_X(h1) = TOP_X(v0);
  BOT_X(h0) = BOT_X(h1) = BOT_X(v1);
  TOP_Y(v0) = TOP_Y(v1) = TOP_Y(h0);
  BOT_Y(v0) = BOT_Y(v1) = BOT_Y(h1);
  
  if (this->single || strict_width_x < 3 || strict_width_y < 3)
  {
    this->draw_border_in_drect(view, v0, index_vertical);
    this->draw_border_in_drect(view, v1, index_vertical);
    this->draw_border_in_drect(view, h0, index_horizontal);
    this->draw_border_in_drect(view, h1, index_horizontal);
  }
  else
  {
    int width[2];
    {
      int foo = view->ScaleDunitX(DOUBLE_LINE_GAP);
      SET_BOUND(foo, 1, strict_size[INDEX_X] - 2);
      width[INDEX_X] = (strict_size[INDEX_X] - foo) / 2;
    }
    {
      int foo = view->ScaleDunitY(DOUBLE_LINE_GAP);
      SET_BOUND(foo, 1, strict_size[INDEX_Y] - 2);
      width[INDEX_Y] = (strict_size[INDEX_Y] - foo) / 2;
    }
    int width_x = width[index_x];
    int width_y = width[index_y];
    int signed_width_y = SIGN(width_y);
    
    HBRUSH brush[2];
    brush[INDEX_HORIZONTAL] = this->get_brush(INDEX_HORIZONTAL);
    brush[INDEX_VERTICAL] = this->get_brush(INDEX_VERTICAL);
    HBRUSH vbrush = brush[index_vertical];
    HBRUSH hbrush = brush[index_horizontal];
    
    HDC dc = view->GetDC();
    Drect r[1];
    
    if (!RECT_IS_EMPTY_P(v0))
    {
      *r = *v0;
      view->set_brush_org(r, vbrush);
      BOT_X(r) = TOP_X(v0) + width_x;
      fill_rect(dc, r, vbrush);
      
      *r = *v0;
      TOP_X(r) = BOT_X(v0) - width_x;
      if (!RECT_IS_EMPTY_P(h0))
      {
	TOP_Y(r) = BOT_Y(h0) - signed_width_y;
      }
      if (!RECT_IS_EMPTY_P(h1))
      {
	BOT_Y(r) = TOP_Y(h1) + signed_width_y;
      }
      fill_rect(dc, r, vbrush);
    }
    
    if (!RECT_IS_EMPTY_P(v1))
    {
      *r = *v1;
      view->set_brush_org(r, vbrush);
      TOP_X(r) = BOT_X(v1) - width_x;
      fill_rect(dc, r, vbrush);
      
      *r = *v1;
      BOT_X(r) = TOP_X(v1) + width_x;
      if (!RECT_IS_EMPTY_P(h0))
      {
	TOP_Y(r) = BOT_Y(h0) - signed_width_y;
      }
      if (!RECT_IS_EMPTY_P(h1))
      {
	BOT_Y(r) = TOP_Y(h1) + signed_width_y;
      }
      fill_rect(dc, r, vbrush);
    }
    
    if (!RECT_IS_EMPTY_P(h0))
    {
      *r = *h0;
      view->set_brush_org(r, hbrush);
      BOT_Y(r) = TOP_Y(h0) + signed_width_y;
      fill_rect(dc, r, hbrush);
      
      *r = *h0;
      TOP_Y(r) = BOT_Y(h0) - signed_width_y;
      if (!RECT_IS_EMPTY_P(v0))
      {
	TOP_X(r) = BOT_X(v0) - width_x;
      }
      if (!RECT_IS_EMPTY_P(v1))
      {
	BOT_X(r) = TOP_X(v1) + width_x;
      }
      fill_rect(dc, r, hbrush);
    }
    
    if (!RECT_IS_EMPTY_P(h1))
    {
      *r = *h1;
      view->set_brush_org(r, hbrush);
      TOP_Y(r) = BOT_Y(h1) - signed_width_y;
      fill_rect(dc, r, hbrush);
      
      *r = *h1;
      BOT_Y(r) = TOP_Y(h1) + signed_width_y;
      if (!RECT_IS_EMPTY_P(v0))
      {
	TOP_X(r) = BOT_X(v0) - width_x;
      }
      if (!RECT_IS_EMPTY_P(v1))
      {
	BOT_X(r) = TOP_X(v1) + width_x;
      }
      fill_rect(dc, r, hbrush);
    }
  }
}

// ------------------------------------------------------------
// frame border control

Iunit
BorderLine::GetLeftWidth(uword index)
{
  if (index != 0)
  {
    BorderLine* b = BorderLine::Get(index);
    switch (b->GetVAlign())
    {
     case AlignLeft:
      return(b->GetWidth());
      
     case AlignCenter:
      return(b->GetWidth() / 2);
    }
  }
  return(0);
}

Iunit
BorderLine::GetRightWidth(uword index)
{
  if (index != 0)
  {
    BorderLine* b = BorderLine::Get(index);
    switch (b->GetVAlign())
    {
     case AlignCenter:
      return((b->GetWidth() + 1) / 2);
      
     case AlignRight:
      return(b->GetWidth());
    }
  }
  return(0);
}

Iunit
BorderLine::GetUpperWidth(uword index)
{
  if (index != 0)
  {
    BorderLine* b = BorderLine::Get(index);
    switch (b->GetHAlign())
    {
     case AlignAbove:
      return(b->GetWidth());
      
     case AlignCenter:
      return(b->GetWidth() / 2);
    }
  }
  return(0);
}

Iunit
BorderLine::GetLowerWidth(uword index)
{
  if (index != 0)
  {
    BorderLine* b = BorderLine::Get(index);
    switch (b->GetHAlign())
    {
     case AlignCenter:
      return((b->GetWidth() + 1) / 2);
      
     case AlignBelow:
      return(b->GetWidth());
    }
  }
  return(0);
}

// ------------------------------------------------------------
// attribute control

// get indexed borderline
BorderLine&
BorderLine::GetBorderLine(uword index)
{
  if (index == 0)
  {
    static BorderLine* test = NULL;
    if (test == NULL)
    {
      test = new BorderLine(Solid, Point_to_iu(1), Point_to_iu(1));
    }
    return(*test);
  }
  return(::get_border_entry(index) .GetBorderLine());
}

unsigned int BorderLine::Enter()
{
  BorderEntry entry(*this);
  return GlobalBorderTable.MakeEntry(entry);
}

// Present -- return True if this borderline "exists"

// static
bool
BorderLine::Present(uword index)
{
  return(index == 0 ? False : BorderLine::Get(index) ->Present());
}

bool BorderLine::Present()
{
  return algn!=0 ? True : False;
}

void
BorderLine::Mark(uword index)
{
  if (VALID_ATTR_INDEX_P(index))
  {
    ::get_border_entry(index) .SetMark();
  }
}

// get force alignmented index
uword
BorderLine::ForceAlignment(uword index, BLAlignment algn)
{
  BorderLine* bl = BorderLine::Get(index);
  if (bl->GetAlignment() && bl->GetAlignment() != algn)
  {
    BorderLine bl_fix(*bl);
    bl_fix.SetAlign(algn);
    index = bl_fix.Enter();
  }
  return(index);
}
