// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: new_fmt.cpp,v 3.4 2000/05/03 12:18:34 kudou Exp $
// formatter

#include "pword.h"
#include "attribut.h"
#include "borderli.h"
#include "bufnew.h"
#include "docconte.h"
#include "docprese.h"
#include "document.h"
#include "docwindo.h"
#include "frameins.h"
#include "layoutin.h"
#include "layoutte.h"
#include "line.h"
#include "panel.h"
#include "pref.h"
#include "pwordpre.h"
#include "rats.h"
#include "rect.h"
#include "screen.h"
#include "sjis.h"
#include "textflow.h"
#include "wstdio.h"

// Aliases for tategaki
#define index_top index_x
#define index_bot index_y
#define index_width index_x
#define index_height index_y
#define index_horizontal index_x
#define index_vertical index_y

#define TOP_X(r) RECT_UNIT(r, INDEX_TOP, index_x)
#define BOT_X(r) RECT_UNIT(r, INDEX_BOT, index_x)
#define TOP_Y(r) RECT_UNIT(r, index_top, index_y)
#define BOT_Y(r) RECT_UNIT(r, index_bot, index_y)

#define SIGN(value) (index_x == INDEX_X ? (value) : -(value))

#ifndef MAX_ULONG
#define MAX_ULONG	((unsigned long) 0 - 1)
#endif

#ifndef MAX_USHORT
#define MAX_USHORT	((unsigned short) -1)
#endif

#ifndef MAX_LONG
#define MAX_LONG	((long) (MAX_ULONG >> 1))
#endif

#ifndef MAX_SHORT
#define MAX_SHORT	((short) (MAX_USHORT >> 1))
#endif

#define MAX_IUNITS	((Iunit) MAX_SHORT)
#define MAX_LPOS	((Lunit) MAX_LONG)

// ------------------------------------------------------------
// character class operations

void
soft_japanese_spaces()
{
  // set katakana space class
  kana_char_class_table[0xa0] = CharClass_RANDOM_PUNCT;
  
  // set zenkaku space class
  sjis_char_class_table[0x8140 - SJIS_TABLE_BASE] = CharClass_RANDOM_PUNCT;
}

CharClass
GetCharClass(sjis c)
{
  switch (GET_CHAR_TYPE(c))
  {
   case CharType_ANSI:
    return(ansi_char_class_table[c]);
    
   case CharType_KANA:
    return(kana_char_class_table[c & 0xff]);
    
   case CharType_KANJI:
    return(SJIS_KANJI_BASE < c ? CharClass_KANJI
	    : SJIS_RUSSIAN_GREEK_BASE <= c ? CharClass_EUROCHAR
	    : (SJIS_TABLE_BASE <= c
	       ? sjis_char_class_table[c - SJIS_TABLE_BASE]
	       : CharClass_RANDOM_PUNCT));
    
   case CharType_PANEL:
    return(CharClass_SPECIAL);
    
   default:
    return(CharClass_RANDOM_PUNCT);
  }
}

// ------------------------------------------------------------
// TextFlow's methods

void
TextFlow::FormatNeeded()
{
  this->format_needed_p = True;
  this->GetDocContent() ->SetFormatNeededP(True);
}

FrameInstance* 
TextFlow::GetMaybeBrokenFrame()
{
#if (!defined(NDEBUG))
  if (this->GetFirstFrame() == NULL)
  {
    syserr("TextFlow::GetMaybeBrokenFrame");
  }
#endif
  FrameInstance* frame = this->formatted_frame;
  return(frame == NULL ? this->GetFirstFrame() : frame);
}

void
TextFlow::InitializeOnFlags()
{
  if (!this->on_flags_is_initialized_p)
  {
    this->on_table_p = False;
    this->on_zero_page_p = False;
    TextFlow* flow = this;
    LayoutInstance* layout;
    for (;;)
    {
      FrameInstance* frame = flow->GetFirstFrame();
      if (frame == NULL)
      {
	// for nest table delete
	return;
      }
      layout = frame->GetLayout();
      Panel* panel = layout->GetPanel();
      if (panel == NULL)
      {
	break;
      }
      this->on_table_p = True;
      flow = panel->GetRightBP() ->GetTextFlow();
    }
    this->on_zero_page_p = layout->GetTemplate()->IsZeroPage();
    this->on_flags_is_initialized_p = True;
  }
}

bool
TextFlow::OnZeroPageP()
{
  if (!this->on_flags_is_initialized_p)
  {
    this->InitializeOnFlags();
  }
  return (bool)(this->on_zero_page_p != 0);
}

bool
TextFlow::OnTableP()
{
  if (!this->on_flags_is_initialized_p)
  {
    this->InitializeOnFlags();
  }
  return (bool)(this->on_table_p != 0);
}

// ------------------------------------------------------------
// LayoutInstance's methods

void
LayoutInstance::FixPosition(DocumentWindow* window)
{
  assert(window != NULL);
  
  Panel* panel = this->GetPanel();
  if (panel != NULL)
  {
    Lpoint lp[1];
    BP* bp = panel->GetOwnedRealBP();
    Line* line = bp->find_line_and_lpoint(lp, window);
    delete bp;
    if (line != NULL)
    {
      line->set_panel_position_by_lpoint(panel, lp);
    }
  }
  else if (this->GetTemplate() ->IsZeroPage())
  {
    LayoutInstance* mapped = window->GetZeroCaretLayout();
    if (mapped)
    {
      mapped->GetZeroPageLayout();
    }
  }
}

void
LayoutInstance::FixPosition()
{
  assert(Selected_Window != NULL);
  this->FixPosition(Selected_Window);
}

// ------------------------------------------------------------
// FrameInstance's methods

Iunit
FrameInstance::line_x_to_iunit(Iunit x)
{
  FrameTemplate* t = this->GetTemplate();
  return((this->GetFastTextFlow() ->tategaki_flow_p() ? t->GetCY()
	   : t->GetCX())
	  + x);
}

Iunit
FrameInstance::line_y_to_iunit(Iunit y)
{
  FrameTemplate* t = this->GetTemplate();
  return(this->GetFastTextFlow() ->tategaki_flow_p()
	  ? t->GetCX() + t->GetWidth() - y
	  : t->GetCY() + y);
}

Iunit
FrameInstance::ix_to_line_unit(Iunit ix)
{
  FrameTemplate* t = this->GetTemplate();
  return(this->GetFastTextFlow() ->tategaki_flow_p()
	  ? t->GetCX() + t->GetWidth() - ix
	  : ix - t->GetCX());
}

Iunit
FrameInstance::iy_to_line_unit(Iunit iy)
{
  return(iy - this->GetTemplate() ->GetCY());
}

void
FrameInstance::line_point_to_ipoint(Ipoint* ip)
{
  Iunit x = this->line_x_to_iunit(POINT_X(ip));
  Iunit y = this->line_y_to_iunit(POINT_Y(ip));
  if (this->GetFastTextFlow() ->tategaki_flow_p())
  {
    SET_POINT(ip, y, x);
  }
  else
  {
    SET_POINT(ip, x, y);
  }
}

void
FrameInstance::ipoint_to_line_point(Ipoint* ip)
{
  Iunit x = this->ix_to_line_unit(POINT_X(ip));
  Iunit y = this->iy_to_line_unit(POINT_Y(ip));
  if (this->GetFastTextFlow() ->tategaki_flow_p())
  {
    SET_POINT(ip, y, x);
  }
  else
  {
    SET_POINT(ip, x, y);
  }
}

bool
FrameInstance::EmptyLinesP()
{
  assert(this->lines != NULL);
  
  return(this->lines != NULL
	  && this->lines->GetNextImmediately() == this->lines);
}

// This method makes `this' has nil line only.
void
FrameInstance::EmptyLines()
{
  if (this->lines == NULL)
  {
    this->lines = Line::MakeInstance(this, NULL);
  }
  else
  {
    this->lines->ReportLast();
  }
}

int
FrameInstance::GetNumLines()
{
  int num_lines = 0;
  Line* line = this->lines;
  for (;;)
  {
    line = line->GetNextImmediately();
    if (line->NilP() || line->GetShadowP())
    {
      break;
    }
    ++num_lines;
  }
  return(num_lines);
}

Line* 
FrameInstance::GetNearestLine(Lunit ly)
{
  this->MaybeFormat();
  Lunit layout_ly = (this->GetLayout())->GetLY();
  Line* found_line = NULL;
  Lunit distance = MAX_LPOS;
  for (Line* line = this->lines; (line = line->GetNext()) != NULL;)
  {
    Lunit new_distance = ly - (layout_ly + line->GetBaseLineY());
    if (new_distance < 0)
    {
      new_distance = -new_distance;
    }
    if (new_distance < distance)
    {
      distance = new_distance;
      found_line = line;
    }
  }
  return(found_line);
}

Line* 
FrameInstance::GetFirstLine()
{
  this->MaybeFormat();
  return(this->lines->GetNext());
}

Line* 
FrameInstance::GetLastLine()
{
  this->MaybeFormat();
  Line* line = this->lines->GetNext();
  if (line != NULL)
  {
    for (;;)
    {
      Line* next_line = line->GetNext();
      if (next_line == NULL)
      {
	break;
      }
      line = next_line;
    }
  }
  return(line);
}

// This method deletes some members for formatter.  Destructor must call
// this method.
void
FrameInstance::DeleteForFormatter()
{
  if (this->lines != NULL)
  {
    this->EmptyLines();
    Line::KillInstance(this->lines);
    this->lines = NULL;
  }
  
  delete this->first_bp;
  delete this->next_bp;
  delete this->next_shadow_bp;
  
  this->first_bp = NULL;
  this->next_bp = NULL;
  this->next_shadow_bp = NULL;
}

#define FRAME_FLOW(frame) (frame->GetFirstBP() ->GetTextFlow())

// This method initialize buffer pointers, and lines.
void
FrameInstance::ReportTextFlow()
{
  this->DeleteForFormatter();
  this->EmptyLines();
  
  TextFlow* flow = this->GetTextFlow();
  
  this->first_bp = flow->GetBeginningBPCreate();
  this->next_bp = flow->GetBeginningBPCreate();
  this->next_shadow_bp = flow->GetBeginningBPCreate();
  this->SetFormatNeededP(True);
  this->virginp = True;
}

void
FrameInstance::PrepareToChangeFlow()
{
  this->SetFormatNeededP(True);
  FrameInstance* next = this->GetNextTextFrame();
  if (next != NULL)
  {
    next->ReportTextFlow();
  }
  LayoutTemplate* t = this->GetLayout() ->GetTemplate();
  if (t->IsZeroPage() || t->IsTable())
  {
    FrameInstance* prev = this->GetPrevTextFrame();
    if (prev)
    {
      prev->DelayFormat();
    }
  }
}

void
FrameInstance::SetFormatNeededP(bool format_needed_p)
{
  this->format_needed_p = format_needed_p;
  TextFlow* flow = FRAME_FLOW(this);
  flow->SetOverflowP(False);
  FrameInstance* prev = this->GetPrevTextFrame();
  
  if (format_needed_p)
  {
    flow->FormatNeeded();
    if (prev == NULL || !prev->maybe_format_needed_p)
    {
      flow->SetFormattedFrame(prev);
    }
  }
  else if (prev != NULL)
  {
    format_needed_p = (bool)(prev->maybe_format_needed_p != 0);
  }
  
  FrameInstance* frame = this;
  if (format_needed_p)
  {
    for (;;)
    {
      frame->maybe_format_needed_p = True;
      frame = frame->GetNextTextFrame();
      if (frame == NULL || frame->maybe_format_needed_p)
      {
	break;
      }
    }
  }
  else
  {
    for (;;)
    {
      frame->maybe_format_needed_p = False;
      prev = frame;
      {
	Line* line = frame->GetLines();
	for (;;)
	{
	  line = line->GetNextImmediately();
	  if (line->NilP())
	  {
	    break;
	  }
	  BP* line_first_bp = line->GetFirstBP();
	  if (line_first_bp->BeginningOfBufferP())
	  {
	    line_first_bp->GetParagraph() ->SetFirstFrame(frame);
	  }
	}
      }
      frame = frame->GetNextTextFrame();
      if (frame == NULL || frame->format_needed_p)
      {
	break;
      }
    }
    flow->SetFormattedFrame(prev);
    (this->layout_instance->GetDocumentPresentation())->CheckCaret();
  }
}

void
FrameInstance::DelayFormat()
{
  this->SetFormatNeededP(True);
  this->EmptyLines();
  this->SetVirginP(True);
}

void
FrameInstance::get_drawing_irect(Irect* result)
{
  Irect r[1];
  this->get_irect(r);
  int m;
  m = this->GetLeftMargin();
  if (0 < m)
  {
    RECT_TOP_X(r) += m;
  }
  m = this->GetTopMargin();
  if (0 < m)
  {
    RECT_TOP_Y(r) += m;
  }
  m = this->GetRightMargin();
  if (0 < m)
  {
    RECT_BOT_X(r) -= m;
  }
  m = this->GetBottomMargin();
  if (0 < m)
  {
    RECT_BOT_Y(r) -= m;
  }
  if (RECT_BOT_X(r) < RECT_TOP_X(r))
  {
    RECT_TOP_X(r) = RECT_BOT_X(r) = (RECT_TOP_X(r) + RECT_BOT_X(r)) / 2;
  }
  if (RECT_BOT_Y(r) < RECT_TOP_Y(r))
  {
    RECT_TOP_Y(r) = RECT_BOT_Y(r) = (RECT_TOP_Y(r) + RECT_BOT_Y(r)) / 2;
  }
  *result = *r;
}

TextFlow* 
FrameInstance::GetFastTextFlow()
{
  BP* first_bp = this->GetFirstBP();
  return(first_bp == NULL ? this->GetAssignedTextFlow()
	  : first_bp->GetTextFlow());
}

FrameInstance* 
FrameInstance::GetFirstFrame()
{
  return(this->GetFastTextFlow() ->GetFirstFrame());
}

// for zero page invalidation
FrameInstance* 
FrameInstance::GetTopLevelFirstFrame()
{
  FrameInstance* frame = this->GetFirstFrame();
  Panel* panel;
  while ((panel = frame->GetLayout() ->GetPanel()) != NULL)
  {
    frame = panel->GetRightBP() ->GetTextFlow() ->GetFirstFrame();
  }
  return(frame);
}

// ------------------------------------------------------------
// BP's methods

// return True, if `this' is formatted completely.
bool
BP::FormattedP()
{
  FrameInstance* frame = (this->GetTextFlow())->GetFormattedFrame();
  return(frame != NULL
	  && (frame->EndOfTextP() || LT(this, frame->GetNextBP())));
}

FrameInstance* 
BP::FindFrame()
{
  TextFlow* flow = this->GetTextFlow();
  
  FrameInstance* frame
  = (flow->OnZeroPageP() ? flow->GetFirstFrame()
     : this->FormattedP() ? (this->GetParagraph())->GetFirstFrame()
     : flow->GetMaybeBrokenFrame());
  
  for (; frame; frame = frame->EnsureNextTFI())
  {
    frame->MaybeFormat();
    if (frame->EndOfTextP() || LT(this, frame->GetNextBP()))
    {
      break;
    }
  }
  return(frame);
}

FrameInstance* 
BP::MaybeFindFrame()
{
  TextFlow* flow = this->GetTextFlow();
  
  FrameInstance* frame
  = (flow->OnZeroPageP() ? flow->GetFirstFrame()
     : this->FormattedP() ? (this->GetParagraph())->GetFirstFrame()
     : flow->GetMaybeBrokenFrame());
  
  if (frame == NULL)
  {
    frame = flow->GetFirstFrame();
  }
  
  for (;;)
  {
    frame->MaybeFormat();
    if (frame->EndOfTextP() || LT(this, frame->GetNextBP()))
    {
      break;
    }
    FrameInstance* next_frame = frame->EnsureNextTFI();
    if (next_frame == NULL)
    {
      break;
    }
    frame = next_frame;
  }
  return(frame);
}

LayoutInstance* 
BP::FindLayout()
{
  FrameInstance* frame = this->FindFrame();
  return(frame == NULL ? NULL : frame->GetLayout());
}

LayoutInstance* 
BP::MaybeFindLayout()
{
  return(this->MaybeFindFrame() ->GetLayout());
}

Line* 
BP::FindLine()
{
  FrameInstance* frame = this->FindFrame();
  if (frame == NULL)
  {
    return(NULL);
  }
  Line* line = frame->GetLines();
  for (;;)
  {
    Line* prev_line = line;
    line = line->GetNextImmediately();
    if (line->NilP()
	|| line->GetShadowP() || LT(this, line->GetFirstBP()))
    {
#ifndef NDEBUG
      if (prev_line->NilP() || prev_line->GetShadowP())
      {
	assert(!prev_line->NilP());
	assert(!prev_line->GetShadowP());
      }
#endif
      return(prev_line);
    }
  }
}

Line* 
BP::find_line_and_lpoint(Lpoint* lp, DocumentWindow* window)
{
  Line* line = this->FindLine();
  if (line != NULL)
  {
    LineCharInfo* info = line->GetInfo();
    {
      int num_chars = line->GetNumDrawingChars();
      int i = (this->GetLogicalOffset()
	       - line->GetFirstBP() ->GetLogicalOffset());
      int n;
      for (n = 0; n < num_chars; ++n)
      {
	if (i <= info[n].i)
	{
	  break;
	}
      }
      
      Ipoint ip[1];
      SET_POINT(ip, info[n].x, line->GetBaseLineY());
      FrameInstance* frame = line->GetFrame();
      frame->line_point_to_ipoint(ip);
      LayoutInstance* layout = frame->GetLayout();
      layout->FixPosition(window);
      layout->ipoint_to_lpoint(ip, lp);
    }
    line->ReleaseInfo(info);
  }
  return(line);
}

Line* 
BP::find_line_and_lpoint(Lpoint* lp)
{
  return(this->find_line_and_lpoint(lp, Selected_Window));
}

Line* 
BP::maybe_find_line_and_lpoint(Lpoint* lp, DocumentWindow* window)
{
  Line* line = this->find_line_and_lpoint(lp, window);
  if (line == NULL)
  {
    FrameInstance* frame = this->MaybeFindFrame();
    while (frame->EmptyLinesP())
    {
      frame = frame->GetPrevTextFrame();
    }
    line = frame->GetLastLine();
    Ipoint ip[1];
    SET_POINT(ip, line->GetBotX(), line->GetBaseLineY());
    frame->line_point_to_ipoint(ip);
    LayoutInstance* layout = frame->GetLayout();
    layout->FixPosition(window);
    layout->ipoint_to_lpoint(ip, lp);
  }
  return(line);
}

Line* 
BP::maybe_find_line_and_lpoint(Lpoint* lp)
{
  return(this->maybe_find_line_and_lpoint(lp, Selected_Window));
}

LayoutInstance* 
BP::FindTopLevelLayout()
{
  BP* bp = new BP(this);
  for (;;)
  {
    TextFlow* flow = bp->GetTextFlow();
    if (!flow->OnTableP())
    {
      break;
    }
    bp->Set(flow
	     ->GetFirstFrame() ->GetLayout() ->GetPanel() ->GetRightBP());
    bp->Decrement();
  }
  LayoutInstance* layout = bp->MaybeFindLayout();
  delete bp;
  return(layout);
}

static void
MoveInFrame(BP* bp,
	     FrameInstance* frame, Lunit lx, Lunit ly, int dir_x, int dir_y)
{
  Lpoint lp[1];
  SET_POINT(lp, lx, ly);
  Lrect r[1];
  frame->get_lrect(r);
  if (lrect_insidep(r, lp))
  {
    dir_x = -dir_x;
    dir_y = -dir_y;
  }
  
  Line* line = (dir_y < 0 ? frame->GetLastLine()
		: 0 < dir_y ? frame->GetFirstLine()
		: frame->GetNearestLine(ly));
  if (line == NULL)
  {
    bp->Set(frame->GetFirstBP());
  }
  else
  {
    if (dir_x == 0)
    {
      line->navigate_bp(lp, bp, NULL);
    }
    else
    {
      bp->Set(line->GetFirstBP());
      if (dir_x < 0)
      {
	bp->Add(line->GetNumChars());
	if (!bp->EndOfBufferP())
	{
	  bp->Decrement();
	}
      }
    }
  }
}

static bool NEAR
FrameMoveInLayout(BP* bp,
		   LayoutInstance* layout,
		   Lunit lx, Lunit ly, int dir_x, int dir_y)
{
  FrameInstance* found_frame = NULL;
  Lunit distance0 = MAX_LPOS;
  Lunit distance1 = MAX_LPOS;
  Lrect r[1];
  FrameInstance* frame;
  
  for (frame = layout->GetFirstFrame();
       frame != NULL; frame = frame->GetNextLFI())
  {
    frame->get_lrect(r);
    if ((dir_x < 0 && RECT_BOT_X(r) <= lx)
	|| (0 < dir_x && lx <= RECT_TOP_X(r))
	|| (dir_y < 0 && RECT_BOT_Y(r) <= ly)
	|| (0 < dir_y && ly <= RECT_TOP_Y(r)))
    {
      Lunit distance_x = (RECT_BOT_X(r) <= lx ? lx - RECT_BOT_X(r) + 1
			  : lx < RECT_TOP_X(r) ? RECT_TOP_X(r) - lx : 0);
      Lunit distance_y = (RECT_BOT_Y(r) <= ly ? ly - RECT_BOT_Y(r) + 1
			  : ly < RECT_TOP_Y(r) ? RECT_TOP_Y(r) - ly : 0);
      
      if (distance_x != 0 || distance_y != 0)
      {
	Lunit new_distance_0 = distance_x;
	Lunit new_distance_1 = distance_y;
	if (dir_x != 0)
	{
	  new_distance_0 = distance_y;
	  new_distance_1 = distance_x;
	}
	
	if (new_distance_0 < distance0
	    || (new_distance_0 == distance0 && new_distance_1 < distance1))
	{
	  distance0 = new_distance_0;
	  distance1 = new_distance_1;
	  found_frame = frame;
	}
      }
    }
  }
  
  if (found_frame != NULL)
  {
    MoveInFrame(bp, found_frame, lx, ly, dir_x, dir_y);
    return(True);
  }
  return(False);
}

// This function is called from BP::FrameMove.  Returns
// layout-instance, if `ch' is a panel, and it has one or more text-frame.
// Otherwise, returns NULL.
static LayoutInstance* 
GetSpecialLayout(DocumentContent* conte, sjis ch, Line* line, Iunit line_x)
{
  if (IsPanel(ch))
  {
    Panel* panel = conte->GetPanel(GetSpecialCode(ch));
    if (panel != NULL)
    {
      LayoutInstance* layout = panel->GetLayout();
      if (layout != NULL && layout->GetFirstFrame() != NULL)
      {
	line->set_panel_position_by_line_x(panel, line_x);
	return(layout);
      }
    }
  }
  return(NULL);
}

bool
BP::FrameMove0(Line* line, Lunit lx, Lunit ly, int dir_x, int dir_y)
{
  BP* old_this = new BP(this);
  FrameInstance* frame = line->GetFrame();
  LayoutInstance* layout = frame->GetLayout();
  Lunit layout_lx = layout->GetLX();
  
  // adjusting position into frame
  Lrect frame_lrect[1];
  frame->get_lrect(frame_lrect);
  SET_BOUND(lx, RECT_TOP_X(frame_lrect), RECT_BOT_X(frame_lrect) - 1);
  SET_BOUND(ly, RECT_TOP_Y(frame_lrect), RECT_BOT_Y(frame_lrect) - 1);
  
  DocumentContent* conte = this->GetTextFlow() ->GetDocContent();
  bool found = False;
  
  if (dir_x != 0)
  {
    dir_x = dir_x < 0 ? -1 : 1;
    dir_y = 0;
    if (line->HasSpecialP())
    {
      int logical_offset = this->GetLogicalOffset();
      LineCharInfo* info = line->GetInfo();
      int line_offset = line->GetFirstBP() ->GetLogicalOffset();
      int num_chars = line->GetNumDrawingChars();
      for (int n = 0 < dir_x ? 0 : num_chars - 1;
	   !found && 0 <= n && n < num_chars; n += dir_x)
      {
	sjis ch = info[n].ch;
	if (IsPanel(ch)
	    && ((0 < dir_x && logical_offset <= line_offset + info[n].i)
		|| (dir_x < 0 && line_offset + info[n].i < logical_offset)))
	{
	  LayoutInstance* new_layout = GetSpecialLayout(conte,
							 ch, line, info[n].x);
	  
	  if (new_layout != NULL)
	  {
	    found = FrameMoveInLayout(this,
				       new_layout, lx, ly, dir_x, dir_y);
	  }
	}
      }
      line->ReleaseInfo(info);
    }
  }
  else
  {
    dir_y = dir_y < 0 ? -1 : 1;
    while (!found)
    {
      line = dir_y < 0 ? line->GetPrev() : line->GetNext();
      if (line == NULL)
      {
	break;
      }
      if (line->HasSpecialP())
      {
	LineCharInfo* info = line->GetInfo();
	Iunit target_x = (int) (lx - layout_lx);
	int num_chars = line->GetNumDrawingChars();
	LayoutInstance* found_layout = NULL;
	Iunit found_distance = MAX_IUNITS;
	int n;
	for (n = 0; n < num_chars; ++n)
	{
	  sjis ch = info[n].ch;
	  if (IsPanel(ch))
	  {
	    LayoutInstance* new_layout = GetSpecialLayout(conte,
							   ch,
							   line, info[n].x);
	    if (new_layout != NULL)
	    {
	      Iunit x0 = info[n].x;
	      Iunit x1 = info[n + 1].x;
	      Iunit distance = (target_x < x0 ? x0 - target_x
				: x1 < target_x ? target_x - x1 : 0);
	      if (distance < found_distance)
	      {
		found_distance = distance;
		found_layout = new_layout;
		if (distance == 0)
		{
		  break;
		}
	      }
	    }
	  }
	}
	line->ReleaseInfo(info);
	if (found_layout != NULL)
	{
	  found = FrameMoveInLayout(this,
				     found_layout, lx, ly, dir_x, dir_y);
	}
      }
    }
  }
  
  if (!found)
  {
    found = FrameMoveInLayout(this, layout, lx, ly, dir_x, dir_y);
  }
  
  if (!found)
  {
    Panel* panel = layout->GetPanel();
    if (panel != NULL)
    {
      found = True;
      BP* panel_bp = panel->GetOwnedRealBP();
      if (dir_y != 0)
      {
	Lpoint panel_lp[1];
	Lunit old_lx = panel->GetX();
	line = panel_bp->find_line_and_lpoint(panel_lp);
	if (line != NULL)
	{
	  line->set_panel_position_by_lpoint(panel, panel_lp);
	  line = dir_y < 0 ? line->GetPrev() : line->GetNext();
	  if (line != NULL)
	  {
	    BP* bp = line->GetBPAtXCreate(lx + (POINT_X(panel_lp) - old_lx));
	    this->Set(bp);
	    bp->Set(line->GetFirstBP());
	    bp->Add(line->GetNumChars());
	    if (LE(bp, this))
	    {
	      this->XDecrement();
	    }
	    if (LT(this, line->GetFirstBP()))
	    {
	      this->XIncrement();
	    }
	    delete bp;
	  }
	}
      }
      else
      {
	this->Set(panel_bp);
	if (0 < dir_x)
	{
	  this->Increment();
	}
      }
      delete panel_bp;
    }
  }
  
  if (!found && dir_y < 0)
  {
    LayoutInstance* prev = layout->GetPrevLayout();
    if (prev != NULL)
    {
      found = FrameMoveInLayout(this, prev, lx, ly, dir_x, dir_y);
    }
  }
  
  if (!found && 0 < dir_y)
  {
    LayoutInstance* next = layout->GetNextLayout();
    if (next != NULL)
    {
      found = FrameMoveInLayout(this, next, lx, ly, dir_x, dir_y);
    }
  }
  
  if (!found)
  {
    MoveInFrame(this, frame, lx, ly, dir_x, dir_y);
  }
  
  found = !EQ(old_this, this);
  delete old_this;
  return(found);
}

bool
BP::FrameMove(int dir_x, int dir_y)
{
  Lpoint lp[1];
  Line* line = this->maybe_find_line_and_lpoint(lp);
  return(this->FrameMove0(line, POINT_X(lp), POINT_Y(lp), dir_x, dir_y));
}

// ------------------------------------------------------------
// Invalidate

void
DocumentWindow::Invalidate(Lrect* lrect)
{
  Lrect this_lrect[1];
  this->get_lrect(this_lrect);
  if (lrect_and(this_lrect, lrect))
  {
    RECT this_rect[1];
    this->lrect_to_drect(this_lrect, CastWindowsRECTToDrect(this_rect));
    if (PWordPresentation::GetEditModeStatus() == SetMargin_MODE)
    {
      ++this_rect->bottom;
    }
    this->InvalidateRect(this_rect);
  }
}

void
DocumentWindow::InvalidateZeroPage(FrameInstance* frame)
{
  LayoutInstance* save = LayoutInstance::GetZeroMappedLayout();
  LayoutInstance* zero = this->GetContent() ->GetZeroLayout();
  Lunit zero_lx = zero->GetLX();
  Lunit zero_ly = zero->GetLY();
  
  Lrect this_lrect[1];
  this->get_lrect(this_lrect);
  FrameInstance* first_frame = frame->GetTopLevelFirstFrame();
  for (LayoutInstance* layout = this->beginning_layout;
       layout != this->end_layout; layout = layout->GetNextLayout())
  {
    Lrect zero_lrect[1];
    layout->get_zero_page_lrect(zero_lrect);
    for (FrameInstance* fi = first_frame; fi; fi = fi->GetNextTextFrame())
    {
      Lrect lrect[1];
      fi->get_lrect(lrect);
      if (lrect_and(lrect, zero_lrect) && lrect_and(lrect, this_lrect))
      {
	RECT rect[1];
	this->lrect_to_drect(lrect, CastWindowsRECTToDrect(rect));
	this->InvalidateRect(rect);
      }
    }
  }
  
  zero->SetLX(zero_lx);
  zero->SetLY(zero_ly);
  LayoutInstance::SetZeroMappedLayout(save);
}

void
DocumentPresentation::Invalidate(Lrect* r)
{
  for (DocumentWindow* w = this->first_docwindow;
       w != NULL; w = w->GetNextDocWindow())
  {
    w->Invalidate(r);
  }
}

void
DocumentPresentation::InvalidateZeroPage(FrameInstance* frame)
{
  for (DocumentWindow* w = this->GetFirstDocWindow();
       w != NULL; w = w->GetNextDocWindow())
  {
    w->InvalidateZeroPage(frame);
  }
}

void
LayoutInstance::Invalidate()
{
  bool has_rim_p = !this->IsTable();
  
  Lrect layout_lrect[1];
  this->get_zero_page_lrect(layout_lrect);
  
  for (DocumentWindow* w = (this->GetDocumentPresentation()
			    ->GetFirstDocWindow());
       w; w = w->GetNextDocWindow())
  {
    RECT w_rect[1];
    ::GetClientRect(w->GetHandle(), w_rect);
    
    long top_x = w->LongDunitX(RECT_TOP_X(layout_lrect));
    long bot_x = w->LongDunitX(RECT_BOT_X(layout_lrect));
    long top_y = w->LongDunitY(RECT_TOP_Y(layout_lrect));
    long bot_y = w->LongDunitY(RECT_BOT_Y(layout_lrect));
    if (has_rim_p)
    {
      bot_x += 2;
      top_y -= 1;
      bot_y += 2;
    }
    
    if (top_x < w_rect->right
	&&  0 < bot_x && top_y < w_rect->bottom && 0 < bot_y)
    {
      RECT r[1];
      r->left = (int) top_x;
      r->right = (int) bot_x;
      r->top = (int) top_y;
      r->bottom = (int) bot_y;
      w->InvalidateRect(r);
    }
  }
}

// invalidate with Irect
void
FrameInstance::Invalidate(Irect* r)
{
  LayoutInstance* layout = this->GetLayout();
  DocumentPresentation* pre = layout->GetDocumentPresentation();
  
  if (this->GetFastTextFlow() ->OnZeroPageP())
  {
    pre->InvalidateZeroPage(this);
  }
  else
  {
    Irect pure_r[1];
    this->get_irect(pure_r);
    if (irect_and(pure_r, r))
    {
      Lrect lrect[1];
      Lrect layout_lrect[1];
      layout->get_lrect(layout_lrect);
      layout->irect_to_lrect(pure_r, lrect);
      if (lrect_and(lrect, layout_lrect))
      {
	pre->Invalidate(lrect);
      }
    }
  }
}

// invalidate whole frame
void
FrameInstance::Invalidate()
{
  Irect r[1];
  this->get_irect(r);
  this->Invalidate(r);
}

// ------------------------------------------------------------
// DelayFormat

void
TextFlow::DelayFormat(BP* bp00, BP* bp11)
{
  assert(bp00->GetTextFlow() == this);
  assert(bp11->GetTextFlow() == this);
  
  this->SetOverflowP(False);
  this->FormatNeeded();
  
  // adjust BP's order
  if (LT(bp11, bp00))
  {
    BP* tmp = bp00;
    bp00 = bp11;
    bp11 = tmp;
  }
  
  BP bp0(bp00);
  BP bp1(bp11);
  bp0.XDecrement();
  bp1.XIncrement();
  
  // working buffer pointer
  BP bp(&bp0);
  
  FrameInstance* frame = this->formatted_frame;
  if (frame == NULL)
  {
    frame = this->GetFirstFrame();
  }
  else if (LT(&bp0, frame->GetFirstBP()))
  {
    frame = (bp0.GetParagraph())->GetFirstFrame();
  }
  
  for (;;)
  {
#if (!defined(NDEBUG))
    if (frame == NULL)
    {
      syserr("DelayFormat : frame == NULL");
    }
#endif
    if (LE(&bp0, frame->GetNextShadowBP()))
    {
      FrameInstance* prev = frame->GetPrevTextFrame();
      if (prev != NULL)
      {
	frame = prev;
	continue;
      }
    }
    break;
  }
  
  FrameInstance* last_frame = frame;
  for (; frame != NULL && LT(frame->GetNextShadowBP(), &bp0);
       frame = frame->GetNextTextFrame())
  {
    last_frame = frame;
  }
  
  if (frame == NULL
      && last_frame != NULL && (this->OnZeroPageP() || this->OnTableP()))
  {
    // set overflowp flag on text-flow
    last_frame->EnsureNextTFI();
  }
  
  for (; frame != NULL && LE(frame->GetFirstBP(), &bp1);
       frame = frame->GetNextTextFrame())
  {
    frame->SetFormatNeededP(True);
    if (frame->VirginP())
    {
      // skip
    }
    else if (LE(&bp0, frame->GetFirstBP()) && LE(frame->GetNextBP(), &bp1))
    {
      if (!frame->EmptyLinesP())
      {
	frame->EmptyLines();
	frame->SetVirginP(True);
      }
    }
    else
    {
      Line* line = frame->GetLines();
      while (!(line = line->GetNextImmediately())->NilP()
	     && LE(line->GetFirstBP(), &bp1))
      {
	bp.Set(line->GetFirstBP());
	bp.Add(line->GetNumShadowChars());
	if (LE(&bp0, &bp))
	{
	  line->SetFormatNeededP(True);
	  bp.Set(line->GetFirstBP());
	  bp.Add(line->GetNumChars());
	  if (LE(&bp0, &bp))
	  {
	    line->SetForceRedisplayP(True);
	  }
	}
      }
    }
  }
}

void
TextFlow::DelayFormat()
{
  for (FrameInstance* frame = this->GetFirstFrame();
       frame != NULL; frame = frame->GetNextTextFrame())
  {
    frame->DelayFormat();
  }
}

// ------------------------------------------------------------
// formatting

// private method
void NEAR
FrameInstance::Format1(bool do_invalidate)
{
  BP* first_bp = this->first_bp;
  BP* next_bp = this->next_bp;
  BP* next_shadow_bp = this->next_shadow_bp;
  bool end_of_text_p = False;
  this->has_page_no_p = False;
  
  if (this->virginp)
  {
    this->virginp = False;
    if (do_invalidate)
    {
      do_invalidate = False;
      this->Invalidate();
    }
  }
  
  FrameInstance* prev_frame = this->GetPrevTextFrame();
  if (prev_frame == NULL)
  {
    // THIS is a first frame on text-flow.
    first_bp->BeginningOfText();
  }
  else
  {
    first_bp->Set(prev_frame->next_bp);
    end_of_text_p = (bool)(prev_frame->end_of_text_p != 0);
  }
  this->prev_end_of_text_p = end_of_text_p;
  
  Paragraph* first_para = first_bp->GetParagraph();
  int* first_atts = get_atts(first_para->GetAtts());
  LayoutInstance* layout = this->GetLayout();
  bool zero_or_table_p = (layout->GetTemplate() ->IsZeroPage()
			  || layout->GetTemplate() ->IsTable());
  
  if (end_of_text_p
      || (first_atts[PA_placement] == PL_page
	  && !zero_or_table_p
	  && first_bp->BeginningOfBufferP()
	  && prev_frame != NULL && prev_frame->GetLayout() == layout))
  {
    next_bp->Set(first_bp);
    next_shadow_bp->Set(first_bp);
    if (!this->EmptyLinesP())
    {
      this->EmptyLines();
      if (do_invalidate)
      {
	this->Invalidate();
      }
    }
  }
  else
  {
    struct YY
    {
      Iunit top_y;
      Iunit bot_y;
      Buffer* buffer;
      int logical_offset;
      int num_chars;
      int invalidp;
    };
    
    int tategaki_flow_p = first_bp->GetTextFlow() ->tategaki_flow_p();
    int index_x = tategaki_flow_p ? INDEX_Y : INDEX_X;
    int index_y = REVERSED_INDEX(index_x);
    Iunit old_top = 0;
    Iunit old_bot = 0;
    int old_num_yy = 0;
    YY* old_yy = NULL;
    Irect this_ir[1];
    this->get_irect(this_ir);
    if (do_invalidate)
    {
      old_bot = (RECT_UNIT(this_ir, INDEX_BOT, index_y)
		 - RECT_UNIT(this_ir, INDEX_TOP, index_y));
      old_num_yy = this->GetNumLines();
      if (old_num_yy != 0)
      {
	old_yy = new YY[old_num_yy];
	YY* yy = old_yy;
	Line* line = this->lines;
	for (int n = 0; n != old_num_yy; ++n, ++yy)
	{
	  line = line->GetNextImmediately();
	  yy->top_y = line->GetY();
	  yy->bot_y = yy->top_y + line->GetHeight();
	  yy->top_y -= line->GetTopBorder();
	  yy->bot_y += line->GetBotBorder();
	  {
	    BP* bp = line->GetFirstBP();
	    yy->buffer = bp->GetEditor();
	    yy->logical_offset = bp->GetLogicalOffset();
	  }
	  yy->num_chars = line->GetNumChars();
	  yy->invalidp = line->GetForceRedisplayP();
	}
	yy = &old_yy[0];
	old_top = yy->top_y - (yy->bot_y - yy->top_y) / 2;
	yy = &old_yy[old_num_yy - 1];
	old_bot = yy->bot_y + (yy->bot_y - yy->top_y) / 2;
      }
    }
    
    Irect drawing_ir[1];
    this->get_drawing_irect(drawing_ir);
    
    Iunit y = 0;
    Iunit drawing_height = (RECT_UNIT(drawing_ir, INDEX_BOT, index_y)
			    - RECT_UNIT(drawing_ir, INDEX_TOP, index_y));
    BP* bp = new BP(first_bp);
    Line* line = this->lines;
    int last_border = 0;
    
    for (bool firstp = True;; firstp = False)
    {
      int* atts = get_atts(bp->GetParagraph() ->GetAtts());
      Iunit leading = 0;
      if (!firstp)
      {
	if (drawing_height < y)
	{
	  break;
	}
	leading = atts[PA_linespace];
	if (bp->BeginningOfBufferP())
	{
	  leading += atts[PA_parspace];
	  enum Placement place = (enum Placement) atts[PA_placement];
	  if (place == PL_frame)
	  {
	    if (!zero_or_table_p || this->GetNextTextFrame())
	    {
	      break;
	    }
	  }
	  else if (place == PL_page && !zero_or_table_p)
	  {
	    break;
	  }
	}
	y += leading;
      }
      
      line = line->GetNextImmediately();
      for (;;)
      {
	Line* old_line = line;
	if (line->NilP() || LT(bp, line->GetFirstBP()))
	{
	  line = Line::MakeInstance(this, bp);
	  line->LinkToPrev(old_line);
	  break;
	}
	if (EQ(line->GetFirstBP(), bp))
	{
	  break;
	}
	line = line->GetNextImmediately();
	Line::KillInstance(old_line);
      }
      if (line->GetFormatNeededP() || line->HasPageNoP())
      {
	line->Format(TOP_X(drawing_ir) - TOP_X(this_ir),
		      BOT_X(drawing_ir) - TOP_X(this_ir));
      }
      line->SetY(leading);
      y += line->GetHeight();
      line->SetBorderChangedP(False);
      line->SetTopBorder(0);
      line->SetBotBorder(0);
      int border = atts[PA_lines];
      if (border != last_border)
      {
	line->SetBorderChangedP(True);
	last_border = border;
	if (border != 0)
	{
	  BorderLine* b = BorderLine::Get(border);
	  int align = b->GetAlignment();
	  if ((align & (BorderLine::AlignAbove | BorderLine::AlignBelow))
	      != 0)
	  {
	    Iunit width = b->strict_width(index_horizontal) + Point_to_iu(2);
	    if ((align & BorderLine::AlignAbove) != 0)
	    {
	      line->SetTopBorder(width);
	    }
	    if ((align & BorderLine::AlignBelow) != 0)
	    {
	      line->SetBotBorder(width);
	    }
	  }
	}
      }
      
      bp->Add(line->GetNumChars());
      if (line->EndOfBufferP() && !bp->XIncrement())
      {
	break;
      }
    }
    next_shadow_bp->Set(bp);
    line->ReportLast();
    
    enum VerticalAlignment va = this->GetTemplate() ->frame_align();
    if (!vertical_alignment_valid_p(va))
    {
      va = (enum VerticalAlignment) first_atts[PA_vertical_alignment];
      if (!vertical_alignment_valid_p(va))
      {
	va = AL_vertical_adjust;
      }
    }
    line = this->lines;
    y = 0;
    int num_lines = 0;
    for (;;)
    {
      line = line->GetNextImmediately();
      if (line->NilP())
      {
	if (va == AL_vertical_adjust)
	{
	  va = AL_vertical_up;
	}
	break;
      }
      // add leading
      y += line->GetY();
      if (va == AL_vertical_adjust && drawing_height < y && 3 <= num_lines)
      {
	line = line->GetPrevImmediately();
	--num_lines;
	break;
      }
      y += line->GetHeight() + line->GetTopBorder() + line->GetBotBorder();
      if (num_lines != 0 && drawing_height < y)
      {
	break;
      }
      ++num_lines;
    }
    assert(0 < num_lines);
    
    Line* last_line = line->GetPrevImmediately();
    Paragraph* last_para = last_line->GetFirstBP() ->GetParagraph();
    if (last_para != first_para
	&& !last_line->EndOfBufferP()
	&& GetAttValue(last_para->GetAtts(), PA_nobreak))
    {
      if (va == AL_vertical_adjust)
      {
	va = AL_vertical_up;
      }
      for (;;)
      {
	--num_lines;
	assert(0 < num_lines);
	last_line = last_line->GetPrevImmediately();
	if (last_line->GetFirstBP() ->GetParagraph() != last_para)
	{
	  break;
	}
      }
    }
    
    next_bp->Set(last_line->GetFirstBP());
    next_bp->Add(last_line->GetNumChars());
    
    if (last_line->EndOfBufferP() && !next_bp->XIncrement())
    {
      end_of_text_p = True;
    }
    if (num_lines == 1 && va == AL_vertical_adjust)
    {
      va = AL_vertical_up;
    }
    
    line = this->lines;
    Iunit total_height = 0;
    Iunit bot_border = 0;
    for (;;)
    {
      Line* prev_line = line;
      line = line->GetNextImmediately();
      line->SetShadowP(False);
      total_height += (line->GetY()
		       + line->GetHeight()
		       + line->GetTopBorder() + line->GetBotBorder());
      if (line->HasPageNoP())
      {
	this->has_page_no_p = True;
      }
      if (line->GetBorderChangedP())
      {
	prev_line->SetBotBorder(bot_border);
	bot_border = line->GetBotBorder();
	line->SetBotBorder(0);
      }
      if (line == last_line)
      {
	line->SetBotBorder(bot_border);
	break;
      }
    }
    for (;;)
    {
      line = line->GetNextImmediately();
      if (line->NilP())
      {
	break;
      }
      line->SetShadowP(True);
    }
    
    y = (tategaki_flow_p
	 ? RECT_BOT_X(this_ir) - RECT_BOT_X(drawing_ir)
	 : RECT_TOP_Y(drawing_ir) - RECT_TOP_Y(this_ir));
    Iunit adjust = drawing_height - total_height;
    if (va == AL_vertical_adjust)
    {
      int n;
      assert(2 <= num_lines);
      assert(!(last_line->GetNextImmediately())->NilP());
      total_height = 0;
      line = this->lines;
      Line* next_line = line->GetNextImmediately();
      for (n = 0; n != num_lines; ++n)
      {
	line = next_line;
	next_line = next_line->GetNextImmediately();
	Iunit height = (line->GetHeight()
			+ line->GetTopBorder()
			+ line->GetBotBorder() + next_line->GetY());
	total_height += height;
	line->SetY(height);
      }
      line = this->lines;
      adjust = drawing_height - total_height;
      for (n = 0; n != num_lines; ++n)
      {
	line = line->GetNextImmediately();
	Iunit height = line->GetY();
	Iunit foo = 0;
	if (0 < adjust)
	{
	  foo = (Iunit) (((long) adjust * height + ((total_height - 1) >> 1))
			 / total_height);
	  total_height -= height;
	  adjust -= foo;
	}
	line->SetY(y + foo + line->GetTopBorder());
	y += foo + height;
      }
    }
    else
    {
      if (0 < adjust)
      {
	switch (va)
	{
	 case AL_vertical_center:
	  y += adjust / 2;
	  break;
	  
	 case AL_vertical_down:
	  y += adjust;
	  break;
	}
      }
      line = this->lines;
      for (int n = 0; n != num_lines; ++n)
      {
	line = line->GetNextImmediately();
	y += line->GetY() + line->GetTopBorder();
	line->SetY(y);
	y += line->GetHeight() + line->GetBotBorder();
      }
    }
    
    delete bp;
    
    if (do_invalidate)
    {
      if (this->HasPageNoP() && (first_bp->GetTextFlow())->OnZeroPageP())
      {
	this->Invalidate();
      }
      else
      {
	int n;
	int o;
	assert(0 < num_lines);
	YY* new_yy = new YY[num_lines];
	YY* yy = new_yy;
	line = this->lines;
	for (n = 0; n != num_lines; ++n, ++yy)
	{
	  line = line->GetNextImmediately();
	  yy->top_y = line->GetY();
	  yy->bot_y = yy->top_y + line->GetHeight();
	  yy->top_y -= line->GetTopBorder();
	  yy->bot_y += line->GetBotBorder();
	  {
	    BP* bp = line->GetFirstBP();
	    yy->buffer = bp->GetEditor();
	    yy->logical_offset = bp->GetLogicalOffset();
	  }
	  yy->num_chars = line->GetNumChars();
	  yy->invalidp = line->GetForceRedisplayP();
	  line->SetForceRedisplayP(False);
	}
	
	Iunit top;
	Iunit bot;
	yy = &new_yy[0];
	top = yy->top_y - (yy->bot_y - yy->top_y) / 2;
	yy = &new_yy[num_lines - 1];
	bot = yy->bot_y + (yy->bot_y - yy->top_y) / 2;
	SET_MIN(top, old_top);
	SET_MAX(bot, old_bot);
	
	n = 0;
	o = 0;
	YY* nyy = new_yy;
	YY* oyy = old_yy;
	bool invalidp = False;
	Irect ir[1];
	*ir = *this_ir;
	for (;;)
	{
	  if (n == num_lines || o == old_num_yy)
	  {
	    if (n != num_lines || o != old_num_yy)
	    {
	      invalidp = True;
	    }
	    break;
	  }
	  if (nyy->top_y != oyy->top_y)
	  {
	    invalidp = True;
	    if (nyy->top_y < oyy->top_y)
	    {
	      ++n;
	      ++nyy;
	    }
	    else
	    {
	      ++o;
	      ++oyy;
	    }
	  }
	  else if (nyy->bot_y != oyy->bot_y
		   || nyy->invalidp
		   || oyy->invalidp
		   || nyy->num_chars != oyy->num_chars
		   || nyy->logical_offset != oyy->logical_offset
		   || nyy->buffer != oyy->buffer)
	  {
	    invalidp = True;
	    ++n;
	    ++nyy;
	    ++o;
	    ++oyy;
	  }
	  else
	  {
	    Iunit next_top = (nyy->top_y + nyy->bot_y) / 2;
	    if (invalidp)
	    {
	      invalidp = False;
	      if (top < next_top)
	      {
		if (tategaki_flow_p)
		{
		  RECT_TOP_X(ir) = RECT_BOT_X(this_ir) - next_top;
		  RECT_BOT_X(ir) = RECT_BOT_X(this_ir) - top;
		}
		else
		{
		  RECT_TOP_Y(ir) = RECT_TOP_Y(this_ir) + top;
		  RECT_BOT_Y(ir) = RECT_TOP_Y(this_ir) + next_top;
		}
		this->Invalidate(ir);
	      }
	    }
	    top = next_top;
	    ++n;
	    ++nyy;
	    ++o;
	    ++oyy;
	  }
	}
	if (invalidp && top < bot)
	{
	  if (tategaki_flow_p)
	  {
	    RECT_TOP_X(ir) = RECT_BOT_X(this_ir) - bot;
	    RECT_BOT_X(ir) = RECT_BOT_X(this_ir) - top;
	  }
	  else
	  {
	    RECT_TOP_Y(ir) = RECT_TOP_Y(this_ir) + top;
	    RECT_BOT_Y(ir) = RECT_TOP_Y(this_ir) + bot;
	  }
	  this->Invalidate(ir);
	}
      }
      
      delete old_yy;
    }
  }
  
  this->end_of_text_p = end_of_text_p;
}

// private method
void NEAR
FrameInstance::Format0(bool do_invalidate)
{
  assert(this->first_bp != NULL);
#if (0)
  // I think, following code is not needed. (KONNO_NOT_NEEDED)
  if (Selected_Window != NULL
      && (this->layout_instance->GetDocumentPresentation()
	  == Selected_Window->GetPresentation()))
  {
    Selected_Window->RequestCaret();
  }
#endif
  LayoutInstance* old_top_level = current_top_level_layout;
  if (!(this->first_bp->GetTextFlow())->OnZeroPageP())
  {
    LayoutInstance* layout = this->GetLayout();
    for (;;)
    {
      Panel* panel = layout->GetPanel();
      if (panel == NULL)
      {
	break;
      }
      BP* panel_bp = panel->GetRightBP();
      if (!panel_bp->FormattedP())
      {
	break;
      }
      panel_bp = new BP(panel_bp);
      panel_bp->Decrement();
      layout = panel_bp->FindLayout();
      delete panel_bp;
    }
    current_top_level_layout = layout;
  }
  
  this->Format1(do_invalidate);
  current_top_level_layout = old_top_level;
  
  if (do_invalidate)
  {
    FrameInstance* next_frame = this->GetNextTextFrame();
    if (next_frame != NULL && !next_frame->format_needed_p)
    {
      BP* next_first_bp = next_frame->first_bp;
      if (this->end_of_text_p != next_frame->prev_end_of_text_p
	  || next_first_bp == NULL || !EQ(next_first_bp, this->next_bp))
      {
	next_frame->SetFormatNeededP(True);
      }
    }
  }
  this->SetFormatNeededP(False);
}

void
FrameInstance::Format(bool do_invalidate)
{
  this->Format0(do_invalidate);
  
  bool regularp = (this->GetTemplate() ->GetLayoutTemplate() ->GetKind()
		   == LayoutTemplate::RegularTemplate);
  
  if (this->EndOfTextP())
  {
    if (regularp)
    {
      this->ReportLast();
    }
    FrameInstance* frame = this;
    while ((frame = frame->GetNextTextFrame()) != NULL)
    {
      if (frame->FormatNeededP() || !frame->EmptyLinesP())
      {
	frame->Format0(True);
      }
    }
    if (regularp)
    {
      this->ReportLast();
    }
  }
  else if (!regularp && this->GetNextTextFrame() == NULL)
  {
    this->GetFirstBP() ->GetTextFlow() ->SetOverflowP(True);
  }
}

void
FrameInstance::MaybeFormat()
{
  if (this->FormatNeededP())
  {
    this->Format(True);
  }
  else if (this->HasPageNoP())
  {
    LayoutTemplate* layout_t = (this->GetLayout())->GetTemplate();
    if (layout_t->IsZeroPage() || layout_t->IsTable())
    {
      this->Format(False);
    }
  }
}

// private method
void NEAR
TextFlow::Format0(bool one_step_p)
{
  if (this->OnZeroPageP())
  {
    current_top_level_layout = this->GetDocContent() ->GetFirstLayout();
  }
  FrameInstance* frame;
  for (frame = this->GetMaybeBrokenFrame();
       frame; frame = frame->EnsureNextTFI())
  {
    if (frame->FormatNeededP())
    {
      frame->Format(True);
      if (one_step_p)
      {
	break;
      }
    }
    if (frame->EndOfTextP())
    {
      break;
    }
  }
  if (frame == NULL || frame->EndOfTextP())
  {
    this->format_needed_p = False;
    if (frame == NULL)
    {
      this
      ->GetDocContent()
      ->GetDocument() ->GetDocumentPresentation() ->CheckCaret();
    }
  }
}

void
TextFlow::Format()
{
  this->Format0(False);
}

void
TextFlow::FormatOneStep()
{
  this->Format0(True);
}

// ------------------------------------------------------------
// redisplay

static uword NEAR
get_eol_flags(Line* line)
{
  BP* line_first_bp = line->GetFirstBP();
  EditText* editor = line_first_bp->GetEditor();
  return(line->EndOfBufferP() ? editor->GetEOBFlags()
	  : editor->GetFlags(line_first_bp->GetLogicalOffset()
			      + line->GetNumChars() - 1));
}

static uword NEAR
get_bol_flags(Line* line)
{
  uword flags = 0;
  BP* line_first_bp = line->GetFirstBP();
  EditText* editor = line_first_bp->GetEditor();
  if (line_first_bp->BeginningOfBufferP())
  {
    Paragraph* para = editor->GetPrev();
    if (para)
    {
      flags = para->GetEOBFlags();
    }
  }
  else
  {
    flags = editor->GetFlags(line_first_bp->GetLogicalOffset() - 1);
  }
  return(flags);
}

static void NEAR
fill_rect(HDC dc, Drect* dr, HBRUSH brush)
{
  FillRect(dc, CastDrectToWindowsRECT(dr), brush);
}

static DWORD NEAR
set_text_color(HDC dc, DWORD color)
{
  return(SetTextColor(dc, color));
}

void
FrameInstance::Redisplay(View* view, HRGN region)
{
  extern bool painting_selected_window;
  Dpoint dp[1];
  Ipoint ip[1];

  this->virginp = False;
  LayoutInstance* layout = this->GetLayout();
  
  // Get geometry of THIS.
  Irect this_ir[1];
  Drect this_dr[1];
  Lrect this_lr[1];
  this->get_irect(this_ir);
  layout->irect_to_lrect(this_ir, this_lr);
  view->lrect_to_drect(this_lr, this_dr);
  
  // Return if drawing area is empty.
  if (RECT_IS_EMPTY_P(this_dr))
  {
    return;
  }
  
  // Make sure formatting is done.
  this->MaybeFormat();
  
  // DRAW_DR has rectangle of current drawing line.
  Drect draw_dr[1];
  *draw_dr = *this_dr;
  
  // CLIP_DR is used for fast clipping check.
  Drect clip_dr[1];
  GetRgnBox(region, CastDrectToWindowsRECT(clip_dr));
  
  // LAYOUT_XY keep layout's left top position.
  Lpoint layout_xy[1];
  SET_POINT(layout_xy, layout->GetLX(), layout->GetLY());
  
  int tategaki_flow_p = this->GetFastTextFlow() ->tategaki_flow_p();
  int index_x = tategaki_flow_p ? INDEX_Y : INDEX_X;
  int index_y = REVERSED_INDEX(index_x);

  Lunit base_ly = TOP_Y(this_lr);
  
  int device = view->GetDeviceType();
  HDC dc = view->GetDC();

  // If STUPID_TEXT_OUT_P is nonzero, we call ExtTextOut per one character.
  int stupid_text_out_p = False;
  if (device == PRINTER)
  {
    stupid_text_out_p = !Pref_fast_print;
  }

  int stupid_gdi_font_p = False;
  if (!windows_31_or_later_p() && tategaki_flow_p)
  {
    // Modern, Roman, and Script fonts in Windows 3.0 have bug in tategaki
    // ExtTextOut.
    stupid_gdi_font_p = True;
  }
  
  // SHOW_SELECTION_P indicates if selection exists then draw it !
  bool show_selection_p = (device != PRINTER && painting_selected_window);
  if (show_selection_p)
  {
    HBRUSH brush = 0;
    if (this->primary_background)
    {
      brush = Screen::primary_background_brush;
    }
    else if (this->secondary_background)
    {
      brush = Screen::secondary_background_brush;
    }
    if (brush)
    {
      show_selection_p = False;
      fill_rect(dc, this_dr, brush);
    }
  }
  
  DWORD foreground = BLACK;
  DWORD old_foreground = set_text_color(dc, BLACK);
  HFONT font = 0;
  HFONT old_font = (HFONT)::SelectObject(dc, GetStockObject(SYSTEM_FONT));
  int old_bk_mode = SetBkMode(dc, TRANSPARENT);
  
  // There are synchronized buffers.
  int buffer_size = 0;
  char* chars = NULL;
  int* gaps = NULL;
  int* device_x = NULL;
  uword* flags_buf = NULL;
  
  // when `base_drawp[N + 1]' is nonzero, we must draw line N.
  // `base_drawp[0]' is a dummy.
#ifdef HAS_BOOL
  bool* base_drawp;
#else /* HAS_BOOL */
  char* base_drawp;
#endif /* HAS_BOOL */
  {
    int num_lines = this->GetNumLines();
    base_drawp = new bool[num_lines + 2];
    memset(base_drawp, 0, sizeof(bool) * (num_lines + 2));
  }
  
  // We set up array BASE_DRAWP in this loop.
#ifdef HAS_BOOL
  bool* drawp = base_drawp;
#else /* HAS_BOOL */
  char* drawp = base_drawp;
#endif /* HAS_BOOL */
  Line* line = this->lines;
  for (;;)
  {
    ++drawp;
    line = line->GetNextImmediately();
    if (line->NilP() || line->GetShadowP())
    {
      break;
    }
    
    // Set pseudo top-y.
    POINT_Y(ip) = -(POINT_X(ip) = line->GetY());
    TOP_Y(draw_dr) = view->lpos_to_dpos(base_ly + POINT_UNIT(ip, index_x),
					  index_y);
    
    // Set pseudo bot-y.
    POINT_Y(ip) = -(POINT_X(ip) = line->GetY() + line->GetHeight());
    BOT_Y(draw_dr) = view->lpos_to_dpos(base_ly + POINT_UNIT(ip, index_x),
					  index_y);

    if (drect_intersectp(clip_dr, draw_dr)
	&& RectInRegion(region, CastDrectToWindowsRECT(draw_dr)))
    {
      drawp[-1] = drawp[0] = drawp[1] = True;
    }
  }
  
  FontMetric* fm = NULL;
  drawp = base_drawp;
  line = this->lines;
  for (;;)
  {
    ++drawp;
    line = line->GetNextImmediately();
    if (line->NilP() || line->GetShadowP())
    {
      break;
    }
    if (!drawp[0])
    {
      continue;
    }
    
    // Set pseudo top-y.
    POINT_Y(ip) = -(POINT_X(ip) = line->GetY());
    (RECT_UNIT(draw_dr, index_top, index_y)
     = view->lpos_to_dpos(base_ly + POINT_UNIT(ip, index_x), index_y));
    
    // Set pseudo bot-y.
    POINT_Y(ip) = -(POINT_X(ip) = line->GetY() + line->GetHeight());
    (RECT_UNIT(draw_dr, index_bot, index_y)
     = view->lpos_to_dpos(base_ly + POINT_UNIT(ip, index_x), index_y));
    
    if (RECT_IS_EMPTY_P(draw_dr))
    {
      continue;
    }
    
    Drect fill_dr[1];
    *fill_dr = *draw_dr;
    
    BP* line_first_bp = line->GetFirstBP();
    EditText* editor = line_first_bp->GetEditor();
    int line_offset = line_first_bp->GetLogicalOffset();
    int num_drawing_chars = line->GetNumDrawingChars();
    
    int request_buffer_size = (num_drawing_chars + 1) * 2;
    if (buffer_size < request_buffer_size)
    {
      if (buffer_size != 0)
      {
	delete chars;
	delete gaps;
	delete device_x;
	delete flags_buf;
      }
      buffer_size = MAX(request_buffer_size, 200);
      chars = new char[buffer_size];
      gaps = new int[buffer_size];
      device_x = new int[buffer_size];
      flags_buf = new uword[buffer_size];
      if (!show_selection_p)
      {
	memset(flags_buf, 0, sizeof(flags_buf[0]) * buffer_size);
      }
    }
    
    LineCharInfo* info = line->GetInfo();
    int n;

    {
      Lunit base_lx = RECT_UNIT(this_lr, INDEX_TOP, index_x);
      for (n = 0; n <= num_drawing_chars; ++n)
      {
	device_x[n] = view->lpos_to_dpos(base_lx + info[n].x, index_x);
      }
    }
    
    if (show_selection_p)
    {
      for (n = 0; n != num_drawing_chars; ++n)
      {
	flags_buf[n] = editor->GetFlags(line_offset + info[n].i);
      }
    }
    
    for (n = 0; n != num_drawing_chars;)
    {
      int top_n = n;
      int flags = flags_buf[n];
      int inverted = (int) (info[n].ch == EOB ? 0 : info[n].atts[CA_inverted]);
      while (++n != num_drawing_chars
	     && flags == (int) flags_buf[n]
	     && (inverted
		 == (int) (info[n].ch == EOB ? 0 : info[n].atts[CA_inverted])))
      {
      }
      DWORD color = ((flags & IV_MASK) ? PRIMARY_BACKGROUND
		     : (flags & IV_SUBMASK) ? SECONDARY_BACKGROUND
		     : inverted ? BLACK : WHITE);
      if (color != WHITE)
      {
	RECT_UNIT(fill_dr, INDEX_TOP, index_x) = device_x[top_n];
	RECT_UNIT(fill_dr, INDEX_BOT, index_x) = device_x[n];
	if (RECT_UNIT(clip_dr, INDEX_BOT, index_x) <= device_x[top_n])
	{
	  break;
	}
	if (foreground != color)
	{
	  set_text_color(dc, foreground = color);
	}
	fill_rect(dc, fill_dr, Screen::black_brush);
      }
    }
    if (show_selection_p)
    {
      {
	int flags = get_bol_flags(line);
	if (flags)
	{
	  DWORD color = ((flags & IV_MASK) ? PRIMARY_BACKGROUND
			 : (flags & IV_SUBMASK) ? SECONDARY_BACKGROUND
			 : WHITE);
	  if (color != WHITE)
	  {
	    if (foreground != color)
	    {
	      set_text_color(dc, foreground = color);
	    }
	    (RECT_UNIT(fill_dr, INDEX_TOP, index_x)
	     = RECT_UNIT(this_dr, INDEX_TOP, index_x));
	    
	    RECT_UNIT(fill_dr, INDEX_BOT, index_x) = device_x[0];
	    fill_rect(dc, fill_dr, Screen::black_brush);
	  }
	}
      }
      {
	int flags = get_eol_flags(line);
	if (flags)
	{
	  DWORD color = ((flags & IV_MASK) ? PRIMARY_BACKGROUND
			 : (flags & IV_SUBMASK) ? SECONDARY_BACKGROUND
			 : WHITE);
	  if (color != WHITE)
	  {
	    if (foreground != color)
	    {
	      set_text_color(dc, foreground = color);
	    }
	    (RECT_UNIT(fill_dr, INDEX_TOP, index_x)
	     = device_x[num_drawing_chars]);
	    
	    (RECT_UNIT(fill_dr, INDEX_BOT, index_x)
	     = RECT_UNIT(this_dr, INDEX_BOT, index_x));
	    
	    fill_rect(dc, fill_dr, Screen::black_brush);
	  }
	}
      }
    }
    
    n = 0;
    while (n != num_drawing_chars)
    {
      sjis ch = info[n].ch;
      if (ch == EOB)
      {
	break;
      }
      int top_n = n;
      if (RECT_UNIT(clip_dr, INDEX_BOT, index_x) <= device_x[top_n])
      {
	break;
      }
      uword flags = flags_buf[n];
      attr attrib = info[n].attrib;
      int* atts = info[n].atts;
      if (IsPanel(ch))
      {
	Panel* panel = (FRAME_FLOW(this)
			->GetDocContent() ->GetPanel(GetSpecialCode(ch)));
	if (panel && panel->GetPanelID() == Panel::TablePanelID)
	{
	  line->set_panel_position_by_line_x(panel, info[top_n].x);
	  LayoutInstance* panel_layout = panel->GetLayout();
	  assert(panel_layout != NULL);
	  Lrect panel_lr[1];
	  Drect panel_dr[1];
	  panel_layout->get_lrect(panel_lr);
	  view->lrect_to_drect(panel_lr, panel_dr);
	  HRGN panel_region = (CreateRectRgnIndirect
			       (CastDrectToWindowsRECT(panel_dr)));
	  if (::CombineRgn(panel_region, panel_region, region, RGN_AND)
	      != NULLREGION)
	  {
	    ::SelectObject(dc, panel_region);
	    if (foreground != BLACK)
	    {
	      set_text_color(dc, foreground = BLACK);
	    }
	    panel_layout
	    ->Redisplay(view,
			 panel_region,
			 ((flags & IV_MASK) ? PRIMARY_BACKGROUND
			  : (flags & IV_SUBMASK) ? SECONDARY_BACKGROUND
			  : BLACK));
	    ::SelectObject(dc, region);
	  }
	  ::DeleteObject(panel_region);
	}
	++n;
      }
      else
      {
	// normal character
	int kanjip = USE_KANJI_FONT(ch);
	{
	  FontMetric* new_fm = get_font_metric(attrib,
						kanjip, tategaki_flow_p);
	  if (fm != new_fm)
	  {
	    LOGFONT logfont[1];
	    fm = new_fm;
	    fm->SimulatedLogfont(logfont, view);
	    if (tategaki_flow_p)
	    {
	      logfont->lfEscapement = 270 * 10;
	      logfont->lfOrientation = 270 * 10;
	    }
	    HFONT new_font = CreateFontIndirect(logfont);
	    SelectObject(dc, new_font);
	    if (font)
	    {
	      DeleteObject(font);
	    }
	    font = new_font;
	  }
	}
	TEXTMETRIC tm[1];
	GetTextMetrics(dc, tm);
	
	int inverted = atts[CA_inverted];
	{
	  DWORD new_foreground = inverted ? WHITE : BLACK;
	  if (foreground != new_foreground)
	  {
	    set_text_color(dc, foreground = new_foreground);
	  }
	}
	int subscript = atts[CA_subscript];
	
	int count = 0;
	for (;;)
	{
	  if (IsKanji(ch))
	  {
	    chars[count] = ch >> 8;
	    gaps[count] = 0;
	    ++count;
	  }
	  chars[count] = (char)ch;
	  gaps[count] = device_x[n + 1] - device_x[n];
	  ++count;
	  ++n;
	  if (stupid_text_out_p
	      || (stupid_gdi_font_p && tm->tmCharSet == ANSI_CHARSET))
	  {
	    break;
	  }
	  if (n == num_drawing_chars)
	  {
	    break;
	  }
	  ch = info[n].ch;
	  if (ch == EOB || IsPanel(ch))
	  {
	    break;
	  }
	  attr new_attrib = info[n].attrib;
	  int new_kanjip = USE_KANJI_FONT(ch);
	  if ((attrib != new_attrib || kanjip != new_kanjip)
	      && fm != get_font_metric(new_attrib,
					new_kanjip, tategaki_flow_p))
	  {
	    break;
	  }
	  if (attrib != new_attrib)
	  {
	    int* new_atts = info[n].atts;
	    if (inverted != new_atts[CA_inverted]
		|| subscript != new_atts[CA_subscript])
	    {
	      break;
	    }
	  }
	}
	gaps[count - 1] -= info[n - 1].overhang;
	POINT_Y(ip) = -(POINT_X(ip) = (line->GetBaseLineY()
					 - ((fm->GetAscent() * subscript)
					    / 3)));
	POINT_Y(dp) = -(POINT_X(dp) = tm->tmAscent);
	POINT_UNIT(dp, index_y) += (view
				     ->lpos_to_dpos((base_ly
						      + POINT_UNIT(ip,
								    index_x))
						     , index_y));
	POINT_UNIT(dp, index_x) = device_x[top_n];
	ExtTextOut(dc,
		    POINT_X(dp),
		    POINT_Y(dp),
		    0, NULL, chars, count, stupid_text_out_p ? NULL : gaps);
      }
    }
    
    if (line->HasBorderP())
    {
      n = 0;
      while (n != num_drawing_chars)
      {
	if (info[n].ch == EOB)
	{
	  break;
	}
	if (info[n].atts[CA_lines] == 0)
	{
	  ++n;
	}
	else
	{
	  Drect dr[1];
	  RECT_UNIT(dr, INDEX_TOP, index_x) = device_x[n];
	  int border = info[n].atts[CA_lines];
	  while (++n != num_drawing_chars
		 && info[n].ch != EOB && info[n].atts[CA_lines] == border)
	  {
	  }
	  BorderLine* b = BorderLine::Get(border);
	  int overlinep = b->overlinep();
	  int underlinep = b->underlinep();
	  if (overlinep | underlinep)
	  {
	    RECT_UNIT(dr, INDEX_BOT, index_x) = device_x[n];
	    if (RECT_UNIT(dr, INDEX_TOP, index_x)
		< RECT_UNIT(dr, INDEX_BOT, index_x))
	    {
	      if (foreground != BLACK)
	      {
		set_text_color(dc, foreground = BLACK);
	      }
	      int strict_width = b->strict_pixel_width(view,
							index_horizontal);
	      int width = (RECT_UNIT(draw_dr, INDEX_BOT, index_y)
			   - RECT_UNIT(draw_dr, INDEX_TOP, index_y));
	      SET_MIN(width, strict_width);
	      SET_POINT(dp, width, -width);
	      if (overlinep)
	      {
		(RECT_UNIT(dr, index_bot, index_y)
		 = ((RECT_UNIT(dr, index_top, index_y)
		     = RECT_UNIT(draw_dr, index_top, index_y))
		    + POINT_UNIT(dp, index_x)));
		b->draw_border_in_drect(view, dr, index_horizontal);
	      }
	      if (underlinep)
	      {
		(RECT_UNIT(dr, index_top, index_y)
		 = ((RECT_UNIT(dr, index_bot, index_y)
		     = RECT_UNIT(draw_dr, index_bot, index_y))
		    - POINT_UNIT(dp, index_x)));
		b->draw_border_in_drect(view, dr, index_horizontal);
	      }
	    }
	  }
	}
      }
    }
    line->ReleaseInfo(info);
  }
  
  if (buffer_size != 0)
  {
    delete chars;
    delete gaps;
    delete device_x;
    delete flags_buf;
  }
  delete base_drawp;
  
  // drawing paragraph borders
  {
    if (foreground != BLACK)
    {
      set_text_color(dc, foreground = BLACK);
    }
    Irect drawing_irect[1];
    this->get_drawing_irect(drawing_irect);
    Iunit drawing_top = TOP_X(drawing_irect);
    Iunit drawing_bot = BOT_X(drawing_irect);
    
    Paragraph* last_para = NULL;
    attr last_attrib = 0;
    int last_border = 0;
    line = this->lines;
    Line* prev_line = NULL;
    Iunit border_top = MAX_IUNITS;
    Iunit border_bot = 0;
    Irect border_ir[1];
    Lrect border_lr[1];
    
    for (;;)
    {
      prev_line = line;
      line = line->GetNextImmediately();
      if (line->NilP() || line->GetShadowP())
      {
	break;
      }
      
      Paragraph* para = line->GetFirstBP() ->GetParagraph();
      if (last_para != para)
      {
	last_para = para;
	attr attrib = para->GetAtts();
	if (last_attrib != attrib)
	{
	  last_attrib = attrib;
	  Iunit new_first_top;
	  Iunit new_top;
	  Iunit new_bot;
	  line->calc_margin(drawing_top,
			     drawing_bot, &new_first_top, &new_top, &new_bot);
	  
	  int border = GetAttValue(attrib, PA_lines);
	  if (last_border == border)
	  {
	    SET_MIN(border_top, new_first_top);
	    SET_MIN(border_top, new_top);
	    SET_MAX(border_bot, new_bot);
	  }
	  else
	  {
	    if (last_border != 0)
	    {
	      TOP_X(border_ir) = border_top;
	      BOT_X(border_ir) = border_bot;
	      (BOT_Y(border_ir)
	       = this->line_y_to_iunit(prev_line->GetY()
					+ prev_line->GetHeight()
					+ prev_line->GetBotBorder()));
	      layout->irect_to_lrect(border_ir, border_lr);
	      (BorderLine::Get(last_border)
	       ->draw_para_border(view, border_lr, index_x));
	    }
	    last_border = border;
	    border_top = MIN(new_first_top, new_top);
	    border_bot = new_bot;
	    TOP_Y(border_ir) = (this
				 ->line_y_to_iunit(line->GetY()
						    - line->GetTopBorder()));
	  }
	}
      }
    }
    if (last_border != 0)
    {
      TOP_X(border_ir) = border_top;
      BOT_X(border_ir) = border_bot;
      BOT_Y(border_ir) = this->line_y_to_iunit(prev_line->GetY()
						 + prev_line->GetHeight()
						 + prev_line->GetBotBorder());
      layout->irect_to_lrect(border_ir, border_lr);
      BorderLine::Get(last_border) ->draw_para_border(view,
							border_lr, index_x);
    }
  }
  
  if (foreground != old_foreground)
  {
    set_text_color(dc, old_foreground);
  }
  if (TRANSPARENT != old_bk_mode)
  {
    ::SetBkMode(dc, old_bk_mode);
  }
  SelectObject(dc, old_font);
  if (font)
  {
    DeleteObject(font);
  }
}

void
FrameInstance::RedisplayFrameMargin(View* view)
{
  LayoutInstance* layout = this->GetLayout();
  
  Lrect this_lrect[1];
  RECT this_rect[1];
  this->get_lrect(this_lrect);
  view->lrect_to_drect(this_lrect, CastWindowsRECTToDrect(this_rect));
  
  Irect irect[1];
  Lrect lrect[1];
  RECT rect[1];
  this->get_drawing_irect(irect);
  layout->irect_to_lrect(irect, lrect);
  view->lrect_to_drect(lrect, CastWindowsRECTToDrect(rect));
  
  HDC dc = view->GetDC();
  DWORD old_color = set_text_color(dc, BLUE);
  HBRUSH brush;
  RECT r[1];
  
  brush = Screen::simple_dotted_brush[INDEX_VERTICAL];
  r->top = this_rect->top;
  r->bottom = this_rect->bottom;
  r->left = rect->left - 1;
  r->right = rect->left + 1;
  fill_rect(dc, CastWindowsRECTToDrect(r), brush);
  r->left = rect->right - 1;
  r->right = rect->right + 1;
  fill_rect(dc, CastWindowsRECTToDrect(r), brush);
  
  brush = Screen::simple_dotted_brush[INDEX_HORIZONTAL];
  r->left = this_rect->left;
  r->right = this_rect->right;
  r->top = rect->top - 1;
  r->bottom = rect->top + 1;
  fill_rect(dc, CastWindowsRECTToDrect(r), brush);
  r->top = rect->bottom - 1;
  r->bottom = rect->bottom + 1;
  fill_rect(dc, CastWindowsRECTToDrect(r), brush);
  
  if (old_color != BLUE)
  {
    set_text_color(dc, old_color);
  }
}

static void NEAR
draw_para_hmargin(HDC dc, Dunit x0, Dunit x1, Dunit y, int index_x)
{
  if (x0 != x1)
  {
    Drect dr[1];
    if (x0 < x1)
    {
      RECT_UNIT(dr, INDEX_TOP, index_x) = x0;
      RECT_UNIT(dr, INDEX_BOT, index_x) = x1;
    }
    else
    {
      RECT_UNIT(dr, INDEX_TOP, index_x) = x1;
      RECT_UNIT(dr, INDEX_BOT, index_x) = x0;
    }
    int index_y = REVERSED_INDEX(index_x);
    RECT_UNIT(dr, INDEX_TOP, index_y) = y;
    RECT_UNIT(dr, INDEX_BOT, index_y) = y + 1;
    fill_rect(dc, dr, Screen::one_dot_brush);
  }
}

static void NEAR
draw_para_vmargin(HDC dc, Dunit x, Dunit y0, Dunit y1, int index_x)
{
  if (y0 != y1)
  {
    Drect dr[1];
    RECT_UNIT(dr, INDEX_TOP, index_x) = x - 1;
    RECT_UNIT(dr, INDEX_BOT, index_x) = x + 1;
    int index_y = REVERSED_INDEX(index_x);
    if (y0 < y1)
    {
      RECT_UNIT(dr, INDEX_TOP, index_y) = y0;
      RECT_UNIT(dr, INDEX_BOT, index_y) = y1;
    }
    else
    {
      RECT_UNIT(dr, INDEX_TOP, index_y) = y1;
      RECT_UNIT(dr, INDEX_BOT, index_y) = y0;
    }
    fill_rect(dc, dr, Screen::black_brush);
  }
}

// display paragraph margin
void
FrameInstance::RedisplayParagraphMargin(View* view)
{
  Irect drawing_ir[1];
  HDC dc = view->GetDC();
  DWORD old_color = set_text_color(dc, RED);
  int index_x = this->GetFastTextFlow() ->get_index_x();
  int index_y = REVERSED_INDEX(index_x);
  this->get_drawing_irect(drawing_ir);
  Iunit line_top = RECT_UNIT(drawing_ir, INDEX_TOP, index_x);
  Iunit line_bot = RECT_UNIT(drawing_ir, INDEX_BOT, index_x);
  LayoutInstance* layout = this->GetLayout();
  
  Line* line = this->GetLines() ->GetNext();
  while (line != NULL)
  {
    Ipoint ip0[1];
    Ipoint ip1[1];
    Ipoint ip2[1];
    
    Iunit x0;
    Iunit x1;
    Iunit x2;
    line->calc_margin(line_top, line_bot, &x0, &x1, &x2);
    BP* first_bp = line->GetFirstBP();
    if (!first_bp->BeginningOfBufferP())
    {
      x0 = x1;
    }
    POINT_UNIT(ip0, index_x) = x0;
    POINT_UNIT(ip1, index_x) = x1;
    POINT_UNIT(ip2, index_x) = x2;
    
    Iunit y0;
    Iunit y1;
    Iunit y2;
    Paragraph* para = first_bp->GetParagraph();
    y0 = line->GetY();
    y1 = y0 + line->GetHeight();
    y0 -= line->GetTopBorder();
    Line* last_line = line;
    while ((line = line->GetNext()) != NULL
	   && line->GetFirstBP() ->GetParagraph() == para)
    {
      last_line = line;
    }
    y2 = (last_line->GetY()
		+ last_line->GetHeight() + last_line->GetBotBorder());
    POINT_UNIT(ip0, index_y) = this->line_y_to_iunit(y0);
    POINT_UNIT(ip1, index_y) = this->line_y_to_iunit(y1);
    POINT_UNIT(ip2, index_y) = this->line_y_to_iunit(y2);

    Lpoint lp0[1];
    Lpoint lp1[1];
    Lpoint lp2[1];
    layout->ipoint_to_lpoint(ip0, lp0);
    layout->ipoint_to_lpoint(ip1, lp1);
    layout->ipoint_to_lpoint(ip2, lp2);
    
    Dpoint dp0[1];
    Dpoint dp1[1];
    Dpoint dp2[1];
    view->lpoint_to_dpoint(lp0, dp0);
    view->lpoint_to_dpoint(lp1, dp1);
    view->lpoint_to_dpoint(lp2, dp2);
    
    draw_para_vmargin(dc,
		       POINT_UNIT(dp0, index_x),
		       POINT_UNIT(dp0, index_y),
		       POINT_UNIT(dp1, index_y), index_x);
    draw_para_vmargin(dc,
		       POINT_UNIT(dp1, index_x),
		       POINT_UNIT(dp1, index_y),
		       POINT_UNIT(dp2, index_y), index_x);
    draw_para_vmargin(dc,
		       POINT_UNIT(dp2, index_x),
		       POINT_UNIT(dp0, index_y),
		       POINT_UNIT(dp2, index_y), index_x);
    draw_para_hmargin(dc,
		       POINT_UNIT(dp0, index_x),
		       POINT_UNIT(dp2, index_x),
		       POINT_UNIT(dp0, index_y), index_x);
    draw_para_hmargin(dc,
		       POINT_UNIT(dp0, index_x),
		       POINT_UNIT(dp1, index_x),
		       POINT_UNIT(dp1, index_y), index_x);
    draw_para_hmargin(dc,
		       POINT_UNIT(dp1, index_x),
		       POINT_UNIT(dp2, index_x),
		       POINT_UNIT(dp2, index_y), index_x);
  }
  if (old_color != RED)
  {
    set_text_color(dc, old_color);
  }
}

// ------------------------------------------------------------
// mouse navigation

LayoutInstance* 
DocumentWindow::navigate_layout1(Lpoint* lp)
{
  current_top_level_layout = NULL;
  for (LayoutInstance* layout = this->beginning_layout;
       layout != this->end_layout; layout = layout->GetNextLayout())
  {
    LayoutInstance* zero = layout->GetZeroPageLayout();
    Lrect lrect[1];
    zero->get_lrect(lrect);
    if (lrect_insidep(lrect, lp))
    {
      current_top_level_layout = layout;
      layout->get_lrect(lrect);
      return(lrect_insidep(lrect, lp) ? layout : zero);
    }
  }
  return(NULL);
}

FrameInstance* 
LayoutInstance::navigate_frame1(Lpoint* lp)
{
  if (this != NULL)
  {
    Lrect this_lr[1];
    this->get_lrect(this_lr);
    if (lrect_insidep(this_lr, lp))
    {
      Ipoint ip[1];
      this->lpoint_to_ipoint(lp, ip);
      for (FrameInstance* frame = this->GetFirstFrame();
	   frame != NULL; frame = frame->GetNextLFI())
      {
	Irect frame_ir[1];
	frame->get_irect(frame_ir);
	if (irect_insidep(frame_ir, ip))
	{
	  return(frame);
	}
      }
    }
  }
  return(NULL);
}

Line* 
FrameInstance::navigate_line1(Lpoint* lp)
{
  if (this != NULL)
  {
    Lrect this_lr[1];
    this->get_lrect(this_lr);
    if (lrect_insidep(this_lr, lp))
    {
      this->MaybeFormat();
      Ipoint ip[1];
      this->GetLayout() ->lpoint_to_ipoint(lp, ip);
      Iunit y = (this->GetFastTextFlow() ->tategaki_flow_p()
		 ? this->ix_to_line_unit(POINT_X(ip))
		 : this->iy_to_line_unit(POINT_Y(ip)));
      Line* line = this->GetLines();
      Iunit last_y = 0;
      for (;;)
      {
	Line* last_line = line;
	line = line->GetNext();
	if (line == NULL)
	{
	  break;
	}
	Iunit line_top_y = line->GetY();
	Iunit line_bot_y = (line_top_y
			    + line->GetHeight() + line->GetBotBorder());
	if (y < line_bot_y)
	{
	  line_top_y -= line->GetTopBorder();
	  if (line_top_y <= y)
	  {
	    return(line);
	  }
	  if (!last_line->NilP())
	  {
	    return(y < (last_y + line_top_y) / 2 ? last_line : line);
	  }
	  break;
	}
	last_y = line_bot_y;
      }
    }
  }
  return(NULL);
}

LayoutInstance* 
DocumentWindow::navigate_layout(Lpoint* lp)
{
  LayoutInstance* layout = this->navigate_layout1(lp);
  for (;;)
  {
    LayoutInstance* panel_layout;
    (layout
     ->navigate_frame1(lp)
     ->navigate_line1(lp) ->navigate_bp(lp, NULL, &panel_layout));
    if (panel_layout == NULL)
    {
      break;
    }
    layout = panel_layout;
  }
  return(layout);
}

Line* 
DocumentWindow::navigate_line(Lpoint* lp)
{
  LayoutInstance* layout = this->navigate_layout1(lp);
  for (;;)
  {
    Line* line = layout->navigate_frame1(lp) ->navigate_line1(lp);
    line->navigate_bp(lp, NULL, &layout);
    if (layout == NULL || layout->GetFirstFrame() == NULL)
    {
      return(line);
    }
  }
}

static BP* NEAR
create_nearest_bp(FrameInstance* frame, Line* line, Lpoint* lp)
{
  BP* bp = new BP(frame->GetNextBP());
  if (line != NULL)
  {
    line->navigate_bp(lp, bp, NULL);
  }
  else
  {
    int topp = False;
    Line* first_line = frame->GetLines() ->GetNext();
    if (first_line != NULL)
    {
      Ipoint line_ip[1];
      Lpoint line_lp[1];
      (POINT_X(line_ip)
       = POINT_Y(line_ip)
       = frame->line_y_to_iunit(first_line->GetBaseLineY()));
      frame->GetLayout() ->ipoint_to_lpoint(line_ip, line_lp);
      if (frame->GetFastTextFlow() ->tategaki_flow_p())
      {
	if (POINT_X(line_lp) < POINT_X(lp))
	{
	  topp = True;
	}
      }
      else
      {
	if (POINT_Y(lp) < POINT_Y(line_lp))
	{
	  topp = True;
	}
      }
    }
    if (topp)
    {
      bp->Set(frame->GetFirstBP());
    }
    else if (!frame->EndOfTextP())
    {
      bp->XDecrement();
    }
  }
  return(bp);
}

// FRAME_RETURN is a pointer of "FrameInstance *" or a NULL.
BP* 
DocumentWindow::navigate_bp(Lpoint* lp,
			     BP* advice_bp, FrameInstance** frame_return)
{
  if (frame_return != NULL)
  {
    *frame_return = NULL;
  }
  TextFlow* flow = advice_bp == NULL ? NULL : advice_bp->GetTextFlow();
  LayoutInstance* layout = this->navigate_layout1(lp);
  while (layout != NULL)
  {
    FrameInstance* frame = layout->navigate_frame1(lp);
    layout = NULL;
    if (frame != NULL)
    {
      Line* line = frame->navigate_line1(lp);
      line->navigate_bp(lp, NULL, &layout);
      if (frame->GetTextFlow() == flow
	  || (flow == NULL
	      && (layout == NULL || layout->GetFirstFrame() == NULL)))
      {
	if (frame_return != NULL)
	{
	  *frame_return = frame;
	}
	return(create_nearest_bp(frame, line, lp));
      }
    }
  }
  
  if (advice_bp != NULL)
  {
    Lrect frame_lr[1];
    Lpoint effective_lp[1];
    FrameInstance* frame = advice_bp->MaybeFindFrame();
    frame->get_lrect(frame_lr);
    *effective_lp = *lp;
    if (frame->GetFastTextFlow() ->tategaki_flow_p())
    {
      SET_BOUND(POINT_Y(effective_lp),
		 RECT_TOP_Y(frame_lr), RECT_BOT_Y(frame_lr) - 1);
    }
    else
    {
      SET_BOUND(POINT_X(effective_lp),
		 RECT_TOP_X(frame_lr), RECT_BOT_X(frame_lr) - 1);
    }
    return(create_nearest_bp(frame,
			       frame->navigate_line1(effective_lp), lp));
  }
  
  return(NULL);
}

void
DocumentWindow::Invalidate(BP* bp00, BP* bp11)
{
  if (!EQ(bp00, bp11))
  {
    // adjust BP order
    if (LT(bp11, bp00))
    {
      BP* tmp = bp00;
      bp00 = bp11;
      bp11 = tmp;
    }
    
    BP* bp0 = new BP(bp00);
    BP* bp1 = new BP(bp11);
    bp0->XDecrement();
    bp1->XIncrement();
    
    TextFlow* flow = bp0->GetTextFlow();
    if (flow->OnZeroPageP())
    {
      this->InvalidateZeroPage(flow->GetFirstFrame());
    }
    else if (flow->OnTableP())
    {
      LayoutInstance* layout = flow->GetFirstFrame() ->GetLayout();
      layout->FixPosition();
      for (Line* line = bp0->FindLine();
	   line && !line->NilP() && LE(line->GetFirstBP(), bp1);
	   line = line->XGetNext())
      {
	Lrect frame_lrect[1];
	Irect irect[1];
	Lrect lrect[1];
	FrameInstance* frame = line->GetFrame();
	frame->get_irect(irect);
	layout->irect_to_lrect(irect, frame_lrect);
	Iunit line_y = line->GetY();
	Iunit line_height = line->GetHeight();
	int index_x = flow->get_index_x();
	int index_y = REVERSED_INDEX(index_x);
	(RECT_UNIT(irect, index_top, index_y)
	 = frame->line_y_to_iunit(line_y - line_height / 2));
	(RECT_UNIT(irect, index_bot, index_y)
	 = frame->line_y_to_iunit(line_y + line_height * 3 / 2));
	layout->irect_to_lrect(irect, lrect);
	if (lrect_and(lrect, frame_lrect))
	{
	  this->Invalidate(lrect);
	}
      }
    }
    else
    {
      for (LayoutInstance* layout = this->beginning_layout;
	   layout != this->end_layout; layout = layout->GetNextLayout())
      {
	for (FrameInstance* frame = layout->GetFirstFrame();
	     frame != NULL; frame = frame->GetNextLFI())
	{
	  if (frame->GetTextFlow() == flow)
	  {
	    frame->MaybeFormat();
	    if (LE(frame->GetFirstBP(), bp1)
		&& LT(bp0, frame->GetNextBP()))
	    {
	      Line* top_line = NULL;
	      Line* line = frame->GetLines();
	      while ((line = line->GetNext()) != NULL)
	      {
		if (LE(line->GetFirstBP(), bp0))
		{
		  top_line = line;
		}
		else
		{
		  break;
		}
	      }
	      if (top_line)
	      {
		line = top_line;
	      }
	      if (line)
	      {
		Line* bot_line;
		for (;;)
		{
		  bot_line = line;
		  if ((line = line->GetNext()) == NULL
		      || LT(bp1, line->GetFirstBP()))
		  {
		    break;
		  }
		}
		Lrect frame_lrect[1];
		Lrect lrect[1];
		Irect irect[1];
		frame->get_irect(irect);
		layout->irect_to_lrect(irect, frame_lrect);
		int index_x = flow->get_index_x();
		int index_y = REVERSED_INDEX(index_x);
		if (top_line)
		{
		  (RECT_UNIT(irect, index_top, index_y)
		   = frame->line_y_to_iunit(top_line->GetY()
					     - top_line->GetHeight() / 2));
		}
		(RECT_UNIT(irect, index_bot, index_y)
		 = frame->line_y_to_iunit(bot_line->GetY()
					   + bot_line->GetHeight() * 3 / 2));
		layout->irect_to_lrect(irect, lrect);
		if (lrect_and(lrect, frame_lrect))
		{
		  this->Invalidate(lrect);
		}
	      }
	    }
	  }
	}
      }
    }
    delete bp0;
    delete bp1;
  }
}
