// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pagesize.cpp,v 3.3 1999/05/12 00:22:16 kudou Exp $
// PageSize class
// pTCYǗNX

#include "pword.h"
#include "pmenus.h"
#include "tflist.h"
#include "margin.h"
#include "fission.h"
#include "pagesize.h"
#include "blueprin.h"
#include "layoutte.h"
#include "layoutin.h"
#include "allayout.h"
#include "objectid.h"
#include "docconte.h"
#include "pref.h"
#include "pstreams.h"
#include "foiterat.h"
#include "view.h"
#include "textflow.h"
#include "xstr.h"

// ------------------------------------------------------------
// static Paper size infomations

// next macro converts inch * 100 to Iunit
#define Iu(x) ((Iunit) (((double) (x) * IUNITS_PER_INCH_F) / 100.0))

//
// paper size table data
//
static Iunit paper_size [][2] = 
{
  { Iu(1169), Iu(1652) }, // A3
  { Iu(826),  Iu(1169) }, // A4
  { Iu(583),  Iu(826)  }, // A5
  { Iu(1012), Iu(1433) }, // B4
  { Iu(716),  Iu(1012) }, // B5
  { Iu(400),  Iu(600)  }, // PostCard
  { Iu(850),  Iu(1100) }, // Letter
  { Iu(826),  Iu(1169) }, // LetterSmall
  { Iu(1100), Iu(1700) }, // Tabloid
  { Iu(1700), Iu(1100) }, // Ledger
  { Iu(850),  Iu(1400) }, // Legal
  { Iu(550),  Iu(850)  }, // Statement
  { Iu(725),  Iu(1050) }, // Executive
  { Iu(850),  Iu(1300) }, // Folio
  { Iu(846),  Iu(1083) }, // Quarto
  { Iu(1000), Iu(1400) }, // P10x14
  { Iu(1100), Iu(1700) }, // P11x17
  { Iu(850),  Iu(1100) }, // Note
  { Iu(388),  Iu(888)  }, // Env9
  { Iu(413),  Iu(950)  }, // Env10
  { Iu(450),  Iu(1038) }, // Env11
  { Iu(475),  Iu(1100) }, // Env12
  { Iu(500),  Iu(1150) }, // Env14
  { Iu(950),  Iu(1200) }, // LetterExtra
  { Iu(950),  Iu(1500) }, // LegalExtra
  { Iu(1169), Iu(1800) }, // TabloidExtra
  { Iu(927),  Iu(1269) }, // A4Extra
  { 0,        0        }, // UserSize
};

//
// paper max and min size
//
const Iunit max_paper_size = Iu(3000);		// 30 Inch
const Iunit min_paper_size = Iu(197);		// <5 cm>
const Iunit min_regular_size = Point_to_iu(DEFAULT_POINT);

//
// default paper info
//
PageSize::Kind STATIC_NEAR PageSize::default_kind = PageSize::A4;
PageSize::Orientation STATIC_NEAR PageSize::default_orientation = PageSize::Portrait;

Vector STATIC_NEAR PageSize::def_usr_size;
Margin STATIC_NEAR PageSize::def_usr_margin;
Margin STATIC_NEAR PageSize::min_margin;
Iunit STATIC_NEAR PageSize::margin_div;
Iunit STATIC_NEAR PageSize::header_size;
Iunit STATIC_NEAR PageSize::footer_size;

Enum_Desc paper_size_desc[] =
{
  { S_paper_size_A3,		(int) PageSize::A3, },
  { S_paper_size_A4,		(int) PageSize::A4, },
  { S_paper_size_A5,		(int) PageSize::A5, },
  { S_paper_size_B4,		(int) PageSize::B4, },
  { S_paper_size_B5,		(int) PageSize::B5, },
  { S_paper_size_PostCard,	(int) PageSize::PostCard, },
  { S_paper_size_Letter,	(int) PageSize::Letter, },
  { S_paper_size_LetterSmall,	(int) PageSize::LetterSmall, },
  { S_paper_size_Tabloid,	(int) PageSize::Tabloid, },
  { S_paper_size_Ledger,	(int) PageSize::Ledger, },
  { S_paper_size_Legal,		(int) PageSize::Legal, },
  { S_paper_size_Statement,	(int) PageSize::Statement, },
  { S_paper_size_Executive,	(int) PageSize::Executive, },
  { S_paper_size_Folio,		(int) PageSize::Folio, },
  { S_paper_size_Quarto,	(int) PageSize::Quarto, },
  { S_paper_size_P10x14,	(int) PageSize::P10x14, },
  { S_paper_size_P11x17,	(int) PageSize::P11x17, },
  { S_paper_size_Note,		(int) PageSize::Note, },
  { S_paper_size_Env9,		(int) PageSize::Env9, },
  { S_paper_size_Env10,		(int) PageSize::Env10, },
  { S_paper_size_Env11,		(int) PageSize::Env11, },
  { S_paper_size_Env12,		(int) PageSize::Env12, },
  { S_paper_size_Env14,		(int) PageSize::Env14, },
  { S_paper_size_LetterExtra,	(int) PageSize::LetterExtra, },
  { S_paper_size_LegalExtra,	(int) PageSize::LegalExtra, },
  { S_paper_size_TabloidExtra,	(int) PageSize::TabloidExtra, },
  { S_paper_size_A4Extra,	(int) PageSize::A4Extra, },
  { S_paper_size_UserSize,	(int) PageSize::UserSize, },
  { S_paper_size2_A3,		(int) PageSize::A3, },
  { S_paper_size2_A4,		(int) PageSize::A4, },
  { S_paper_size2_A5,		(int) PageSize::A5, },
  { S_paper_size2_B4,		(int) PageSize::B4, },
  { S_paper_size2_B5,		(int) PageSize::B5, },
  { S_paper_size2_PostCard,	(int) PageSize::PostCard, },
  { S_paper_size2_Letter,	(int) PageSize::Letter, },
  { S_paper_size2_LetterSmall,	(int) PageSize::LetterSmall, },
  { S_paper_size2_Tabloid,	(int) PageSize::Tabloid, },
  { S_paper_size2_Ledger,	(int) PageSize::Ledger, },
  { S_paper_size2_Legal,	(int) PageSize::Legal, },
  { S_paper_size2_Statement,	(int) PageSize::Statement, },
  { S_paper_size2_Executive,	(int) PageSize::Executive, },
  { S_paper_size2_Folio,	(int) PageSize::Folio, },
  { S_paper_size2_Quarto,	(int) PageSize::Quarto, },
  { S_paper_size2_P10x14,	(int) PageSize::P10x14, },
  { S_paper_size2_P11x17,	(int) PageSize::P11x17, },
  { S_paper_size2_Note,		(int) PageSize::Note, },
  { S_paper_size2_Env9,		(int) PageSize::Env9, },
  { S_paper_size2_Env10,	(int) PageSize::Env10, },
  { S_paper_size2_Env11,	(int) PageSize::Env11, },
  { S_paper_size2_Env12,	(int) PageSize::Env12, },
  { S_paper_size2_Env14,	(int) PageSize::Env14, },
  { S_paper_size2_LetterExtra,	(int) PageSize::LetterExtra, },
  { S_paper_size2_LegalExtra,	(int) PageSize::LegalExtra, },
  { S_paper_size2_TabloidExtra,	(int) PageSize::TabloidExtra, },
  { S_paper_size2_A4Extra,	(int) PageSize::A4Extra, },
  { S_paper_size2_UserSize,	(int) PageSize::UserSize, },
  { NULL,			(int) PageSize::A4, },
};

Enum_Desc orientation_desc[] =
{
  { "Portrait",		(int) PageSize::Portrait, },
  { "Landscape",	(int) PageSize::Landscape, },
  { NULL,		(int) PageSize::Portrait, },
};

void 
PageSize::read_profile()
{
  default_kind = (PageSize::Kind) Pref_default_paper;
  default_orientation = (PageSize::Orientation) Pref_default_orientation;
  
  def_usr_size.Set(mm0_to_iu(Pref_default_user_page_width),
		    mm0_to_iu(Pref_default_user_page_height));
  
  def_usr_margin.Set(mm0_to_iu(Pref_default_user_top_margin),
		      mm0_to_iu(Pref_default_user_bottom_margin),
		      mm0_to_iu(Pref_default_user_left_margin),
		      mm0_to_iu(Pref_default_user_right_margin));
  
  min_margin.Set(mm0_to_iu(Pref_default_min_top_margin),
		  mm0_to_iu(Pref_default_min_bottom_margin),
		  mm0_to_iu(Pref_default_min_left_margin),
		  mm0_to_iu(Pref_default_min_right_margin));
  
  header_size = mm0_to_iu(Pref_default_header_size);
  footer_size = mm0_to_iu(Pref_default_header_size);
  margin_div = Pref_default_margin_div;
  
  // check default value consistency
  if ((default_kind < A3) || (default_kind > UserSize)) 
  {
    default_kind = A4;
  }
  if ((default_orientation != Portrait) && 
      (default_orientation != Landscape)) 
  {
    default_orientation = Portrait;
  }
  if (def_usr_size.x < min_paper_size) 
  {
    def_usr_size.x = min_paper_size;
  }
  if (def_usr_size.x > max_paper_size) 
  {
    def_usr_size.x = max_paper_size;
  }
  if (def_usr_size.y < min_paper_size) 
  {
    def_usr_size.y = min_paper_size;
  }
  if (def_usr_size.y > max_paper_size) 
  {
    def_usr_size.y = max_paper_size;
  }
  
  // check def_usr_margin
  GetProperMargin(def_usr_margin, def_usr_size);
  
  if (margin_div < 3) 
  {
    margin_div = 3;
  }
  
  // make template blueprint
  BluePrint::MakeTemplates();
}

void
PageSize::write_profile()
{
}

Iunit 
PageSize::GetMaxPaperSize()
{
  return max_paper_size;
}

Iunit 
PageSize::GetMinPaperSize()
{
  return min_paper_size;
}

Iunit 
PageSize::GetMinRegularSize()
{
  return min_regular_size;
}

Iunit 
PageSize::GetWidth(Kind pk, Orientation o)
{
  if (pk == PageSize::UserSize) 
  {
    return(o == Portrait) ? def_usr_size.x : def_usr_size.y;
  }
  return(o == Portrait) ? paper_size[pk][0] : paper_size[pk][1];
}

Iunit 
PageSize::GetHeight(Kind pk, Orientation o)
{
  if (pk == PageSize::UserSize) 
  {
    return(o == Portrait) ? def_usr_size.y : def_usr_size.x;
  }
  return(o == Portrait) ? paper_size[pk][1] : paper_size[pk][0];
}

void 
PageSize::GetMinTopBottomMargin(Kind pk, Orientation o, 
				 Iunit& min_top, Iunit& min_bot)
{
  Iunit height = GetHeight(pk, o);
  min_top = min_margin.GetTop();
  min_bot = min_margin.GetBottom();
  
  Iunit h = height - min_regular_size;
  if (min_top > h) 
  {
    min_top = h;
  }
  if (min_bot > h) 
  {
    min_bot = h;
  }
  if (min_top + min_bot > h) 
  {
    min_top = (min_top + h - min_bot) / 2;
    min_bot = h - min_top;
  }
}

void 
PageSize::GetMinLeftRightMargin(Kind pk, Orientation o,
				 Iunit& min_left, Iunit& min_right)
{
  Iunit width = GetWidth(pk, o);
  min_left = min_margin.GetLeft();
  min_right = min_margin.GetRight();
  Iunit w = width - min_regular_size;
  if (min_left > w) 
  {
    min_left = w;
  }
  if (min_right > w) 
  {
    min_right = w;
  }
  if (min_left + min_right > w) 
  {
    min_left = (min_left + w - min_right) / 2;
    min_right = w - min_left;
  }
}

Iunit 
PageSize::GetTopMargin(Kind pk, Orientation o)
{
  Iunit min_top;
  Iunit min_bot;
  GetMinTopBottomMargin(pk, o, min_top, min_bot);
  Iunit top = (pk != PageSize::UserSize) 
  ? GetHeight(pk, o) / margin_div 
  : def_usr_margin.GetTop();
  if (top < min_top) 
  {
    return min_top;
  }
  else 
  {
    return top;
  }
}

Iunit 
PageSize::GetBottomMargin(Kind pk, Orientation o)
{
  Iunit min_top;
  Iunit min_bot;
  GetMinTopBottomMargin(pk, o, min_top, min_bot);
  Iunit bottom = (pk != PageSize::UserSize) 
  ? GetHeight(pk, o) / margin_div 
  : def_usr_margin.GetBottom();
  if (bottom < min_bot) 
  {
    return min_bot;
  }
  else 
  {
    return bottom;
  }
}

Iunit 
PageSize::GetLeftMargin(Kind pk, Orientation o)
{
  Iunit min_left;
  Iunit min_right;
  GetMinLeftRightMargin(pk, o, min_left, min_right);
  Iunit left = (pk != PageSize::UserSize) 
  ? GetWidth(pk, o) / margin_div 
  : def_usr_margin.GetLeft();
  if (left < min_left) 
  {
    return min_left;
  }
  else 
  {
    return left;
  }
}

Iunit 
PageSize::GetRightMargin(Kind pk, Orientation o)
{
  Iunit min_left;
  Iunit min_right;
  GetMinLeftRightMargin(pk, o, min_left, min_right);
  Iunit right = (pk != PageSize::UserSize) 
  ? GetWidth(pk, o) / margin_div 
  : def_usr_margin.GetRight();
  if (right < min_right) 
  {
    return min_right;
  }
  else 
  {
    return right;
  }
}

Vector 
PageSize::GetSize(Kind pk, Orientation o)
{
  return Vector(GetWidth(pk, o), GetHeight(pk, o));
}

Margin 
PageSize::GetMargin(Kind pk, Orientation o)
{
  return Margin(GetTopMargin(pk, o), 
                 GetBottomMargin(pk, o),
                 GetLeftMargin(pk, o),
                 GetRightMargin(pk, o));
}

// get default paper size
Vector 
PageSize::GetDefaultSize()
{
  return GetSize(default_kind, default_orientation);
}

// get default margin size
Margin 
PageSize::GetDefaultMargin()
{
  return GetMargin(default_kind, default_orientation);
}

// get default header size
Iunit 
PageSize::GetDefaultHeaderSize()
{
  Iunit top_margin = GetTopMargin(default_kind, default_orientation);
  return(header_size <= top_margin)
  ? header_size
  : top_margin;
}

// get default footer size
Iunit 
PageSize::GetDefaultFooterSize()
{
  Iunit bot_margin = GetBottomMargin(default_kind, default_orientation);
  return(footer_size <= bot_margin)
  ? footer_size
  : bot_margin;
}

// get proper margin
void 
PageSize::GetProperMargin(Margin& m, Vector& page_size)
{
  Iunit size_y = page_size.y - min_regular_size;
  if ((m.GetTop() + m.GetBottom()) > size_y) 
  {
    Iunit top = (m.GetTop() < size_y) ? m.GetTop() : size_y;
    Iunit bottom = (m.GetBottom() < size_y) ? m.GetBottom() : size_y;
    if (top + bottom > size_y) 
    {
      top = (top + size_y - bottom) / 2;
      bottom = size_y - top;
    }
    m.SetTop(top);
    m.SetBottom(bottom);
  }
  Iunit size_x = page_size.x - min_regular_size;
  if ((m.GetLeft() + m.GetRight()) > size_x) 
  {
    Iunit left = (m.GetLeft() < size_x) ? m.GetLeft() : size_x;
    Iunit right = (m.GetRight() < size_x) ? m.GetRight() : size_x;
    if (left + right > size_x) 
    {
      left = (left + size_x - right) / 2;
      right = size_x - left;
    }
    m.SetLeft(left);
    m.SetRight(right);
  }
}


// ------------------------------------------------------------
// document page size

// constructor
PageSize::PageSize(DocumentContent* doc)
  : PageMap(doc, CreateLayoutTemplate(doc))
{
  uword gate_size = GetGateSize();
  for (uword i = 1; i <= gate_size; i++) 
  {
    TextFlow* tf = new TextFlow(doc);
    Replace(i, tf);			// textflow - gate map
    tf->SeedParagraph();
  }
  
  layout = GenerateNPage(1);
  kind   = default_kind;
  orient = default_orientation;
  size   = GetDefaultSize();
  margin = GetDefaultMargin();
  first_page_number = 1;
  FOIterator foi(GetLayoutTemplate()->GetFissionFrame());
  all_layout = 0;
  FrameObject* fo;
  while ((fo = foi()) != 0) 
  {
    if (fo->GetObjectID() == IDAllLayoutObject) 
    {
      all_layout = (AllLayout*)fo;
      break;
    }
  }
  assert(all_layout != 0);
}

// constructor for file read
PageSize::PageSize()
  : PageMap()
{
  layout = 0;
  kind   = default_kind;
  orient = default_orientation;
  size   = GetDefaultSize();
  margin = GetDefaultMargin();
  all_layout = 0;
  first_page_number = 1;
}

// initialize for file read
void 
PageSize::Initialize()
{
  layout = GeneratePage();
  FOIterator foi(GetLayoutTemplate()->GetFissionFrame());
  all_layout = 0;
  FrameObject* fo;
  while ((fo = foi()) != 0) 
  {
    if (fo->GetObjectID() == IDAllLayoutObject) 
    {
      all_layout = (AllLayout*)fo;
      break;
    }
  }
  assert(all_layout != 0);
}

// destructor
PageSize::~PageSize()
{
  ;
}

// save file
void 
PageSize::WriteToStream(PStream* stream, UniqLTList* ltlist)
{
  // save header and version
  stream->StartBlockOut(PStream::HDR_PAGESIZE, save_version);
  CHECK;
  
  // save page-map info
  PageMap::WriteToStream(stream, ltlist);
  CHECK;
  
  // save page size info
#ifdef _WIN32
  int sz = sizeof(bit16_enum_t);
  bit16_enum_t out = (bit16_enum_t)kind;
  stream->Out(&out, sz);
#else /* _WIN32 */
  int sz = sizeof(Kind);
  stream->Out(&kind, sz);
#endif /* _WIN32 */
  CHECK;
  
#ifdef _WIN32
  out = (bit16_enum_t)orient;
  stream->Out(&out, sz);
#else /* _WIN32 */
  sz = sizeof(Orientation);
  stream->Out(&orient, sz);
#endif /* _WIN32 */
  CHECK;
  
#ifdef _WIN32
  sz = sizeof(Bw2_Vector);
  Bw2_Vector v;
  Vector_to_Bw2_Vector(size, v);
  stream->Out(&v, sz);
#else /* _WIN32 */
  sz = sizeof(Vector);
  stream->Out(&size, sz);
#endif /* _WIN32 */
  CHECK;
  
#ifdef _WIN32
  sz = sizeof(Bw2_Margin);
  Bw2_Margin m;
  Margin_to_Bw2_Margin(margin, m);
  stream->Out(&m, sz);
#else /* _WIN32 */
  sz = sizeof(Margin);
  stream->Out(&margin, sz);
#endif /* _WIN32 */
  CHECK;
  
  sz = sizeof(word);
  stream->Out(&first_page_number, sz);
  CHECK;
  
  stream->EndBlockOut();
}

// load file
PageSize* 
PageSize::ReadFromStream(PStream* stream, 
			 DocumentContent* doc, 
			 UniqLTList* ltlist)
{
  // load header and version
  word version;
  stream->StartTypedBlockIn(PStream::HDR_PAGESIZE, version);
  CHECK 0;
  
  if (version != save_version) 
  {
    stream->SetError(SE_BADVERSION);
    return 0;
  }
  
  PageSize* pgs = new PageSize();
  
  // load page-map info
  PageMap* pm = PageMap::ReadFromStream(pgs, stream, doc, ltlist, 0);
  if ((stream->GetError() != SE_NOERROR) || (pm == 0)) 
  {
    // error
    delete pgs;
    return 0;
  }
  
  // save page size info
#ifdef _WIN32
  int sz = sizeof(bit16_enum_t);
  bit16_enum_t in;
  stream->In(&in, sz);
  pgs->kind = (Kind)in;
#else /* _WIN32 */
  int sz = sizeof(Kind);
  stream->In(&pgs->kind, sz);
#endif /* _WIN32 */
  CHECK 0;

#ifdef _WIN32
  stream->In(&in, sz);
  pgs->orient = (Orientation)in;
#else /* _WIN32 */
  sz = sizeof(Orientation);
  stream->In(&pgs->orient, sz);
#endif /* _WIN32 */  
  CHECK 0;

#ifdef _WIN32
  sz = sizeof(Bw2_Vector);
  Bw2_Vector v;
  stream->In(&v, sz);
  Bw2_Vector_to_Vector(v, pgs->size);
#else /* _WIN32 */
  sz = sizeof(Vector);
  stream->In(&pgs->size, sz);
#endif /* _WIN32 */  
  CHECK 0;

#ifdef _WIN32
  sz = sizeof(Bw2_Margin);
  Bw2_Margin m;
  stream->In(&m, sz);
  Bw2_Margin_to_Margin(m, pgs->margin);
#else /* _WIN32 */
  sz = sizeof(Margin);
  stream->In(&pgs->margin, sz);
#endif /* _WIN32 */
  CHECK 0;
  
  sz = sizeof(word);
  stream->In(&pgs->first_page_number, sz);
  CHECK 0;
  
  stream->EndBlockIn();
  
  return pgs;
}

// create layout-template
LayoutTemplate*
PageSize::CreateLayoutTemplate(DocumentContent* doc)
{
  return new LayoutTemplate(doc, BluePrint::GetZeroLayoutBluePrint(), 
			    LayoutTemplate::PageSizeTemplate);
}

// change margin
void 
PageSize::ChangeMargin()
{
  // update margin info.
  Fission* top;
  Fission* bottom;
  Fission* left;
  Fission* right;
  all_layout->GetMarginFission(&top, &bottom, &left, &right);
  margin.Set(((HFission*)top)->Y(),
              size.y - ((HFission*)bottom)->Y(),
              ((VFission*)left)->X(),
              size.x - ((VFission*)right)->X());
}

// change page size
void 
PageSize::ChangeSize()
{
  // resize all page
  all_layout->ChangeSize();
  GetDocumentContent()->RecalcAllLayoutLunit();
  GetDocumentContent()->RecalcLayouts();
}

// set page size
// this method can't handle user size
void 
PageSize::ChangeSize(Kind k, Orientation o)
{
  size = GetSize(k, o);
  //Margin m = GetMargin(k, o);
  Margin m = margin;
  ChangeSize(k, o, size, m);
}

// set page size
void 
PageSize::ChangeSize(Kind k, Orientation o, Vector& s, Margin& m)
{
  kind   = k;
  orient = o;
  size   = s;
  // check margin size
  margin = m;
  GetProperMargin(margin, size);
  
  Fission* top;
  Fission* bottom;
  Fission* left;
  Fission* right;
  all_layout->GetMarginFission(&top, &bottom, &left, &right);
  GetLayoutTemplate()
  ->GetFissionFrame()
  ->ChangeSizeAndMargin(size.x, size.y,
			 margin.GetTop(), s.y - margin.GetBottom(),
			 margin.GetLeft(), s.x - margin.GetRight(),
			 top, bottom, left, right);
  GetDocumentContent()->RecalcAllLayoutLunit();
  GetDocumentContent()->RecalcLayouts();
}

// set layout position
// x and y is the internal layout position
void 
PageSize::SetLayoutPosition(Lunit x, Lunit y)
{
  layout->SetLX(x - GetAllLayoutsOffsetX());
  layout->SetLY(y - GetAllLayoutsOffsetY());
}

// get all layout offset
Iunit 
PageSize::GetAllLayoutsOffsetX()
{
  return all_layout->GetCx();
}

Iunit 
PageSize::GetAllLayoutsOffsetY()
{
  return all_layout->GetCy();
}

Margin 
PageSize::GetDefaultFrameMargin()
{
  return Margin(0, 0, Point_to_iu(DEFAULT_POINT), 
		 Point_to_iu(DEFAULT_POINT));
}
