// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pstreams.cpp,v 3.8 2000/05/04 13:44:08 kudou Exp $
// implement PStream class for handling streams(files and
// memory blocks) to be read or written to.

#include "pword.h"
#include "pstreams.h"
#ifndef _WIN32
#  include <memory.h>
#endif
#include "vmem.h"

#ifndef NDEBUG
char* stream_errmsg[] =
{
  "no error",
  "bad header",
  "disk write error",
  "disk read error",
  "disk seek error",
  "bad version",
  "text format",
  "disk open error",
  "memory error",
  "not enough error",
  "stack overflow",
  "stack underflow",
  "no more",
  "broken flags",
  "not a block",
  "unexpected end block",
  "unexpected start block",
  "too much",
  "file end",
  "bad value",
  "unexpected binary",
  "overflow",
  "reached max",
};
#endif

// ------------------------------------------------------------
// RawStream class

// public lexical method
// But now, this method is called from `MemoryStream::Put' only.
void
RawStream::CheckMax(udword new_pos)
{
  if (this->max_set && !this->max_reached && this->mx < new_pos)
  {
    this->max_reached = True;
    if (this->max_func == NULL || (*this->max_func) ())
    {
      this->SetError(SE_TOOBIG);
    }
  }
}

// public lexical method
void
RawStream::SetMax(udword mx, bool(*max_func) ())
{
  this->mx = mx;
  this->max_func = max_func;
  this->max_set = True;
  this->max_reached = False;
}

// public virtial destructor
// RawStream, and subclasses of RawStream must have virtual destructor.
RawStream::~RawStream()
{
}

// protected constructor
RawStream::RawStream()
{
  this->error = SE_NOERROR;
  this->pos = 0;
  this->mx = 0;
  this->max_func = 0;
  this->max_set = False;
  this->max_reached = False;
}

// ------------------------------------------------------------
// MemoryStream class

// virtual public method(based on RawStream)
// This method do like a "seek" action.
void
MemoryStream::SkipIn(udword count)
{
  if (this->GetError() == SE_NOERROR)
  {
    this->pos += count;
  }
}

// virtual public method(based on RawStream)
udword
MemoryStream::Get(void* buf, udword count)
{
  udword size = area->GetSize();
  udword pos = this->pos;
  long remaining = (long) (size - pos);
  SET_MAX(remaining, 0);
  SET_MIN(count, (udword) remaining);
  if (count != 0)
  {
    void* stuff = area->GetStuff();
    if (stuff != NULL)
    {
      if (area->IsHuge())
      {
	huge_memcpy(buf, 0L, stuff, pos, count);
      }
      else
      {
	memcpy(buf, (char*) stuff + (uword) pos, (uword) count);
      }
    }
    this->pos += count;
  }
  return(count);
}

// virtual public method(based on RawStream)
void
MemoryStream::Put(void* buf, udword count)
{
  udword old_pos = this->pos;
  udword new_pos = old_pos + count;
  CheckMax(new_pos);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }
#ifndef STANDALONE
  if (area->GetSize() < new_pos)
  {
    area = area->Double(new_pos);
  }
#endif
  void* stuff = area->GetStuff();
  if (stuff != NULL)
  {
    if (area->IsHuge())
    {
      huge_memcpy(stuff, old_pos, buf, 0, count);
    }
    else
    {
      memcpy((char*) stuff + (uword) old_pos, buf, (uword) count);
    }
  }
  this->pos = new_pos;
}

// public virtial destructor(based on RawStream)
MemoryStream::~MemoryStream()
{
}

// public constructor
MemoryStream::MemoryStream(VMem*& _area)
  : area(_area)
{
}

// ------------------------------------------------------------
// FileStream class

// virtual public method(based on RawStream)
// This method do like a "seek" action.
void
FileStream::SkipIn(udword count)
{
  if (fseek(this->file, count, SEEK_CUR) < 0)
  {
    this->SetError(SE_DISKSEEKERROR);
  }
}

// virtual public method(based on RawStream)
udword 
FileStream::Get(void* buf, udword count)
{
  long ret = (count <= 30000
	      ? (long) (int) fread(buf, 1, (uword) count, this->file)
	      : (long) huge_fread(this->file, buf, count));
  if (ret < 0)
  {
    this->SetError(SE_DISKREADERROR);
  }
  return((udword) ret);
}

// virtual public method(based on RawStream)
void 
FileStream::Put(void* buf, udword count)
{
  if (0 < count)
  {
    long ret = (count <= 30000
		? (long) (int) fwrite(buf, 1, (uword) count, this->file)
		: (long) huge_fwrite(this->file, buf, count));
    if (ret < 0)
    {
      this->SetError(SE_DISKREADERROR);
    }
    else if ((udword) ret < count)
    {
      this->SetError(SE_TOOBIG);
    }
  }
}

// public virtial destructor(based on RawStream)
FileStream::~FileStream()
{
}

// public constructor
FileStream::FileStream(FILE* file)
{
  this->file = file;
}

// ------------------------------------------------------------
// BufStream class

// public lexical method
void
BufStream::SetMax(udword mx, bool(*fnc) ())
{
  this->raw->SetMax(mx, fnc);
}

// private lexical method
void NEAR
BufStream::Fill()
{
  int remaining = this->imax - this->ipos;
  if (0 < remaining)
  {
    memmove(this->buf, this->cpos, remaining);
  }
  this->cpos = this->buf;
  this->imax = remaining;
  this->ipos = 0;

  int read = (int) raw->Get(this->cpos + remaining, BUFSIZE - remaining);
  this->SetError(raw->GetError());
  if (0 <= read)
  {
    this->imax += read;
    if (read < BUFSIZE - remaining)
    {
      this->last = True;
    }
  }
}

// public lexical method
void
BufStream::SkipIn(udword count)
{
  int remaining = this->imax - this->ipos;
  if (count < (udword) remaining)
  {
    this->ipos += (uword) count;
    this->cpos += (uword) count;
  }
  else
  {
    this->raw->SkipIn(count - remaining);
    this->SetError(this->raw->GetError());
    this->ipos = this->imax = 0;
    this->cpos = this->buf;
  }
}

// public lexical method
udword
BufStream::GetHuge(void* buf, udword count)
{
  // decide how much to take from local buffer, maybe all
  uword from_buf = this->imax - this->ipos;
  if (count < from_buf)
  {
    from_buf = (uword) count;
  }
  
  // take from local buffer
  if (0 < from_buf)
  {
    huge_memcpy(buf, 0, this->cpos, 0, from_buf);
    this->cpos += from_buf;
    this->ipos += from_buf;
  }

  // take remainder from raw buffer
  udword from_raw = count - from_buf;
  if (0 < from_raw)
  {
    from_raw = raw->Get(huge_add(buf, from_buf), from_raw);
    this->SetError(raw->GetError());
  }
  return(from_buf + from_raw);
}

// public lexical method
uword
BufStream::Get(void* buf, uword count)
{
  uword remaining = this->imax - this->ipos;
  if (remaining < count)
  {
    if (!this->last)
    {
      this->Fill();
      if (this->GetError() != SE_NOERROR)
      {
	return(0);
      }
      remaining = this->imax;
    }
  }
  if (remaining < count)
  {
    count = remaining;
    this->SetError(SE_NOTENOUGHERROR);
  }
  memcpy(buf, this->cpos, count);
  this->cpos += count;
  this->ipos += count;
  return(count);
}

// public lexical method
ubyte
BufStream::GetByte()
{
  if (this->imax < (this->ipos += 1))
  {
    this->ipos -= 1;
    this->Fill();
    if (this->GetError() != SE_NOERROR)
    {
      return(0);
    }
    if (this->imax < (this->ipos += 1))
    {
      this->ipos -= 1;
      this->SetError(SE_NOTENOUGHERROR);
      return(0);
    }
  }
  return(*this->cpos++);
}

// public lexical method
uword
BufStream::GetWord()
{
  if (this->imax < (this->ipos += 2))
  {
    this->ipos -= 2;
    this->Fill();
    if (this->GetError() != SE_NOERROR)
    {
      return(0);
    }
    if (this->imax < (this->ipos += 2))
    {
      this->ipos -= 2;
      this->SetError(SE_NOTENOUGHERROR);
      return(0);
    }
  }
  return(*(*(uword**) &this->cpos)++);
}

// public lexical method
udword
BufStream::GetLong()
{
  if (this->imax < (this->ipos += 4))
  {
    this->ipos -= 4;
    this->Fill();
    if (this->GetError() != SE_NOERROR)
    {
      return(0);
    }
    if (this->imax < (this->ipos += 4))
    {
      this->ipos -= 4;
      this->SetError(SE_NOTENOUGHERROR);
      return(0);
    }
  }
  return(*(*(udword**) &this->cpos)++);
}

// private lexical method
void NEAR
BufStream::Flush()
{
  this->raw->Put(this->buf, this->ipos);
  this->SetError(raw->GetError());
  this->cpos = this->buf;
  this->ipos = this->imax = 0;
}

// public lexical method
void
BufStream::PutHuge(void* buf, udword count)
{
  this->Flush();
  this->raw->Put(buf, count);
  this->SetError(raw->GetError());
}

// public lexical method
void 
BufStream::Put(void* buf, uword count)
{
#ifndef STANDALONE
#ifndef NDEBUG
  if (BUFSIZE < count)
  {
    syserr("block too big in BufStream::Put");
  }
#endif
#endif
  if (BUFSIZE < this->ipos + count)
  {
    this->Flush();
    if (this->GetError() != SE_NOERROR)
    {
      return;
    }
  }
  memcpy(this->cpos, buf, count);
  this->cpos += count;
  this->ipos += count;
}

// public lexical method
void 
BufStream::PutByte(ubyte b)
{
  if (BUFSIZE < (this->ipos += 1))
  {
    this->ipos -= 1;
    this->Flush();
    if (this->GetError() != SE_NOERROR)
    {
      return;
    }
    this->ipos += 1;
  }
  *this->cpos++ = b;
}

// public lexical method
void 
BufStream::PutWord(uword w)
{
  if (BUFSIZE < (this->ipos += 2))
  {
    this->ipos -= 2;
    this->Flush();
    if (this->GetError() != SE_NOERROR)
    {
      return;
    }
    this->ipos += 2;
  }
  *(*(uword**) &this->cpos)++ = w;
}

// public lexical method
void 
BufStream::PutLong(udword l)
{
  if (BUFSIZE < (this->ipos += 4))
  {
    this->ipos -= 4;
    this->Flush();
    if (this->GetError() != SE_NOERROR)
    {
      return;
    }
    this->ipos += 4;
  }
  *(*(udword**) &this->cpos)++ = l;
}

void
BufStream::NewLine()
{
  this->PutByte('\r');
  this->PutByte('\n');
}

// public lexical method
void
BufStream::TermIn()
{
  this->termed = True;
}

// public lexical method
void
BufStream::TermOut()
{
  this->termed = True;
  this->Flush();
}

// public destructor
BufStream::~BufStream()
{
//  assert(this->GetError() != SE_NOERROR || termed);
#ifndef STANDALONE
  xfree_on_log2(this->buf, BUFSIZE_LOG2);
#else
  delete this->buf;
#endif
  if (this->own_raw)
  {
    delete this->raw;
    this->raw = 0;
  }
}

// public(?) lexical method
void
BufStream::Init()
{
  this->raw = raw;
#ifndef STANDALONE
  this->buf = (ubyte*) xmalloc_on_log2(BUFSIZE_LOG2);
#else
  this->buf = new ubyte [BUFSIZE];
#endif
  this->cpos = buf;
  this->ipos = 0;
  this->imax = 0;
  this->last = False;
  this->own_raw = False;
  this->error = SE_NOERROR;
  this->termed = False;
}

// public constructor
BufStream::BufStream(RawStream* raw)
{
  this->Init();
  this->raw = raw;
}

// public constructor
BufStream::BufStream(VMem*& vmem)
{
  this->Init();
  this->raw = new MemoryStream(vmem);
  this->own_raw = True;
}

// public constructor
BufStream::BufStream(FILE* f)
{
  this->Init();
  this->raw = new FileStream(f);
  this->own_raw = True;
}

// ------------------------------------------------------------
// BlockStream class

const int BlockStream::PSTREAM_VERSION = 0x0100;

// private lexical method
udword NEAR
BlockStream::xUValIn(byte flags, int& binary)
{
  if (!this->hit_endblock && this->GetError() == SE_NOERROR)
  {
    // Default error code.
    StreamError e = SE_BROKENFLAGS;
    
    binary = flags & BHFBinary;
    switch (flags & BHFType)
    {
     case BHFHeader:
      e = SE_UNEXPECTEDSTARTBLOCK;
      if (flags & BHFEndBlock)
      {
	e = SE_NOMORE;
	this->hit_endblock = 1;
      }
      break;
      
     case BHFImm:
      return(flags & BHFImmVal);
      
     case BHFVal:
      switch (flags & BHFLen)
      {
       case BHFLenByte:
	return(this->GetByte());
	
       case BHFLenWord:
	return(this->GetWord());
	
       case BHFLenLong:
	return(this->GetLong());    
      }
    }
    this->SetError(e);
  }
  return(0);
}

// private lexical method
dword NEAR
BlockStream::xValIn(byte flags, int& binary)
{
  if (!this->hit_endblock && this->GetError() == SE_NOERROR)
  {
    // Default error code.
    StreamError e = SE_BROKENFLAGS;
    
    binary = flags & BHFBinary;
    switch (flags & BHFType)
    {
     case BHFHeader:
      e = SE_UNEXPECTEDSTARTBLOCK;
      if (flags & BHFEndBlock)
      {
	e = SE_NOMORE;
	this->hit_endblock = 1;
      }
      break;
      
     case BHFImm:
      return(flags & BHFImmVal);
      
     case BHFVal:
      switch (flags & BHFLen)
      {
       case BHFLenByte:
#ifdef _MSC_VER
	// VC++ byte is unsigned char.
	return((dword) ((signed char) this->GetByte()));
#else /* _MSC_VER */
	return((dword) (byte) this->GetByte());
#endif /* _MSC_VER */
	
       case BHFLenWord:
	return((dword) (word) this->GetWord());
	
       case BHFLenLong:
	return((dword) this->GetLong());
      }
    }
    this->SetError(e);
  }
  return(0);
}

dword 
BlockStream::ValIn(int& binary)
{
  byte flags = BufStream::GetByte();
  if (GetError() != SE_NOERROR)
  {
    return 0;
  }
  return xValIn(flags, binary);
}

udword 
BlockStream::UValIn(int& binary)
{
  byte flags = BufStream::GetByte();
  if (GetError() != SE_NOERROR)
  {
    return 0;
  }
  return xUValIn(flags, binary);
}

byte 
BlockStream::InByte()
{
  int binary;
  dword d = ValIn(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  byte b = (byte) d;
  if (d != (dword)b)
  {
    SetError(SE_OVERFLOW);
  }
  return b;
}

ubyte 
BlockStream::InUByte()
{
  int binary;
  udword d = UValIn(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  ubyte b = (ubyte) d;
  if (d != (udword)b)
  {
    SetError(SE_OVERFLOW);
  }
  return b;
}

word 
BlockStream::InWord()
{
  int binary;
  dword d = ValIn(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  word w = (word) d;
  if (d != (dword)w)
  {
    SetError(SE_OVERFLOW);
  }
  return w;
}

uword
BlockStream::InUWord(int& binary)
{
  udword d = UValIn(binary);
  uword w = (uword)d;
  if (d != (udword)w)
  {
    SetError(SE_OVERFLOW);
  }
  return w;
}

uword 
BlockStream::InUWord()
{
  int binary;
  uword w = InUWord(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  return w;
}

dword 
BlockStream::InLong()
{
  int binary;
  dword d = ValIn(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  return d;
}

udword 
BlockStream::InULong()
{
  int binary;
  udword d = UValIn(binary);
  if (binary)
  {
    SetError(SE_UNEXPECTEDBINARY);
    return 0;
  }
  return d;
}

// public lexical method
void
BlockStream::In(void* buf, uword len)
{
  int binary;
  uword our_len = this->InUWord(binary);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }

  if (our_len > len)
  {
    Get(buf, len);
    if (GetError() != SE_NOERROR)
    {
      return;
    }
#ifndef STANDALONE
#if (!defined(NDEBUG))
    syserr("PStream::In -- PadIn called");
#endif
#endif
    //    PadIn(our_len-len);
    if (GetError() != SE_NOERROR)
    {
      return;
    }
    SetError(SE_TOOMUCH);
  }
  else if (our_len < len)
  {
    Get(buf, our_len);
    if (GetError() != SE_NOERROR) return;
    SetError(SE_NOTENOUGHERROR);
  }
  else 
  {
    Get(buf, len);
  }
}

// public lexical method
void
BlockStream::InHuge(void* buf, udword len)
{
  int binary;
  udword our_len = len;
  
  our_len = this->UValIn(binary);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }

  if (our_len > len)
  {
    GetHuge(buf, len); 
    if (GetError() != SE_NOERROR)
    {
      return;
    }
#ifndef STANDALONE
#if (!defined(NDEBUG))
    syserr("PStream::In -- PadIn called");
#endif
#endif
    //    PadIn(our_len-len);
    if (GetError() != SE_NOERROR)
    {
      return;
    }
    SetError(SE_TOOMUCH);
  }
  else if (our_len < len)
  {
    GetHuge(buf, our_len);
    if (GetError() != SE_NOERROR) return;
    SetError(SE_NOTENOUGHERROR);
  }
  else 
  {
    GetHuge(buf, len);
  }
}

// public lexical method
void
BlockStream::ValOut(dword val, int binary)
{
  byte flags = BHFVal;
  if (binary)
  {
    flags |= BHFBinary;
  }
  if (0 <= val && val <= 15)
  {
    PutByte(flags | BHFImm | (char)val);
  }
  else if (-128 <= val && val <= 127)
  {
    PutByte(flags | BHFLenByte);
    PutByte((ubyte) val);
  }
  else if (-32768L <= val && val <= 32767L)
  {
    PutByte(flags | BHFLenWord);
    PutWord((uword) val);
  }
  else
  {
    PutByte(flags | BHFLenLong);
    PutLong((udword) val);
  }
}

// public lexical method
void
BlockStream::ValOut(dword val)
{
  this->ValOut(val, 0);
}

// public lexical method
void
BlockStream::UValOut(udword val, int binary)
{
  byte flags = BHFVal;
  if (binary)
  {
    flags |= BHFBinary;
  }
  if (val <= 15)
  {
    PutByte(flags | BHFImm | (char) val);
  }
  else if (val <= 255)
  {
    PutByte(flags | BHFLenByte);
    PutByte((ubyte) val);
  }
  else if (val <= 65535L)
  {
    PutByte(flags | BHFLenWord);
    PutWord((uword) val);
  }
  else
  {
    PutByte(flags | BHFLenLong);
    PutLong(val);
  }
}

// public lexical method
void
BlockStream::UValOut(udword val)
{
  this->UValOut(val, 0);
}

// public lexical method
void
BlockStream::Out(void* buf, uword len)
{
  this->UValOut(len, True);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }
  BufStream::Put(buf, len);
}

// public lexical method
void
BlockStream::OutHuge(void* buf, udword len)
{
  this->UValOut(len, True);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }
  BufStream::PutHuge(buf, len);
}

// private lexical method
void NEAR
BlockStream::GotoEndBlock()
{
  if (this->hit_endblock)
  {
    // already there
    this->hit_endblock = 0;
    return;
  }
  for (;;)
  {
    byte flags = this->GetByte();
    if (this->GetError() != SE_NOERROR)
    {
      return;
    }
    switch (flags & BHFType)
    {
     case BHFImm:
     case BHFVal:
      {
	int binary;
	dword len = this->xValIn(flags, binary);
	if (binary)
	{
	  this->SkipIn(len);
	}
      }
      break;
      
     case BHFHeader:
      if (flags & BHFEndBlock)
      {
	// this is what we were looking for
	return;
      }
      else
      {
	word vers;
	word type;
	this->xStartBlockIn(flags, type, vers);
	// call recursively
	this->GotoEndBlock();
      }
      break;
      
     default:
      this->SetError(SE_BROKENFLAGS);
      return;
    }
  }
}

// public lexical method
void
BlockStream::EndBlockIn()
{
  this->GotoEndBlock();
  this->DownLevel();
}

// public lexical method
void
BlockStream::EndBlockOut()
{
  this->PutByte(BHFHeader | BHFEndBlock);
  this->DownLevel();
}

// public lexical method
void
BlockStream::xStartBlockIn(byte flags, word& type, word& vers)
{
  type = vers = 0;
  switch (flags & BHFType)
  {
   default:
    this->SetError(SE_BROKENFLAGS);
    break;
    
   case BHFVal:
   case BHFImm:
    this->SetError(SE_NOTBLOCK);
    break;
    
   case BHFHeader:
    if (flags & BHFEndBlock)
    {
      this->hit_endblock = 1;
      this->SetError(SE_NOMORE);
      break;
    }
    if (flags & BHFName)
    {
      this->Get(&type, 2);
      if (this->GetError() != SE_NOERROR)
      {
	break;
      }
    }
    if (flags & BHFVers)
    {
      this->Get(&vers, 2);
      if (this->GetError() != SE_NOERROR)
      {
	break;
      }
    }
    break;
  }
}

// public lexical method
void
BlockStream::StartBlockIn(word& type, word& vers)
{
  if (this->hit_endblock)
  {
    this->SetError(SE_NOMORE);
  }
  else
  {
    byte flags = this->GetByte();
    if (this->GetError() == SE_NOERROR)
    {
      this->xStartBlockIn(flags, type, vers);
#ifndef NDEBUG
      if (this->GetError() == SE_NOERROR)
      {
	// `UpLevel' do nothing in NDEBUG time.
	this->UpLevel(type);
      }
#endif
    }
  }
}

// public lexical method
void
BlockStream::StartTypedBlockIn(word type, word& vers)
{
  this->wanted = type;
  this->StartBlockIn(this->got, vers);
  if (this->GetError() != SE_NOERROR)
  {
    return;
  }
  if (this->got != type)
  {
    this->SetError(SE_BADHEADER);
  }
}

// public lexical method
void
BlockStream::StartBlockOut(word type, word vers)
{
  char buf[5];
  int length = 1;
  buf[0] = BHFHeader;
  if (type)
  {
    buf[0] |= BHFName;
    buf[length++] = ((char*) &type)[0];
    buf[length++] = ((char*) &type)[1];
  }
  if (vers)
  {
    buf[0] |= BHFVers;
    buf[length++] = ((char*) &vers)[0];
    buf[length++] = ((char*) &vers)[1];
  }
  this->Put(buf, length);
#ifndef NDEBUG
  if (this->GetError() == SE_NOERROR)
  {
    // `UpLevel' do nothing in NDEBUG time.
    this->UpLevel(type);
  }
#endif
}

// public lexical method
void
BlockStream::StartBlockOut(word type)
{
  this->StartBlockOut(type, 0);
}

// public lexical method
void
BlockStream::StartBlockOut()
{
  this->StartBlockOut(0);
}

// public lexical method
void
BlockStream::TermIn()
{
  this->EndBlockIn();
  BufStream::TermIn();
}

// public lexical method
void
BlockStream::TermOut()
{
  this->EndBlockOut();
  BufStream::TermOut();
}

// public lexical method
void 
BlockStream::InitIn()
{
  word cur_version;
  this->StartTypedBlockIn(HDR_PSTREAM, cur_version);
  if (cur_version != PSTREAM_VERSION)
  {
    this->SetError(SE_BADVERSION);
  }
}

// public lexical method
void 
BlockStream::InitOut()
{
  this->StartBlockOut(HDR_PSTREAM, PSTREAM_VERSION);
}

// public destructor
BlockStream::~BlockStream()
{
#ifndef NDEBUG
  if (this->curlev != 0)
  {
//    syserr("stream closed with pending blocks open");
  }
  delete types;
#endif
}

// private lexical method
void NEAR
BlockStream::Init()
{
  this->hit_endblock = 0;
#ifndef NDEBUG
  this->curlev = 0;
  this->types = new word[MAXLEV];
#endif
}

// public constructor
BlockStream::BlockStream(RawStream* raw)
  : BufStream(raw)
{
  this->Init();
}

// public constructor
BlockStream::BlockStream(VMem*& vmem)
  : BufStream(vmem)
{
  this->Init();
}

// public constructor
BlockStream::BlockStream(FILE* f)
  : BufStream(f)
{
  this->Init();
}

PStream::PStream(FILE* f, WriteWhat _what)
  : what(_what)
, BlockStream(f)
{
}

PStream::PStream(VMem*& v, WriteWhat _what)
  : what(_what)
, BlockStream(v)
{
}

PStream::~PStream()
{
}

// ------------------------------------------------------------
// BlockStream

void
BlockStream::OutString(char* s)
{
  if (s)
  {
    uword len = strlen(s)+1;
    ValOut(len);
#if 0
    Put(s, len);
#else
    PutHuge(s, len);
#endif
  }
  else ValOut(0);
}

char*
BlockStream::InString()
{
  uword len = InUWord();
  if (GetError() != SE_NOERROR) return 0;
  
  char* s = 0;
  if (len)
  {
    s = new char[ len ];
#if 0
    this->Get((void*)s, len);
#else
    this->GetHuge((void*)s, len);
#endif
  }
  return s;
}

#ifndef NDEBUG
char*
BlockStream::MakeErrorString()
{
  int e = GetError();
  char* msg = stream_errmsg[ e ];
  
  char* res = new char[ 1024 ],* buf = res;
  for (int i=0; i < curlev; i++)
  {
    word type = types[i];
    if (type)
    {
      buf += wsprintf(buf, "%c%c/", *(((char*)&type)+1), *(char*)&type);
    }
    else buf += wsprintf(buf, "--/");
  }
  buf += wsprintf(buf, " (%s)", msg);
  if (e == SE_BADHEADER)
  {
    buf += wsprintf(buf, " wanted %c%c, got %c%c", *(((char*)&wanted)+1), *(char*)&wanted, *(((char*)&got)+1), *(char*)&got);
  }
  return res;
}
#endif

#ifndef NDEBUG
void
BlockStream::Dump(FILE* f)
{
  fprintf(f, "\n");
  while (GetError() == SE_NOERROR)
  {
    Dump1(f, 0);
  }
  char* msg = MakeErrorString();
  fprintf(f, " final state: %s\n", msg);
  delete msg;
}

static void
indent_for_dump(FILE* f, int lev)
{
  for (int i= 0; i < 2 * lev; i++)
  {
    fprintf(f, " ");
  }
}

void
BlockStream::Dump1(FILE* f, int lev)
{
  udword len;
  
  byte flags = GetByte();
//  indent_for_dump(f, lev);
  fprintf(f, "[flags:0x%02x] ", flags);
  fprintf(f, "BHFVal:%d ", flags & BHFVal ? 1 : 0);
  fprintf(f, "BHFBinary:%d ", flags & BHFBinary ? 1 : 0);
  fprintf(f, "BHFImm:%d ", flags & BHFImm ? 1 : 0);
  fprintf(f, "BHFLenByte:%d ", flags & BHFLenByte ? 1 : 0);
  fprintf(f, "BHFLenWord:%d ", flags & BHFLenWord ? 1 : 0);
  fprintf(f, "BHFLenLong:%d\n", flags & BHFLenLong ? 1 : 0);
  fflush(f);

  switch (flags & BHFType)
  {
   default:
    indent_for_dump(f, lev);
    fprintf(f, "Unknown Type:0x%02x\n", flags & BHFType);
    break;
   case BHFHeader:
    {
      if (flags & BHFEndBlock)
      {
	SetError(SE_NOMORE);
	return;
      }
      word vers;
      word type;
      xStartBlockIn(flags, type, vers);
      indent_for_dump(f, lev);
      if (type)
      {
	fprintf(f,  "'%c%c' (0x%x) ", *(((char*)&type)+1), *(char*)&type, type); 
      }
      else
      {
	fprintf(f,  "Anon ");
      }
      if (vers)
      {
	fprintf(f,  "v=0x%x ", vers);
      }
      fprintf(f,  "\n");
      
      while (GetError() == SE_NOERROR)
      {
	Dump1(f, lev+1);
      }
      if (GetError() == SE_NOMORE)
      {
	SetError(SE_NOERROR);
      }
    }
    break;

   case BHFImm:
   case BHFVal:
    int binary;
    len = xUValIn(flags, binary);
    indent_for_dump(f, lev);
    fprintf(f,  "%li(0x%lx)\n", len, len);
    if (binary)
    {
      if (len > 32000)
      {
	len = 32000;
      }
      unsigned char* buf = new unsigned char[ (int)len ];
      Get(buf, (uword)len);
      if (GetError() == SE_TOOMUCH)
      {
	SetError(SE_NOERROR);
      }
      const udword n = 16;
      udword nn;
      int first = 1;
      while (len > 0)
      {
	indent_for_dump(f, lev);
	fprintf(f,  "%c", first ? '>' : ' ');
	first = 0;
	
	nn = n;
	if (len < nn)
	{
	  nn = (int)len;
	}
	for (udword i = 0; i < n; i++)
	{
	  if (i >= nn)
	  {
	    fprintf(f,  "   ");
	  }
	  else
	  {
	    unsigned int ii = buf[i];
	    fprintf(f, "%02x ", ii);
	  }
	}
	fprintf(f, " *");
	for (udword j = 0; j < n; j++)
	{
	  if (j >= nn)
	  {
	    fprintf(f,  " ");
	  }
	  else if (buf[j] > ' ' && buf[j]<0x80)
	  {
	    fprintf(f, "%c", buf[j]);
	  }
	  else
	  {
	    fprintf(f, ".");
	  }
	}
	fprintf(f, "*\n");
	len -= nn;
	buf += n;
      }
    }
    break;
  }
}
#endif
