// BeatWord Version 3.0

// BeatWord is a trademark of MSA Co.,LTD.
// Copyright (C) 1992, 1993 Pacifitech Corp.
// Copyright (C) 1999-2000 CYPAC Co.,Inc.

// This file is a free software. CYPAC gives you unlimited
// permission to copy and/or distribute it, as long as this 
// notice is preserved.

// $Id: pundo.cpp,v 3.4 2000/05/13 16:29:05 kudou Exp $
// implement PUndo class

#include "pword.h"
#include "pundo.h"
#include "again.h"
#include "attribut.h"
#include "atts.h"
#include "base.h"
#include "bound.h"
#include "bufnew.h"
#include "derivs.h"
#include "dialogs.h"
#include "docconte.h"
#include "docedit.h"
#include "docsel.h"
#include "docwindo.h"
#include "interval.h"
#include "intlist.h"
#include "mcursor.h"
#include "menu.h"
#include "mwindow.h"
#include "pmenus.h"
#include "pref.h"
#include "pstreams.h"
#include "pwordbas.h"
#include "pwordpre.h"
#include "styles.h"
#include "xstr.h"

DEFINE_SMARTNEW(PUndoAtom)
DEFINE_SMARTNEW(PUndoItem)

// ------------------------------------------------------------
// methods on PUndoPrivate

PUndoPrivate::PUndoPrivate()
{
  for (int i=0; i<MAXUNDO; i++)
  {
    undobuf[i] = 0;
  }
}

// ------------------------------------------------------------
// methods on PUndo

// this table gives the info that needs to be read/written to/from the stream for
// each function
int PUndo::func_data[] =
{
  PStream::WW_UNDOTEXT,
  PStream::WW_UNDOTEXT,
  PStream::WW_PARATTS,
  PStream::WW_CHARATTS,
  PStream::WW_NONE,
  PStream::WW_NONE,
  PStream::WW_NONE,
};
  
PUndo::PUndo(DocumentContent* dc)
  : content(dc)
{
  first_item = cur_item = next_item = 0;
  moved = False;
}

PUndo::~PUndo()
{
  Clear();
}

void 
PUndo::Clear()
{
  for (int i=first_item; i<next_item; i++)
  {
    delete GetAtom(i);
    GetAtom(i) = 0;
  }
  first_item = cur_item = next_item = 0;
}


bool NEAR
PUndo::NextUndoableAtom(int& i)
{
  for (i=cur_item-1; i>=first_item; i--)
  {
    if (GetAtom(i) != NULL)
    {
      return True;
    }
  }
  return False;
}

bool NEAR
PUndo::NextRedoableAtom(int& i)
{
  for (i=cur_item; i<next_item; i++)
  {
    if (GetAtom(i) != NULL)
    {
      return True;
    }
  }
  return False;
}

// private method
long NEAR
PUndo::menu_state(int undop)
{
  long result = MF_GRAYED;
  char* string = S_NIL;
  int i;
  if (undop ? this->NextUndoableAtom(i) : this->NextRedoableAtom(i))
  {    
    result = MF_ENABLED;
    string = GetAtom(i) ->get_string();
  }
  Menu::set_undo_redo(undop, string);
  return(result);
}

long
PUndo::undo(int menu_state_p, int singlep)
{
  if (menu_state_p)
  {
    return(this->menu_state(True));
  }
  if (this->menu_state(True) != MF_ENABLED)
  {
    ::MessageBeep(0);
  }
  else
  {
    SetMoved();
    Again::ToDoAgainInOtherDirection(singlep ? IDM_Edit_Redo
				      : IDM_Edit_RedoRun);
    xUndo(singlep);
  }
  return(0);
}

long
PUndo::redo(int menu_state_p, int singlep)
{
  if (menu_state_p)
  {
    return(this->menu_state(False));
  }
  if (this->menu_state(False) != MF_ENABLED)
  {
    ::MessageBeep(0);
  }
  else
  {
    SetMoved();
    Again::ToDoAgainInOtherDirection(singlep ? IDM_Edit_Undo
				      : IDM_Edit_UndoRun);
    xRedo(singlep);
  }
  return(0);
}

bool NEAR
PUndo::xRedo(int single)
{
  int i, last_i=0;
  bool can = NextRedoableAtom(i);
  PUndoAtom* atom = 0;
  bool big_p = False;
  bool result;
    
  while (can)
  {
    atom = GetAtom(i);
    cur_item = i;

    if (atom->BigP())
    {
      if (!big_p)
      {
	big_p = True;
	MouseCursor::StartLongActivity();
      }
    }

    bool success = atom->Redo();
    if (!success)
    {
      Clear();
      break;
    }
    cur_item++;
    if (single)
    {
      break;
    }
    last_i = i;
    can = this->NextRedoableAtom(i);
    if (!can)
    {
      break;
    }
    if (!this->Run(i, last_i))
    {
      break;
    }
  }
  if (atom != NULL)
  {
    atom->Announce(S_Redid);
    atom->RestoreAfter();
    result = True;
  }
  else
  {
    result = False;
  }

  if (big_p)
  {
    MouseCursor::EndLongActivity();
  }
  return result;
}

bool NEAR
PUndo::xUndo(int single)
{
  int i, last_i=0;
  bool can = NextUndoableAtom(i);
  PUndoAtom* atom = 0;
  bool result;
  bool big_p = False;
  
  while (can)
  {
    atom = GetAtom(i);
    cur_item = i;
    if (atom->BigP())
    {
      if (!big_p)
      {
	big_p = True;
	MouseCursor::StartLongActivity();
      }
    }
    bool success = atom->Undo();
    if (!success)
    {
      Clear();
      break;
    }
    if (single)
    {
      break;
    }
    last_i = i;
    can = this->NextUndoableAtom(i);
    if (!can)
    {
      break;
    }
    if (!PUndo::Run(last_i, i))
    {
      break;
    }
  }
  if (atom != NULL)
  {
    atom->Announce(S_Undid);
    atom->RestoreBefore();
    result = True;
  }
  else
  {
    result = False;
  }
  if (big_p)
  {
    MouseCursor::EndLongActivity();
  }
  return result;
}

// PUndo::Run -- see if two entries are part of the same "run"

bool NEAR
PUndo::Run(int i1, int i2)
{
  bool res = False;
  
  if (i1<first_item || i1>=next_item || i2<first_item || i2>=next_item)
  {
  }
  else
  {
    PUndoAtom* a1=GetAtom(i1),* a2=GetAtom(i2);
    UndoDesc t1=a1->GetUndoDesc(), t2=a2->GetUndoDesc();
    res = !a1->GetMoved() && t1==t2;
  }
  return res;
} 


// Resync is called when the user takes some action in the middle
// of an undo/redo sequence -- this eliminates the ability for him
// to redo stuff he previously undid

void 
PUndo::Resync()
{
  while (next_item > cur_item)
  {
    PUndoAtom*& atom = GetAtom(--next_item);
    delete atom;
    atom = 0;
  }
}

void NEAR
PUndo::Add(PUndoAtom* add_atom)
{
  Resync();

  PUndoAtom*& atom = GetAtom(next_item);
  
  // if there is already something here, it means we must have
  // wrapped around.  Get rid of the least recent undo entry.
  
  if (atom) 
  {
    assert(first_item < next_item);
    first_item++;
  }

  delete atom;
  atom = add_atom;
  
  if (first_item > MAXUNDO)
  {
    first_item -= MAXUNDO;
    next_item -= MAXUNDO;
    cur_item -= MAXUNDO;
  }
}


void 
PUndo::StartAtom(IntervalList* before, PUndo::UndoDesc desc)
{
  if (!Pref_no_undo)
  {
    this->dead_atom = False;
    PUndoAtom* atom = new PUndoAtom(this, before, desc);
    this->Add(atom);
  }
}

void 
PUndo::EndAtom(IntervalList* after)
{
  if (!Pref_no_undo)
  {
    PUndoAtom*& atom = GetAtom(cur_item);
    if ((atom->GetCount() <= 0) || (this->dead_atom))
    {
      delete atom;
      atom = NULL;
      if (this->dead_atom)
      {
	this->Clear();
      }
    }
    else
    {
      atom->SetAfter(after);
      next_item++;
      cur_item++;
    }
  }
}

void
PUndo::Register(Interval* target, UndoFunc func, int stuff)
{
  if (!Pref_no_undo && !this->dead_atom)
  {
    PUndoAtom*& atom = GetAtom(cur_item);
    if (!atom->Add(target, func, stuff))
    {
      this->dead_atom = True;
    }
  }
}

void 
PUndo::SetMoved()
{
  moved = 1;
}

bool
PUndo::GetMovedAndClear()
{
  bool old_moved = moved;
  moved = 0;
  return old_moved;
}

void 
PUndo::ReportDeadTextFlow(TextFlow* tf)
{
  for (int i=first_item; i<next_item; i++)
  {
    PUndoAtom*& atom = GetAtom(i);
    if (atom != NULL)
    {
      bool dead = atom->ReportDeadTextFlow(tf);
      if (dead)
      {
	delete atom;
	atom = 0;
      }
    }
  }
}


// ------------------------------------------------------------
// PUndoAtom

PUndoAtom::PUndoAtom(PUndo* _undo,
		      IntervalList* _before, PUndo::UndoDesc _desc)
  : undo(_undo)
, desc(_desc)
, head(0)
, tail(0)
, after(0)
, count(0)
{
  this->before = NULL;
  if (0 < _desc && _desc < PUndo::UD_UNUSED_MAX)
  {
    this->moved = undo->GetMovedAndClear();
    if (_before != NULL)
    {
      this->before = new BoundIntervalList;
      this->before->Copy(_before);
    }
  }
#ifndef NDEBUG
  else
  {
    syserr("PUndoAtom::PUndoAtom -- invalid desc");
  }
#endif
}

PUndoAtom::~PUndoAtom()
{
  for (PUndoItem* item = head; item != NULL;)
  {
    PUndoItem* next = item->next;
    delete item;
    item = next;
  }
}

void PUndoAtom::SetAfter(IntervalList* _after)
{
  if (_after != NULL)
  {
    this->after = new BoundIntervalList;
    this->after->Copy(_after);
  }
}

void NEAR
PUndoAtom::Announce(char* fmt)
{
  char* s = this->get_string();
  if (s[0] != '\0')
  {
    ::StatusOut(fmt, s);
  }
}

void 
PUndoAtom::RestoreBefore()
{
  this->Restore(this->before);
}

void 
PUndoAtom::RestoreAfter()
{
  this->Restore(this->after);
}

void 
PUndoAtom::Restore(BoundIntervalList* where)
{
  if (where != NULL)
  {
    Selected_Window->GetEditor()->GetSelectionStack()->InstallSelection
    (where->Unbind(undo->GetDocumentContent()));
  }
}

bool 
PUndoAtom::Undo()
{
  bool success = True;

  for (PUndoItem* item = head; item != NULL; item = item->next)
  {
    success = item->Undo() && success;
  }
  return success;
}

bool
PUndoAtom::Redo()
{
  bool success = False;
  
  for (PUndoItem* item = tail; item != NULL; item = item->prev)
  {
    if (item->Redo()) // at least one
    {
      success = True;
    }
  }
  return success;
}

bool 
PUndoAtom::Add(Interval* target, PUndo::UndoFunc func, int misc)
{
  bool result = True;
  
  if (((func == PUndo::UF_Delete) ||
       (func == PUndo::UF_CharAtt)) && 
      target->PointP())
  {
    // no need to remember, but not error
  }

  else
  {
    PUndoItem* item = new PUndoItem(this, target, func, misc);
    result = item->InitialGather();
    if (!result)
    {
      delete item;
    }
  
    else // link in item
    {
      if (tail == NULL)
      {
	tail = head = item;
      }
      else
      {
	head->prev = item;
	item->next = head;
	head = item;
      }
      count++;
    }
  }
  return result;
}

bool 
PUndoAtom::ReportDeadTextFlow(TextFlow* tf)
{
  if (this->before != NULL)
  {
    this->before->WeedDeadTextFlow(tf);
  }
  if (this->after != NULL)
  {
    this->after->WeedDeadTextFlow(tf);
  }

  PUndoItem** pitem = &head;
  while (*pitem)
  {
    BoundInterval* bi = (*pitem)->target;
    if (bi != NULL)
    {
      if ((*pitem)->target->GetTextFlow() == tf)
      {
	*pitem = (*pitem)->next;
	continue;
      }
    }
    pitem = &((*pitem)->next);
  }
  return(head == NULL);
}

char*  NEAR
PUndoAtom::get_string()
{
#define DEFUNDO(desc, j, e) extern char STATIC_NEAR S_ ## desc[];
#include "undo_m.h"
  
  static char* undo_string_table[] =
  {
#define DEFUNDO(desc, j, e) S_ ## desc,
#include "undo_m.h"
  };
  
  char* s = undo_string_table[desc];
  return((char*)(s == NULL ? "" : s));
}

bool
PUndoAtom::BigP()
{
  const int BIG = 1024;
  bool result = False;
  int insert_loc = 0;
    
  udword size = 0;
  for (PUndoItem* item = head; item != NULL && !result; item = item->next)
  {
    switch (item->GetUndoFunc())
    {
     case PUndo::UF_Insert:
      {
	// more than one insertion point is "big"
	if (++insert_loc > 1)
	{
	  result = True;
	}
	else
	{
	  BoundInterval* bi = item->GetTarget();
	  // multiple lines are "big"
	  if (bi->GetBP0().GetParID() != bi->GetBP1().GetParID())
	  {
	    result = True;
	  }
	}
	break;
      }
      
     case PUndo::UF_Delete:
     case PUndo::UF_ParAtt:
     case PUndo::UF_CharAtt:
      {
	VMem* vmem = item->GetStuff();
	if (vmem != NULL)
	{
	  size += vmem->GetSize();
	}
	result = (size > BIG);
      }
      break;
      
     // these guys are all "big"
     case PUndo::UF_StyleDefine:
     case PUndo::UF_StyleUndefine:
     case PUndo::UF_StyleRedefine:
      {
	result = True;
	break;
      }
    }
  }
  return result;
}


// ------------------------------------------------------------
// PUndoItem

PUndoItem::PUndoItem(PUndoAtom* _atom, Interval* _target, 
				  PUndo::UndoFunc _func, int _misc)
  : atom(_atom)
, func(_func)
, misc(_misc)
, next(NULL)
, prev(NULL)
{
  this->stuff = NULL;
  this->target = NULL;
  if (_target != NULL)
  {
    this->target = new BoundInterval(*_target);
  }
}

bool
PUndoItem::InitialGather()
{
  bool success;
  this->stuff = this->Gather(success);
  return success;
}
  
PUndoItem::~PUndoItem()
{
  delete this->stuff;
  delete this->target;
}

// Gather is the routine called when the action is registered,
// as well as as part of Undo and Redo
VMem* 
PUndoItem::Gather(bool& success)
{
  VMem* vmem = NULL;
  success = True;
  
  switch (this->func)
  {
   case PUndo::UF_Insert:
    break;

   case PUndo::UF_Delete:
   case PUndo::UF_ParAtt:
   case PUndo::UF_CharAtt:
    {
      Interval i(*(this->target), atom->GetUndo()->GetDocumentContent());
      vmem = VMem::Make(5);
      PStream stream(vmem, (PStream::WriteWhat)(this->GetWhat()));
      stream.InitOut();
      i.WriteFormattedToStream(&stream);
      if (vmem->GetStuff() == NULL)
      {
	stream.SetError(SE_MEMORYERROR);
      }
      if (stream.GetError() != SE_NOERROR)
      {
	if (stream.GetError() != SE_TOOBIG)
	{
#ifndef NDEBUG
	  char* msg;
	  syserr("PUndoItem::Gather -- writing to stream failed(%s)",
		  msg = stream.MakeErrorString());
	  delete msg;
#endif
	}
	success = False;
	delete vmem;
	vmem = 0;
	break;
      }
      stream.TermOut();
    }
    break;

   case PUndo::UF_StyleUndefine:
    break;
   
   case PUndo::UF_StyleDefine:
    break;
    
   case PUndo::UF_StyleRedefine:
    {
      vmem = VMem::Make(5);
      PStream stream(vmem, (PStream::WriteWhat)(this->GetWhat()));
      stream.InitOut();
      ::get_style_entry(misc) .WriteToStream(&stream);
      if (vmem->GetStuff() == NULL)
      {
	stream.SetError(SE_MEMORYERROR);
      }
      if (stream.GetError() != SE_NOERROR)
      {
#ifndef NDEBUG
	char* msg;
	syserr("PUndoItem::Gather -- writing style to stream failed(%s)",
		msg = stream.MakeErrorString());
	delete msg;
#endif
	success = False;
	delete vmem;
	vmem = 0;
	break;
      }
      stream.TermOut();
    }
    break;
  }
  return vmem;
}

void
PUndoItem::PutBack()
{
  switch (this->func)
  {
   case PUndo::UF_Delete:
    {
      Interval i(*(this->target), atom->GetUndo()->GetDocumentContent());
      i.Cut();
    }
    break;

   case PUndo::UF_Insert:
   case PUndo::UF_ParAtt:
   case PUndo::UF_CharAtt:
    {
      if (stuff != NULL)
      {
	Interval i(*(this->target), atom->GetUndo()->GetDocumentContent());
	PStream stream(stuff, (PStream::WriteWhat)(this->GetWhat()));
	stream.InitIn();
	Interval ii(i);
	i.InsertFromStream(&stream);
	if (this->func == PUndo::UF_ParAtt)
	{
	  ii.GetLeft()->BeginningOfBuffer();
	  ii.GetRight()->EndOfBuffer();
	}
	ii.DelayFormat();
	if (stream.GetError() != SE_NOERROR)
	{
#ifndef NDEBUG
	  char* msg;
	  syserr("PUndoItem::PutBack -- reading from stream failed(%s)",
		  msg = stream.MakeErrorString());
	  delete msg;
#endif
	}
	else
	{
	  stream.TermIn();
	}
      }
    }
    break;

   case PUndo::UF_StyleDefine:
    GlobalStyleTable.Resurrect(misc);
    break;

   case PUndo::UF_StyleUndefine:
    GlobalStyleTable.Kill(misc);
    break;

   case PUndo::UF_StyleRedefine:
    {
      PStream stream(stuff, (PStream::WriteWhat)(this->GetWhat()));
      stream.InitIn();
      ::get_style_entry(misc) .ReadFromStream(&stream);
      if (stream.GetError() != SE_NOERROR)
      {
#ifndef NDEBUG
	char* msg;
	syserr("PUndoItem::PutBack -- reading style from"
		" stream failed(%s)",
		msg = stream.MakeErrorString());
	delete msg;
#endif
      }
      else
      {
	stream.TermIn();
	PWordBase::StyleChange(misc);
      }
    }
    break;
  }
}

bool 
PUndoItem::Undo()
{
  PUndo::UndoFunc old_func = this->func;
  switch (old_func)
  {
   case PUndo::UF_Delete:
    func = PUndo::UF_Insert;
    break;
   case PUndo::UF_Insert:
    this->func = PUndo::UF_Delete;
    break;
   case PUndo::UF_StyleDefine:
    this->func = PUndo::UF_StyleUndefine;
    break;
   case PUndo::UF_StyleUndefine:
    this->func = PUndo::UF_StyleDefine;
    break;
   default:
    break;
  }
    
  bool ret = this->Redo();
  this->func = old_func;
  return ret;
}

bool 
PUndoItem::Redo()
{
  bool success;
  VMem* vmem = this->Gather(success);
  
  // even if not successful, go ahead and put back -- 
  // this undoes large insertions, for instance
  this->PutBack();

  delete this->stuff;
  this->stuff = vmem;
  return success;
}

int
PUndoItem::GetWhat()
{
  return PUndo::func_data [func] | PStream::WW_UNDO;
}
