// -------------------------------------------------------------------- //
//                     Leaf class library for Windows
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// term.cpp:
//
// class TerminalWnd
// class TtySetDlg
//
// terminal sample
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/lw/sample/term/RCS/term.cpp,v 1.3 1994/05/05 16:30:15 kudou Exp $

#include "lw.h"
#include "termwnd.h"
#include "term.h"
#include <string.h>

// -------------------------------------------------------------------- //
// class TerminalWnd
// -------------------------------------------------------------------- //

bool TerminalWnd::register_flag = False;

TerminalWnd::TerminalWnd ()
{
  term_wnd = 0;
  status_wnd = 0;
}

TerminalWnd::~TerminalWnd ()
{
  delete term_wnd;
  delete status_wnd;
}

bool
TerminalWnd::RegisterFlag ()
{
  return register_flag;
}

void
TerminalWnd::SetRegisterFlag ()
{
  register_flag = True;
}

LPSTR
TerminalWnd::GetMenuName ()
{
  return "TermMenu";
}

HANDLE
TerminalWnd::GetIcon ()
{
  return MSWin::LoadIcon ("TermIco");
}

LPSTR
TerminalWnd::GetCaptionStr ()
{
  return "TERM";
}

void
TerminalWnd::OpenChildWindow ()
{
  term_wnd = new TermWnd (this);
  term_wnd->OpenWindow ();
  status_wnd = new StatusWnd (this);
  status_wnd->OpenWindow ();
}

uword
TerminalWnd::Menu (WPARAM wparam, LPARAM lparam)
{
  uword ret = MF_ENABLED;
  bool menu_state = (lparam == GET_MENU_STATE);
  switch (wparam)
  {
   case MENU_ID_CONNECT:
    ret = Connect (menu_state);
    break;
    
   case MENU_ID_CLOSE:
    ret = Close (menu_state);
    break;
    
   case MENU_ID_COMSET:
    ret = ComSet (menu_state);
    break;
    
   case MENU_ID_EXIT:
    ret = Exit (menu_state);
    break;
    
   case MENU_ID_ABOUT:
    ret = About (menu_state);
    break;
  }
  return ret;
}

uword
TerminalWnd::Connect (bool menu_state)
{
  Comm comm = GetTermWnd ()->GetComm ();
  if (menu_state)
  {
    return comm.OpenP () ? (MF_GRAYED | MF_DISABLED) : MF_ENABLED;
  }
  if (!comm.OpenP ())
  {
    GetTermWnd ()->OpenComm ();
  }
  return 0;
}

uword
TerminalWnd::Close (bool menu_state)
{
  Comm comm = GetTermWnd ()->GetComm ();
  if (menu_state)
  {
    return comm.OpenP () ? MF_ENABLED : (MF_GRAYED | MF_DISABLED);
  }
  if (comm.OpenP ())
  {
    GetTermWnd ()->CloseComm ();
  }
  return 0;
}

uword
TerminalWnd::ComSet (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  TtySetDlg dlg (GetTermWnd ());
  dlg.Go ();
  return 0;
}

uword
TerminalWnd::Exit (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  if (MessageBox (STR_ID_EXIT, STR_ID_TERM, MB_OKCANCEL) == IDOK)
  {
    ::PostQuitMessage (0);	
  }
  return 0;
}

uword
TerminalWnd::About (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  MessageBox (STR_ID_ABOUT, STR_ID_TERM, MB_OK);
  return 0;
}

void
TerminalWnd::Display (HDC hdc)
{
  DisplayGrayFrame (hdc, border_size);
}

void
TerminalWnd::SetFocus ()
{
  if (GetTermWnd ())
  {
    GetTermWnd ()->SetFocus ();
  }
}

void
TerminalWnd::Resize (WORD left, WORD top, WORD right, WORD bottom)
{
  WORD statusbar_height = (GetStatusWnd () ?
			   GetStatusWnd ()->GetBarHeight () : 0);
  if (GetStatusWnd ())
  {
    GetStatusWnd ()->MoveWindow (border_size,
				 bottom - top - border_size - statusbar_height,
				 right - left - border_size,
				 bottom - top - border_size);
  }
  if (GetTermWnd ())
  {
    GetTermWnd ()->MoveWindow (border_size,
			       border_size,
			       right - left - border_size,
			       bottom - top - border_size - statusbar_height);
  }
}

void
TerminalWnd::MenuSelect (WPARAM wparam)
{
  switch (wparam)
  {
   case MENU_ID_CONNECT:
   case MENU_ID_CLOSE:
   case MENU_ID_COMSET:
   case MENU_ID_EXIT:
   case MENU_ID_ABOUT:
    GetStatusWnd ()->StrIdPrintf (wparam + MENU_MSG_OFFSET);
    break;

   default:
    GetStatusWnd ()->ClearMsg ();
  }
}

long
TerminalWnd::WindowProc (UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
   case WM_MENUSELECT:
    MenuSelect (wparam);
    break;

   default:
    return BaseWnd::WindowProc (msg, wparam, lparam);
  }
  return 0L;
}

// -------------------------------------------------------------------- //
// class TtySetDlg
// -------------------------------------------------------------------- //
TtySetDlg::TtySetDlg (TermWnd* wnd)
: ModalDlg ("TTYSetDlg"), comm (wnd->GetComm ())
{
  term_wnd = wnd;
}
  
TtySetDlg::~TtySetDlg ()
{
}

static struct baud_rate
{
  int baud;
  int radio_btn_id;
} baud_rate_tbl [] =
{
  //    { CBR_110, TTYSETDLG_110 },
  { CBR_300, TTYSETDLG_300 },
  //    { CBR_600, TTYSETDLG_600 },
  { CBR_1200, TTYSETDLG_1200 },
  { CBR_2400, TTYSETDLG_2400 },
  { CBR_4800, TTYSETDLG_4800 },
  { CBR_9600, TTYSETDLG_9600 },
  { CBR_14400, TTYSETDLG_14400 },
  { CBR_19200, TTYSETDLG_19200 },
  { CBR_38400, TTYSETDLG_38400 },
  { CBR_56000, TTYSETDLG_56000 },
  { CBR_128000, TTYSETDLG_128000 },
  { CBR_256000, TTYSETDLG_256000 },
  0
};

BOOL
TtySetDlg::WmInitDialog ()
{
  // setup baud rate 
  for (baud_rate* b = baud_rate_tbl; b->baud != 0; b++)
  {
    if (comm.GetBaudRate () == b->baud)
    {
      CheckRadioButton (TTYSETDLG_300, TTYSETDLG_256000, b->radio_btn_id);
      break;
    }
  }
  
  // setup byte size
  if (comm.GetByteSize () == 7)
  {
    CheckRadioButton (TTYSETDLG_BIT7, TTYSETDLG_BIT8, TTYSETDLG_BIT7);
  }
  else // 8
  {
    CheckRadioButton (TTYSETDLG_BIT7, TTYSETDLG_BIT8, TTYSETDLG_BIT8);
  }
  
  // setup flow control
  if (comm.XFlowControlP ())
  {
    CheckRadioButton (TTYSETDLG_X, TTYSETDLG_FNORN, TTYSETDLG_X);
  }
  else if (comm.HardFlowControlP ())
  {
    CheckRadioButton (TTYSETDLG_X, TTYSETDLG_FNORN, TTYSETDLG_HARD);
  }
  else if (comm.NornFlowControlP ())
  {
    CheckRadioButton (TTYSETDLG_X, TTYSETDLG_FNORN, TTYSETDLG_FNORN);
  }
  
  // setup stop bit
  int id;
  switch (comm.GetStopBits ())
  {
   default:
   case ONESTOPBIT:
    id = TTYSETDLG_STOP1;
    break;
   case ONE5STOPBITS:
    id = TTYSETDLG_STOP15;
    break;
   case TWOSTOPBITS:
    id = TTYSETDLG_STOP2;
    break;
  }
  CheckRadioButton (TTYSETDLG_STOP1, TTYSETDLG_STOP2, id);
  
  // setup parity
  switch (comm.GetParity ())
  {
   default:
   case NOPARITY:
    id = TTYSETDLG_PNORN;
    break;
   case ODDPARITY:
    id = TTYSETDLG_ODD;
    break;
   case EVENPARITY:
    id = TTYSETDLG_EVEN;
    break;
  }
  CheckRadioButton (TTYSETDLG_PNORN, TTYSETDLG_EVEN, id);
  
  // setup comm port
  switch (comm.GetPortNumber ())
  {
   default:
   case 0:
    id = TTYSETDLG_COM1;
    break;
   case 1:
    id = TTYSETDLG_COM2;
    break;
   case 2:
    id = TTYSETDLG_COM3;
    break;
   case 3:
    id = TTYSETDLG_COM4;
    break;
  }
  CheckRadioButton (TTYSETDLG_COM1, TTYSETDLG_COM4, id);
  
  // CN_RECEIVE
  CheckDlgButton (TTYSETDLG_CN_RECEIVE,
		  comm.UseCnReceiveP () ? 1 : 0);

  return Dlg::WmInitDialog ();
}

void
TtySetDlg::Fixed ()
{
  bool dirty = False;
  // baud rate
  for (baud_rate* b = baud_rate_tbl; b->baud != 0; b++)
  {
    if (DlgButtonCheckedP (b->radio_btn_id))
    {
      if (comm.GetBaudRate () != b->baud)
      {
	comm.SetBaudRate (b->baud);
	dirty = True;
      }
      break;
    }
  }
  // byte size
  if (DlgButtonCheckedP (TTYSETDLG_BIT7))
  {
    if (comm.GetByteSize () != 7)
    {
      comm.SetByteSize (7);
      dirty = True;
    }
  }
  else
  {
    if (comm.GetByteSize () != 8)
    {
      comm.SetByteSize (8);
      dirty = True;
    }
  }
  // flow control
  if (DlgButtonCheckedP (TTYSETDLG_X))
  {
    if (!comm.XFlowControlP ())
    {
      comm.SetXFlowControl ();
      dirty = True;
    }
  }
  else if (DlgButtonCheckedP (TTYSETDLG_HARD))
  {
    if (!comm.HardFlowControlP ())
    {
      comm.SetHardFlowControl ();
      dirty = True;
    }
  }
  else
  {
    if (!comm.NornFlowControlP ())
    {
      comm.SetNornFlowControl ();
      dirty = True;
    }
  }
  // stop bit
  if (DlgButtonCheckedP (TTYSETDLG_STOP1))
  {
    if (comm.GetStopBits () != ONESTOPBIT)
    {
      comm.SetStopBits (ONESTOPBIT);
      dirty = True;
    }
  }
  else if (DlgButtonCheckedP (TTYSETDLG_STOP15))
  {
    if (comm.GetStopBits () != ONE5STOPBITS)
    {
      comm.SetStopBits (ONE5STOPBITS);
      dirty = True;
    }
  }
  else
  {
    if (comm.GetStopBits () != TWOSTOPBITS)
    {
      comm.SetStopBits (TWOSTOPBITS);
      dirty = True;
    }
  }
  // parity
  if (DlgButtonCheckedP (TTYSETDLG_PNORN))
  {
    if (comm.GetParity () != NOPARITY)
    {
      comm.SetParity (NOPARITY);
      dirty = True;
    }
  }
  else if (DlgButtonCheckedP (TTYSETDLG_ODD))
  {
    if (comm.GetParity () != ODDPARITY)
    {
      comm.SetParity (ODDPARITY);
      dirty = True;
    }
  }
  else
  {
    if (comm.GetParity () != EVENPARITY)
    {
      comm.SetParity (EVENPARITY);
      dirty = True;
    }
  }
  // comm port
  bool port_dirty = False;
  int new_port;
  if (DlgButtonCheckedP (TTYSETDLG_COM1))
  {
    if (comm.GetPortNumber () != 0)
    {
      port_dirty = True;
      new_port = 0;
    }
  } 
  else if (DlgButtonCheckedP (TTYSETDLG_COM2))
  {
    if (comm.GetPortNumber () != 1)
    {
      port_dirty = True;
      new_port = 1;
    }
  }
  else if (DlgButtonCheckedP (TTYSETDLG_COM3))
  {
    if (comm.GetPortNumber () != 2)
    {
      port_dirty = True;
      new_port = 2;
    }
  }
  else
  {
    if (comm.GetPortNumber () != 3)
    {
      port_dirty = True;
      new_port = 3;
    }
  }
  // CN_RECEIVE
  bool cn_receive_dirty = False;
  if (DlgButtonCheckedP (TTYSETDLG_CN_RECEIVE))
  {
    if (!comm.UseCnReceiveP ())
    {
      comm.SetUseCnReceive (True);
      cn_receive_dirty = True;
    }
  }
  else
  {
    if (comm.UseCnReceiveP ())
    {
      comm.SetUseCnReceive (False);
      cn_receive_dirty = True;
    }
  }
  
  // change port setting
  if (comm.OpenP ())
  {
    if (dirty || port_dirty || cn_receive_dirty)
    {
      term_wnd->CloseComm ();
      if (port_dirty)
      {
	comm.SetPortNumber (new_port);
      }
      term_wnd->OpenComm ();
    }
  }
  else
  {
    if (port_dirty)
    {
      comm.SetPortNumber (new_port);
    }
  }
}

BOOL
TtySetDlg::WmCommand (WPARAM wparam)
{
  switch (wparam)
  {
   case IDOK:
    Fixed ();
    EndDialog (True);
    return 1;
    
   case IDCANCEL:
    EndDialog (False);
    return 1;
  }
  return 0;
}

// -------------------------------------------------------------------- //
// main
// -------------------------------------------------------------------- //
int
Main ()
{
  TerminalWnd wnd;
  wnd.OpenWindow ();
  return MSWin::MessageLoop ();
}
