// -------------------------------------------------------------------- //
//                        Wool Object Oriented Lisp
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// wool.cc:
//
// class Wool
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/wool/RCS/wool.cc,v 1.12 1994/05/02 16:05:02 kudou Exp $

#include "wool.h"

bool
Wool::AtomP (Object* _x)
{
  GCLink x = _x;
  
  return NullP (x) ? True : !TypeP_ (x, WT_Pair);
}

bool 
Wool::TypeP (Object* _x, WoolType t)
{
  GCLink x = _x;
  
  if (NullP (x))
  {
    return (t == WT_Atom);
  }
  return TypeP_ (x, t);
}

Object*
Wool::SetBind (Object* _atom, Object* _attr, Object* _value)
{
  GCLink atom = _atom;
  GCLink attr = _attr;
  GCLink value = _value;
  
  if (NullP (atom) || !TypeP_ (atom, WT_Atom))
  {
    Error ("SetBind : arg1 isn't atom");
  }
  GCLink blist = ((Atom*)((Object*)atom))->GetBind ();
  GCLink o = Assoc (attr, blist);
  if (NullP (o))
  {
    GCLink g1 = Cons (attr, value);
    GCLink g2 = Cons (g1, blist);
    ((Atom*)((Object*)atom))->SetBind (g2);
  }
  else
  {
     assert (TypeP_(o, WT_Pair));
    ((Pair*)((Object*)o))->ReplaceCdr (value);
  }
  return value;
}

Object*
Wool::SetBind (char* atom, Object* _attr, Object* _value)
{
  GCLink attr = _attr;
  GCLink value = _value;
  
  GCLink a = GetAtom (atom);
  return SetBind (a, attr, value);
}

Object*
Wool::GetBind (Object* _atom, Object* _attr)
{
  GCLink atom = _atom;
  GCLink attr = _attr;
  
  if (!TypeP (atom, WT_Atom))
  {
    Error ("Get : arg1 isn't atom");
  }
  GCLink o = ((Atom*)((Object*)atom))->GetBind ();
  return Assoc (attr, o);
}

Object* 
Wool::Get (Object* _x)
{
  GCLink x = _x;
  
  GCLink apval = GetAtom ("apval");
  GCLink o = GetBind (x, apval);
  return !NullP (o) ? Cdr (o) : 0;
}

Object*
Wool::CSet (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;
  
  GCLink a = GetAtom ("apval");
  return SetBind (x, a, y);
}

Object*
Wool::CSetQ (Object* _x, Object* _y, Object* _a)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink a = _a;
  
  GCLink s = Eval (y, a);
  return CSet (x, s);
}

Object*
Wool::Set (Object* _x, Object* _y, Object* _a)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink a = _a;
  
  GCLink o = Assoc (x, a);
  if (NullP (o))
  {
    GCLink apval = GetAtom ("apval");
    return SetBind (x, apval, y);
  }
  Rplacd (o, y);
  return y;
}

Object*
Wool::SetQ (Object* _x, Object* _y, Object* _a)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink a = _a;
  
  GCLink s = Eval (y, a);
  return Set (x, s, a);
}

Object* 
Wool::EndRun ()
{
  run = False;
  return 0;
}

Object*
Wool::Version ()
{
  return new (this) Str (WOOL_VERSION);
}

Object*
Wool::CallSubr (Object* _f, Object* _e, Object* _a)
{
  GCLink f = _f;
  GCLink e = _e;
  GCLink a = _a;
  
  if (!TypeP (f, WT_Func))
  {
    Error ("CallSubr : invalid subr description");
  }
#define func ((Func*)((Object*)f))
  int arg_num = func->GetArgNum ();
  if (func->UnfixedArgNumP ())
  {
    if (func->CallWithEnvP ())
    {
      if (arg_num == 2)
      {
	GCLink g1 = Evlis (e, a);
	return func->Func2 (*this, g1, a);
      }
    }
    else
    {
      if (arg_num == 1)
      {
	GCLink g1 = Evlis (e, a);
	return func->Func1 (*this, g1);
      }
      else if (arg_num == 2)
      {
	GCLink g1 = Car (e);
	g1 = Eval (g1, a);
	GCLink g2 = Cdr (e);
	g2 = Evlis (g2, a);
	return func->Func2 (*this, g1, g2);
      }
    }
    return Error ("CallSubr : sorry not supported unfixed argument with arg_num %d %s",
		  arg_num,
		  (func->CallWithEnvP () ? "and call with env" : ""));
  }
  GCLink l[5];
  l[0] = e;
  if (func->CallWithEnvP ())
  {
    for (int i = 1; i < arg_num - 1; i++)
    {
      l[i] = Cdr (l[i - 1]);
    }
    switch (arg_num)
    {
     case 0:
      return func->Func0 (*this);
     case 1:
      return func->Func1 (*this, a);
     case 2:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	return func->Func2 (*this, g2, a);
      }
     case 3:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	GCLink g3 = Car (l[1]);
	GCLink g4 = Eval (g3, a);
	return func->Func3 (*this, g2, g4, a);
      }
     case 4:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	GCLink g3 = Car (l[1]);
	GCLink g4 = Eval (g3, a);
	GCLink g5 = Car (l[2]);
	GCLink g6 = Eval (g5, a);
	return func->Func4 (*this, g2, g4, g6, a);
      }
     default:
      Error ("CallSubr : Function argument number is too match");
    }
  }
  else
  {
    for (int i = 1; i < arg_num; i++)
    {
      l[i] = Cdr (l[i - 1]);
    }
    switch (arg_num)
    {
     case 0:
      return func->Func0 (*this);
     case 1:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	return func->Func1 (*this, g2);
      }
     case 2:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	GCLink g3 = Car (l[1]);
	GCLink g4 = Eval (g3, a);
	return func->Func2 (*this, g2, g4);
      }
     case 3:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	GCLink g3 = Car (l[1]);
	GCLink g4 = Eval (g3, a);
	GCLink g5 = Car (l[2]);
	GCLink g6 = Eval (g5, a);
	return func->Func3 (*this, g2, g4, g6);
      }
     case 4:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Eval (g1, a);
	GCLink g3 = Car (l[1]);
	GCLink g4 = Eval (g3, a);
	GCLink g5 = Car (l[2]);
	GCLink g6 = Eval (g5, a);
	GCLink g7 = Car (l[3]);
	GCLink g8 = Eval (g7, a);
	return func->Func4 (*this, g2, g4, g6, g8);
      }
     default:
      Error ("CallSubr : Function argument number is too match");
    }
  }
  return 0; // dummy
}

Object*
Wool::CallFSubr (Object* _f, Object* _e, Object* _a)
{
  GCLink f = _f;
  GCLink e = _e;
  GCLink a = _a;

  if (!TypeP (f, WT_Func))
  {
    Error ("CallFSubr : invalid subr description");
  }
#define func ((Func*)((Object*)f))
  int arg_num = func->GetArgNum ();
  if (func->UnfixedArgNumP ())
  {
    if (func->CallWithEnvP ())
    {
      if (arg_num == 2)
      {
	return func->Func2 (*this, e, a);
      }
    }
    else
    {
      if (arg_num == 1)
      {
	return func->Func1 (*this, e);
      }
    }
    return Error ("CallFSubr : unfixed argument with arg_num %d %s",
		  arg_num,
		  (func->CallWithEnvP () ? "and call with env" : ""));
  }
  GCLink l[5];
  l[0] = e;
  if (func->CallWithEnvP ())
  {
    for (int i = 1; i < arg_num - 1; i++)
    {
      l[i] = Cdr (l[i - 1]);
    }
    switch (arg_num)
    {
     case 0:
      return func->Func0 (*this);
     case 1:
      return func->Func1 (*this, a);
     case 2:
      {
	GCLink g1 = Car (l[0]);
	return func->Func2 (*this, g1, a);
      }
     case 3:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Car (l[1]);
	return func->Func3 (*this, g1, g2, a);
      }
    case 4:
      {
	GCLink g1 = Car (l[0]); 
	GCLink g2 = Car (l[1]);
	GCLink g3 = Car (l[2]);
	return func->Func4 (*this, g1, g2, g3, a);
      }
     default:
      Error ("CallFSubr : Function argument number is too match");
    }
  }
  else
  {
    for (int i = 1; i < arg_num; i++)
    {
      l[i] = Cdr (l[i - 1]);
    }
    switch (arg_num)
    {
     case 0:
      return func->Func0 (*this);
     case 1:
      {
	GCLink g1 = Car (l[0]);
	return func->Func1 (*this, g1);
      }
     case 2:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Car (l[1]);
	return func->Func2 (*this, g1, g2);
      }
     case 3:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Car (l[1]);
	GCLink g3 = Car (l[2]);
	return func->Func3 (*this, g1, g2, g3);
      }
     case 4:
      {
	GCLink g1 = Car (l[0]);
	GCLink g2 = Car (l[1]);
	GCLink g3 = Car (l[2]);
	GCLink g4 = Car (l[3]);
	return func->Func4 (*this, g1, g2, g3, g4);
      }
     default:
      Error ("CallFSubr : Function argument number is too match");
    }
  }
  return 0; // dummy
}

Object*
Wool::GetFuncBind (Object* _fn, FuncKind& fk)
{
  GCLink fn = _fn;

  GCLink o = ((Atom*)((Object*)fn))->GetBind ();
  GCLink tag;
  GCLink t[5];
  t[0] = GetAtom ("subr");
  t[1] = GetAtom ("fsubr");
  t[2] = GetAtom ("expr");
  t[3] = GetAtom ("fexpr");
  t[4] = GetAtom ("macro");
  while (!NullP(o))
  {
    tag = Caar(o);
    for (int i = 0; i < 5; i++)
    {
      if (EqP (tag, t[i]))
      {
	fk = (FuncKind)i;
	return Car (o);
      }
    }
    o = Cdr (o);
  }
  return 0;
}

Object*
Wool::Eval (Object* _e, Object* _a)
{
  GCLink e = _e;
  GCLink a = _a;

  if (AtomP (e))
  {
    if (NullP (e) || !TypeP (e, WT_Atom))
    {
      return e;
    }
    GCLink o = Assoc (e, a);
    if (NullP (o))
    {
      GCLink apval = GetAtom ("apval");
      o = GetBind (e, apval);
      if (NullP (o))
      {
	Error ("Eval : Atom %s is void.",
	       ((Atom*)((Object*)e))->GetName ());
      }
      return Cdr (o);
    }
    return Cdr (o);
  }
  GCLink g0 = Car (e);
  if (AtomP (g0))
  {
    FuncKind fk;
    GCLink o = GetFuncBind (g0, fk);
    if (!NullP (o))
    {
      GCLink g1 = Cdr (o);
      switch (fk)
      {
       case FK_Subr:
	{
	  GCLink g2 = Cdr (e);
	  return CallSubr (g1, g2, a);
	}
       case FK_FSubr:
	{
	  GCLink g2 = Cdr (e);
	  return CallFSubr (g1, g2, a);
	}
       case FK_Expr:
	{
	  GCLink g2 = Cdr (e);
	  GCLink g3 = Evlis (g2, a);
	  return Apply (g1, g3, a);
	}
       case FK_FExpr:
	{
	  GCLink g2 = Cdr (e);
	  return Apply (g1, g2, a);
	}
       case FK_Macro:
	{
	  GCLink g2 = Cons (e, 0);
	  GCLink quote = GetAtom ("quote");
	  GCLink g3 = Cons (quote, g2);
	  GCLink g4 = Cons (g3, 0);
	  GCLink x =  Cons (g1, g4);
	  GCLink g5 = Eval (x, a);
	  return Eval (g5, a);
	}
      }
    }
  }
  GCLink g1 = Car (e);
  GCLink g2 = Cdr (e);
  GCLink g3 = Evlis (g2, a);
  return Apply (g1, g3, a);
}

Object*
Wool::Apply (Object* _fn, Object* _x, Object* _a)
{
  GCLink fn = _fn;
  GCLink x = _x;
  GCLink a = _a;

  if (NullP (fn))
  {
    return 0;
  }
  if (AtomP (fn))
  {
    if (!TypeP (fn, WT_Atom))
    {
      Error ("Apply : Can't Apply object.");
    }
    GCLink expr = GetAtom ("expr");
    GCLink o = GetBind (fn, expr);
    if (!NullP (o))
    {
      GCLink g1 = Cdr (o);
      return Apply (g1, x, a);
    }
    GCLink subr = GetAtom ("subr");
    o = GetBind (fn, subr);
    if (!NullP (o))
    {
      GCLink g1 = Cdr (o);
      return CallSubr (g1, x, a);
    }
    GCLink g1 = Eval (fn, a);
    return Apply (g1, x, a);
  }
  GCLink lambda = GetAtom  ("lambda");
  GCLink g1 = Car (fn);
  if (EqP (g1, lambda))
  {
    GCLink g1 = Caddr (fn);
    GCLink g2 = Cadr (fn);
    GCLink g3 = Pairlis (g2, x, a);
    return Eval (g1, g3);
  }
  GCLink label = GetAtom  ("label");
  if (EqP (g1, label))
  {
    GCLink g1 = Caddr (fn);
    GCLink g2 = Cadr (fn);
    GCLink g3 = Caddr (fn);
    GCLink g4 = Cons (g2, g3);
    GCLink g5 = Cons (g4, a);
    return Apply (g1, x, g5);
  }
  return 0;
}

Object*
Wool::Evcon (Object* _c, Object* _a)
{
  GCLink c = _c;
  GCLink a = _a;

  GCLink g1;
  while (!NullP (c))
  {
    g1 = Caar (c);
    GCLink g2 = Eval (g1, a); 
    if (!NullP(g2))
    {
      GCLink g3 = Cadar (c);
      return Eval (g3, a);
    }
    c = Cdr (c);
  }
  return 0;
}

Object*
Wool::Evlis (Object* _m, Object* _a)
{
  GCLink m = _m;
  GCLink a = _a;

  if (NullP (m))
  {
    return 0;
  }
  else
  {
    GCLink g1 = Car (m);
    GCLink car = Eval (g1, a);
    GCLink g2 = Cdr (m);
    GCLink cdr = Evlis (g2, a);
    return Cons (car, cdr);
  }
}

// (define '((foo (lambda (x) (car x)))))
Object*
Wool::Define (Object* _m)
{
  GCLink m = _m;

  GCLink ret = 0;
  while (!NullP (m))
  {
    GCLink o = Car (m);
    GCLink g1 = Car (o);
    GCLink expr = GetAtom ("expr");
    GCLink g2 = Cadr (o);
    SetBind (g1, expr, g2);
    m = Cdr (m);
    GCLink g3 = Car (o);
    ret = Cons (g3, ret);
  }
  return ret;
}

// (defun foo (x) (car x))
Object*
Wool::Defun (Object* _m)
{
  GCLink m = _m;
   
  GCLink g1 = Cadr (m); 
  GCLink fexpr = GetAtom ("fexpr"); 
  GCLink macro = GetAtom ("macro");
  GCLink lambda = GetAtom ("lambda"); 
  GCLink g4 = Car (m);
  if (TypeP (g1, WT_Atom) && (EqP (g1, fexpr) || EqP (g1, macro)))
  {
    GCLink g2 = Cddr (m);
    GCLink g3 = Cons (lambda, g2);
    SetBind (g4, g1, g3);
  }
  else
  {
    GCLink expr = GetAtom ("expr");
    GCLink g2 = Cdr (m);
    GCLink g3 = Cons (lambda, g2);
    SetBind (g4, expr, g3);
  }
  return g4;
}

Object*
Wool::Let (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  GCLink vals = Car (m);
  GCLink progs = Cdr (m);
  // setup local value
  while (!NullP (vals))
  {
    GCLink val = Car (vals);
    if (NullP (val))
    {
      Error ("Let : local value list includes nul"); 
    }
    if (TypeP_ (val, WT_Atom))
    {
      GCLink g1 = Cons (val, 0);
      alist = Cons (g1, alist);
    }
    else if (TypeP_ (val, WT_Pair))
    {
      GCLink g1 = Cadr (val);
      GCLink g2 = Eval (g1, alist);
      GCLink g3 = Car (val);
      GCLink g4 = Cons (g3, g2);
      alist = Cons (g4, alist);
    }
    else
    {
      Error ("Let : invalid local value list");
    }
    vals = Cdr (vals);
  }
  // exec prog
  GCLink ret = 0;
  while (!NullP (progs))
  {
    GCLink prog = Car (progs);
    ret = Eval (prog, alist);
    progs = Cdr (progs);
  }
  return ret;
}

Object*
Wool::Progn (Object* _progs, Object* _alist)
{
  GCLink progs = _progs;
  GCLink alist = _alist;

  GCLink ret = 0;
  while (!NullP (progs))
  {
    GCLink prog = Car (progs);
    ret = Eval (prog, alist);
    progs = Cdr (progs);
  }
  return ret;
}

Object*
Wool::Prog1 (Object* _progs, Object* _alist)
{
  GCLink progs = _progs;
  GCLink alist = _alist;

  GCLink ret = 0;
  int n = 0;
  while (!NullP (progs))
  {
    GCLink prog = Car (progs);
    if (++n == 1)
    {
      ret = Eval (prog, alist);
    }
    else
    {
      Eval (prog, alist);
    }
    progs = Cdr (progs);
  }
  return ret;
}

Object*
Wool::Prog2 (Object* _progs, Object* _alist)
{
  GCLink progs = _progs;
  GCLink alist = _alist;

  GCLink ret = 0;
  int n = 0;
  while (!NullP (progs))
  {
    GCLink prog = Car (progs);
    if (++n == 2)
    {
      ret = Eval (prog, alist);
    }
    else
    {
      Eval (prog, alist);
    }
    progs = Cdr (progs);
  }
  return ret;
}

Object*
Wool::While (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  GCLink test_exp = Car (m);
  GCLink form = Cdr (m);
  GCLink g1;
  while (((g1 = Eval (test_exp, alist)),
	  !NullP (g1)))
  {
    GCLink progs = form;
    while (!NullP (progs))
    {
      GCLink prog = Car (progs);
      Eval (prog, alist);
      progs = Cdr (progs);
    }
  }
  return 0;
}

Object*
Wool::If (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  GCLink g1 = Car (m);
  GCLink g2 = Eval (g1, alist);
  if (!NullP (g2))
  {
    g1 = Cadr (m);
    return Eval (g1, alist);
  }
  else
  {
    g1 = Cddr (m);
    if (!NullP (g1))
    {
      g1 = Caddr (m);
      return Eval (g1, alist);
    }
  }
  return 0;
}

