// -------------------------------------------------------------------- //
//                        Wool Object Oriented Lisp
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// wool0.cc:
//
// class Wool
// lexer, parser, print
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/wool/RCS/wool0.cc,v 1.12 1994/07/17 13:07:28 kudou Exp $

extern "C"
{
#include <stdio.h>		// for sscanf
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
}
#include "wool.h"
#include "flow.h"

#ifdef _MSC_VER
extern "C"
{
#define _JBLEN  9  /* bp, di, si, sp, ret addr, ds */
typedef  int  jmp_buf[_JBLEN];
int  __cdecl setjmp(jmp_buf);
void __cdecl longjmp(jmp_buf, int);
}
#endif

static jmp_buf env;		// for setjmp and longjmp

Wool::Wool (IFlow* i_flow, OFlow* o_flow, OFlow* err_flow)
: atom_table (this)
{
  Wool::i_flow = i_flow;
  Wool::o_flow = o_flow;
  Wool::err_flow = err_flow;
  scan_pos = line_end = read_buf;
  heap_top = new WoolHeap ();
  gc_message_mode = Show;
  run = False;
  eof_input = False;
  InitAtom ();
  InitSubr ();
}

Wool::~Wool ()
{
}

void
Wool::Printf (char* f, ...)
{
  va_list arg;
  va_start (arg, f);
  o_flow->Vprintf (f, arg);
  va_end (arg);
}

void
Wool::Warnning (char* f, ...)
{
  va_list arg;
  va_start (arg, f);
  err_flow->Printf ("WARNNING :");
  err_flow->Vprintf (f, arg);
  err_flow->Printf ("\n");
  va_end (arg);
}

Object*
Wool::Error (char* f, ...)
{
  va_list arg;
  va_start (arg, f);
  err_flow->Printf ("ERROR :");
  err_flow->Vprintf (f, arg);
  err_flow->Printf ("\n");
  va_end (arg);
  longjmp (env, 1);
  return 0;
}

bool 
Wool::ReadLine ()
{
  scan_pos = read_buf;
  if (i_flow->Gets (read_buf, read_buf_size) != 0)
  {
    line_end = &read_buf[strlen (read_buf)];
    return True;
  }
  else
  {
    return False;
  }
}

void
Wool::ReadByte (int n)
{
  scan_pos += n;
}

bool
Wool::ReadCheck ()
{
  if (scan_pos >= line_end)
  {
    bool eof_p = !ReadLine ();
    if (eof_p)
    {
      return False;
    }
  }
  return True;
}

void
Wool::YyLex (Token& token, GCLink& yylval)
{
  yylval = 0;
  while (ReadCheck ())
  {
    switch (*scan_pos)
    {
    case '(':
     token = TKN_LeftBrace;
     ReadByte (1);
     return;

    case ')':
     token = TKN_RightBrace;
     ReadByte (1);
     return;
     
    case '\'':
     token = TKN_Quote;
     ReadByte (1);
     return;
     
    case '"':
     token = TKN_Str;
     {
       int i = 1;
       for (;;)
       {
	 char c = scan_pos[i++];
	 if (c == '\0' || c == '"')
         {
           break;
         }
       }
       yylval = new (this) Str (scan_pos + 1, i - 2);
       ReadByte (i);
     }
     return;

    case ' ':
    case '\n':
    case '\r':
    case '\t':
     // white space
     ReadByte (1);
     break;

    case ';':
     // comment
     scan_pos = line_end;
     break;

    case '.':
     if (!isdigit(scan_pos[1]))
     {
       token = TKN_Priod;
       ReadByte (1);
       return;
     }

    case '+':
    case '-':
     if (!isdigit(scan_pos[1]))
     {
       token = TKN_Atom;
       yylval = atom_table.GetAtom (scan_pos, 1);
       ReadByte (1);
       return;
     }
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
     {
       bool num_p = True;
       bool fnum_p = False;
       bool hex_p = False;
       int i = 0;
       char c;
       for (;;)
       {
	 c = scan_pos[i++];
	 if (c == '.')
	 {
	   if (fnum_p || hex_p)
	   {
	     break;
	   }
	   fnum_p = True;
	 }
	 else if ((c == '-') || (c == '+'))
	 {
	   if (i != 1)
	   {
	     num_p = False;
	     break;
	   }
	 }
	 else if (c == 'x')
	 {
	   if ((i != 2) ||
	       (scan_pos[0] != '0') ||
	       (fnum_p))
	   {
	     num_p = False;
	     break;
	   }
	   hex_p = True;
	 }
	 else if (c == 'a' ||
		  c == 'b' ||
		  c == 'c' ||
		  c == 'd' ||
		  c == 'e' ||
		  c == 'f')
	 {
	   if (!hex_p)
	   {
	     num_p = False;
	     break;
	   }
	 }
	 else if (!isdigit (c))
	 {
	   if (c == '\0' ||
	       c == '(' ||
 	       c == ')' ||
	       c == '.' ||
	       c == '\'' ||
	       c == '"' ||
	       c == ' ' ||
	       c == '\n' ||
	       c == '\r' ||
	       c == ';')
            {
              break;
            }
            if (fnum_p)
            {
              break;	// ex.) 0.00GGG
            }
	    num_p = False;
	    break;
	  }
	}
	if (num_p)
	{
	  c = scan_pos[i];
	  scan_pos[i] = '\0';
	  if (hex_p)
	  {
	    int i;
	    sscanf (scan_pos, "%x", &i);
	    yylval = new (this) Num (i);
	    token = TKN_Num;
	  }
	  else if (fnum_p)
	  {
	    yylval = new (this) FNum (atof(scan_pos));
	    token = TKN_FNum;
	  }
	  else
	  {
	    yylval = new (this) Num (atol(scan_pos));
	    token = TKN_Num;
	  }
	  scan_pos[i] = c;
	  ReadByte (i - 1);
	  return;
	}
	// go down
      }
      
     default:
      // atom
      token = TKN_Atom;
      {
	int i = 0;
	for (;;)
	{
	  char c = scan_pos[i];
	  if (c == '\0' ||
	      c == '(' ||
              c == ')' ||
	      c == '.' ||
	      c == '\'' ||
              c == '"' ||
	      c == ' ' ||
	      c == '\n' ||
   	      c == '\r' ||
	      c == ';')
          {
	    break;
	  }
	   i++;
	 }
	 yylval = atom_table.GetAtom (scan_pos, i);
	 ReadByte (i);
       }
       return;
     }
  }
  token = TKN_Eof;
  eof_input = True;
}

#ifdef TEST_WOOL
// YyLex test
void
Wool::TestYyLex ()
{
  Token token;
  GCLink yylval;
  bool done = True;
  while (done)
  {
    YyLex (token, yylval);
    switch (token)
    {
    case Wool::TKN_LeftBrace:
      Printf ("(");
      break;
    case Wool::TKN_RightBrace:
      Printf (")");
      break;
    case Wool::TKN_Priod:
      Printf (".");
      break;
    case Wool::TKN_Quote:
      Printf ("'");
      break;
    case Wool::TKN_Atom:
    case Wool::TKN_Str:
    case Wool::TKN_Num:
    case Wool::TKN_FNum:
      yylval->Print (*this);
      break;
    case Wool::TKN_Eof:
      done = False;
      break;
    }
    Printf (" ");
  }
}
#endif /* TEST_WOOL */

Atom*
Wool::GetAtom (char* name)
{
  return atom_table.GetAtom (name);
}

Object* 
Wool::ReadSexp ()
{
  Token token;
  GCLink yylval;
  YyLex (token, yylval);
  switch (token)
  {
  case Wool::TKN_LeftBrace:	// "("
    return ReadLst ();
    
  case Wool::TKN_RightBrace:	// ")"
    Error ("ReadSexp: unexpected )");	// ERROR1
    return 0;
    
  case Wool::TKN_Priod:	// "."
    Error ("ReadSexp: unexpected .");	// ERROR1
    
  case Wool::TKN_Quote:
    return ReadQuote ();
    
  case Wool::TKN_Atom:
  case Wool::TKN_Str:
  case Wool::TKN_Num:
  case Wool::TKN_FNum:
    return yylval;
    
  case Wool::TKN_Eof:
    return 0;
  }
  return 0; // dummy
}

Object*
Wool::ReadLst ()
{
  Token token;
  GCLink yylval;
  YyLex (token, yylval);
  switch (token)
  {
  case Wool::TKN_LeftBrace:	// "("
    {
      GCLink car = ReadLst ();
      GCLink cdr = ReadLst ();
      return new (this) Pair (car, cdr);
    }
    
  case Wool::TKN_RightBrace:	// ")"
    return 0;
    
  case Wool::TKN_Priod:	// "."
    return ReadCdr ();
    
  case Wool::TKN_Quote:
    {
      GCLink car = ReadQuote ();
      GCLink cdr = ReadLst ();
      return new (this) Pair (car, cdr);
    }
    
  case Wool::TKN_Atom:
  case Wool::TKN_Str:
  case Wool::TKN_Num:
  case Wool::TKN_FNum:
    {
      GCLink cdr = ReadLst ();
      return new (this) Pair (yylval, cdr);
    }
    
   case Wool::TKN_Eof:
    return 0;
  }
  return 0; // dummy
}

Object*
Wool::ReadCdr ()
{
  Token token;
  GCLink yylval;
  YyLex (token, yylval);
  switch (token)
  {
   case Wool::TKN_LeftBrace:	// "("
    return ReadPar (ReadLst ());
    
   case Wool::TKN_RightBrace:	// ")"
    Error ("ReadCdr: unexpected )");		// ERROR 2
    
   case Wool::TKN_Priod:	// "."
    Error ("ReadCdr: unexpected .");		// ERROR 2
    
   case Wool::TKN_Quote:
    return ReadPar (ReadQuote ());
    
   case Wool::TKN_Atom:
   case Wool::TKN_Str:
   case Wool::TKN_Num:
   case Wool::TKN_FNum:
    return ReadPar (yylval);
    
   case Wool::TKN_Eof:
    return 0;
  }
  return 0; // dummy
}

Object*
Wool::ReadPar (GCLink x)
{
  Token token;
  GCLink yylval;
  YyLex (token, yylval);
  switch (token)
  {
   case Wool::TKN_RightBrace:	// ")"
    return x;
    
   case Wool::TKN_LeftBrace:	// "("
   case Wool::TKN_Priod:	// "."
   case Wool::TKN_Quote:
   case Wool::TKN_Atom:
   case Wool::TKN_Str:
   case Wool::TKN_Num:
   case Wool::TKN_FNum:
    Error ("ReadPar: unexpected ( or . or atom");	// ERROR3
    
   case Wool::TKN_Eof:
    return 0;
  }
  return 0; // dummy
}

Object*
Wool::ReadQuote ()
{
  GCLink s = ReadSexp ();
  GCLink car = atom_table.GetQuote ();
  GCLink cdr = new (this) Pair (s, 0);
  return new (this) Pair (car, cdr);
}

Object*
Wool::PrintSexp (Object* _x)
{
  GCLink x = _x;

  Print (x);
  Printf ("\n");
  return x;
}

void
Wool::Print (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    Printf ("nil");
  }
  else if (TypeP_ (x, WT_Pair))
  {
    Printf ("(");
    GCLink g1 = Car (x);
    Print (g1);
    g1 = Cdr (x);
    PrintCdr (g1);
    Printf (")");
  }
  else
  {
    ((Object*)x)->Print (*this);
  }
}

void
Wool::PrintCdr (Object* _x)
{
  GCLink x = _x;
  if (NullP (x))
  {
    ;
  }
  else if (AtomP (x))
  {
    Printf (".");
    ((Object*)x)->Print (*this);
  }
  else
  {
    Printf (" ");
    GCLink g1 = Car (x);
    Print (g1);
    g1 = Cdr (x);
    PrintCdr (g1);
  }
}

#ifdef TEST_WOOL
// Read and Print
void
Wool::TestReadPrint ()
{
  for (;;)
  {
    Printf ("\n-> ");
    GCLink x = ReadSexp ();
    if (x)
    {
      Print (x);
    }
    else
    {
      Printf ("Read Nil\n");
      return;
    }
  }
}
#endif /* TEST_WOOL */

void
Wool::Run ()
{
  run = True;
  while (run && !eof_input)
  {
    if (!setjmp (env))
    {
      Printf ("\n> ");
      GCLink o = ReadSexp ();
      o = Eval (o, 0);
      Print (o);
    }
    else
    {
      Printf ("eval error detected.");
      GCLink::ClearTop ();
    }
  }
  Printf ("\n");
}
