// -------------------------------------------------------------------- //
//                     Leaf class library for Windows
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// mdiedit.cpp:
//
// MDI editor sample
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/lw/sample/mdiedit/RCS/mdiedit.cpp,v 1.4 1994/07/17 13:06:04 kudou Exp $

#include "lw.h"
#include "mdiedit.h"
#include <string.h>

// -------------------------------------------------------------------- //
// calss MdiEdit
// -------------------------------------------------------------------- //
const int ProfButtonSize = 100;
const int MaxButtonID = 19;

ProfileEntry prof_ent [] =
{ // entry     default  id
  { "BUTTON00", "new", 0 },    
  { "BUTTON01", "open", 1 },
  { "BUTTON02", "save", 2 },
  { "BUTTON03", "saveas", 3 },
  { "BUTTON04", "close", 4 },
  { "BUTTON05", "exit", 5 },
  { "BUTTON06", "undo", 6 },
  { "BUTTON07", "cut", 7 },
  { "BUTTON08", "copy", 8 },
  { "BUTTON09", "paste", 9 },
  { "BUTTON10", "delete", 10 },
  { "BUTTON11", "cascade", 11 },
  { "BUTTON12", "arrange", 12 },
  { "BUTTON13", "tile", 13 },
  { "BUTTON14", "about", 14 },
  { "BUTTON15", "", 15 },
  { "BUTTON16", "", 16 },
  { "BUTTON17", "", 17 },
  { "BUTTON18", "", 18 },
  { "BUTTON19", "", 19 }, // MaxButtonID
  //
  { "BUTTONSIZE", "auto", ProfButtonSize },
  { 0, 0, 0 }
};

ButtonMenuId button_menu_id [] =
{
  { "new", MENU_ID_FILE_NEW },    
  { "open", MENU_ID_FILE_OPEN },
  { "save", MENU_ID_FILE_SAVE },
  { "saveas", MENU_ID_FILE_SAVE_AS },
  { "close", MENU_ID_FILE_CLOSE },
  { "exit", MENU_ID_FILE_EXIT },
  { "undo", MENU_ID_EDIT_UNDO },
  { "cut", MENU_ID_EDIT_CUT },
  { "copy", MENU_ID_EDIT_COPY },
  { "paste", MENU_ID_EDIT_PASTE },
  { "delete", MENU_ID_EDIT_DELETE },
  { "cascade", MENU_ID_WIN_CASCADE },
  { "arrange", MENU_ID_WIN_ARRANGE },
  { "tile", MENU_ID_WIN_TILE },
  { "about", MENU_ID_HELP_ABOUT },
  0,
};

MdiEdit::MdiEdit ()
: profile ("MDIEDIT", "MDIEDIT.INI", prof_ent)
{
  status_wnd = 0;
  toolbar_wnd = 0;
  docs = 0;
  current_doc = 0;
  serial_doc_num = 0;
}

MdiEdit::~MdiEdit ()
{
  delete status_wnd;
  delete toolbar_wnd;
}

HANDLE
MdiEdit::GetIcon ()
{
  return MSWin::LoadIcon ("MdiEditIco");
}

void
MdiEdit::Display (HDC hdc)
{
  DisplayGrayFrame (hdc, border_size);
}

void
MdiEdit::SetFocus ()
{
  if (current_doc)
  {
    current_doc->SetFocus ();
  }
}

void
MdiEdit::Resize (WORD left, WORD top, WORD right, WORD bottom)
{
  WORD statusbar_height = (GetStatusWnd () ?
			    GetStatusWnd ()->GetBarHeight () : 0);
  WORD toolbar_height = (GetToolBarWnd () ?
			  GetToolBarWnd ()->GetToolBarHeight () : 0);
  if (GetToolBarWnd ())
  {
    GetToolBarWnd ()->MoveWindow (border_size,
				  border_size,
				  right - left - border_size,
				  border_size + toolbar_height);
  }
  if (GetStatusWnd ())
  {
    GetStatusWnd ()->MoveWindow (border_size,
				 bottom - top - border_size - statusbar_height,
				 right - left - border_size,
				 bottom - top - border_size);
  }
  GetMdiClientWnd ()->MoveWindow (border_size,
				  border_size + toolbar_height,
				  right - left - border_size,
				  bottom - top - border_size - statusbar_height);
}

void
MdiEdit::OpenChildWindow ()
{
  toolbar_wnd = new ToolBarWnd (this);
  char* prof_button_size = profile.GetEntry (ProfButtonSize);
  if (strcmp (prof_button_size, "auto"))
  {
    if (!strcmp (prof_button_size, "small"))
    {
      toolbar_wnd->SetToolButtonSize (SmallBtn);
    }
    else if (!strcmp (prof_button_size, "middle"))
    {
      toolbar_wnd->SetToolButtonSize (MiddleBtn);
    }
    else if (!strcmp (prof_button_size, "large"))
    {
      toolbar_wnd->SetToolButtonSize (LargeBtn);
    }
  }
  for (int i = 0; i <= MaxButtonID; i++)
  {
    char* bttn_name = profile.GetEntry (i);
    toolbar_wnd->AddButton (bttn_name,
			    ButtonMenuId::GetMenuId (button_menu_id,
						     bttn_name));
  }
  toolbar_wnd->OpenWindow ();
  toolbar_wnd->OpenWindow ();

  status_wnd = new StatusWnd (this);
  status_wnd->OpenWindow ();

  MdiFrameWnd::OpenChildWindow ();
}

LPSTR 
MdiEdit::GetCaptionStr ()
{
  return "MdiEdit";
}

LPSTR
MdiEdit::GetMenuName ()
{
  return "MdiEditMenu";
}

void
MdiEdit::Close ()
{
  ::DestroyWindow (GetHandle ());
}
  
void
MdiEdit::Destroy ()
{
  ::PostQuitMessage (0);
}

long 
MdiEdit::WindowProc (UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
   case WM_CLOSE:
    Close ();
    break;
   
   case WM_DESTROY:
    Destroy ();
    break;
    
   case WM_MENUSELECT:
    MenuSelect (wparam);
    break;

   default:
    return MdiFrameWnd::WindowProc (msg, wparam, lparam);
  }
  return 0L;
}

void
MdiEdit::MenuSelect (WPARAM wparam)
{
  switch (wparam)
  {
   case MENU_ID_FILE_NEW:
   case MENU_ID_FILE_OPEN:
   case MENU_ID_FILE_SAVE:
   case MENU_ID_FILE_SAVE_AS:
   case MENU_ID_FILE_CLOSE:
   case MENU_ID_FILE_EXIT:
   case MENU_ID_EDIT_UNDO:
   case MENU_ID_EDIT_CUT:
   case MENU_ID_EDIT_COPY:
   case MENU_ID_EDIT_PASTE:
   case MENU_ID_EDIT_DELETE:
   case MENU_ID_WIN_CASCADE:
   case MENU_ID_WIN_ARRANGE:
   case MENU_ID_WIN_TILE:
   case MENU_ID_HELP_ABOUT:
    status_wnd->StrIdPrintf (wparam + MENU_MSG_OFFSET);
    break;

   default:
    status_wnd->ClearMsg ();
    break;
  }
}

uword
MdiEdit::Menu (WPARAM wparam, LPARAM lparam)
{
  uword ret = MF_ENABLED;
  bool menu_state = (lparam == GET_MENU_STATE);
  switch (wparam)
  {
   case MENU_ID_FILE_NEW:
    ret = NewFile (menu_state);
    break;

   case MENU_ID_FILE_OPEN:
    ret = OpenFile (menu_state);
    break;
    
   case MENU_ID_FILE_SAVE:
    ret = SaveFile (menu_state);
    break;

   case MENU_ID_FILE_SAVE_AS:
    ret = SaveAsFile (menu_state);
    break;

   case MENU_ID_FILE_CLOSE:
    ret = CloseFile (menu_state);
    break;

   case MENU_ID_FILE_EXIT:
    ret = BaseWnd::Exit (menu_state);
    break;

   case MENU_ID_EDIT_UNDO:
    ret = UndoEdit (menu_state);
    break;

   case MENU_ID_EDIT_CUT:
    ret = CutEdit (menu_state);
    break;

   case MENU_ID_EDIT_COPY:
    ret = CopyEdit (menu_state);
    break;

   case MENU_ID_EDIT_PASTE:
    ret = PasteEdit (menu_state);
    break;

   case MENU_ID_EDIT_DELETE:
    ret = DeleteEdit (menu_state);
    break;

   case MENU_ID_WIN_CASCADE:
    ret = CascadeWin (menu_state);
    break;
    
   case MENU_ID_WIN_ARRANGE:
    ret = ArrangeWin (menu_state);
    break;

   case MENU_ID_WIN_TILE:
    ret = TileWin (menu_state);
    break;

   case MENU_ID_HELP_ABOUT:
    ret = BaseWnd::About (menu_state);
    break;

   default:
    if (!menu_state)
    {
      return ::DefFrameProc (GetHandle (), GetMdiClientHandle (),
			     WM_COMMAND, wparam, lparam);
    }
    break;
  }
  return ret;
}

MdiChildWnd*
MdiEdit::OpenMdiChildWindow ()
{
  docs = new EditDoc (this, docs);
  docs->OpenWindow ();
  RStr doc_name (StrID_EditWnd_DefaultMdiFileName);
  wsprintf (docs->GetEditWnd ()->GetFileName (), doc_name, ++serial_doc_num);
  return docs;
}

uword
MdiEdit::GetEditDocNumber ()
{
  uword n = 0;
  for (EditDoc* d = docs; d != 0; d = d->GetNext ())
  {
    n++;
  }
  return n;
}

uword
MdiEdit::NewFile (bool menu_state)
{
  if (menu_state)
  {
    if (GetEditDocNumber () >= max_edit_doc)
    {
      return MF_GRAYED | MF_DISABLED;
    }
    return MF_ENABLED;
  }
  if (GetEditDocNumber () >= max_edit_doc)
  {
    return 0;
  }
  EditDoc* doc = (EditDoc*)OpenMdiChildWindow ();
  doc->GetEditWnd ()->UpdateCaption ();
  return 0;
}

uword
MdiEdit::OpenFile (bool menu_state)
{
  if (menu_state)
  {
    if (GetEditDocNumber () >= max_edit_doc)
    {
      return MF_GRAYED | MF_DISABLED;
    }
    return MF_ENABLED;
  }
  if (GetEditDocNumber () >= max_edit_doc)
  {
    return 0;
  }
  EditDoc* doc = (EditDoc*)OpenMdiChildWindow ();
  doc->GetEditWnd ()->OpenFile (menu_state);
  return 0;
}

uword
MdiEdit::SaveFile (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->SaveFile (menu_state);
}

uword
MdiEdit::SaveAsFile (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->SaveAsFile (menu_state);
}

uword
MdiEdit::CloseFile (bool menu_state)
{
  if (menu_state)
  {
    if (current_doc == 0)
    {
      return MF_DISABLED | MF_GRAYED;
    }
    else
    {
      return MF_ENABLED;
    }
  }
  if (current_doc)
  {
    current_doc->PostMessage (WM_CLOSE, 0, 0);
  }
  return 0;
}

uword
MdiEdit::UndoEdit (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->UndoEdit (menu_state);
}

uword
MdiEdit::CutEdit (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->CutEdit (menu_state);
}

uword
MdiEdit::CopyEdit (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->CopyEdit (menu_state);
}

uword
MdiEdit::PasteEdit (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->PasteEdit (menu_state);
}

uword
MdiEdit::DeleteEdit (bool menu_state)
{
  if (current_doc == 0)
  {
    return MF_DISABLED | MF_GRAYED;
  }
  return current_doc->GetEditWnd ()->DeleteEdit (menu_state);
}

uword
MdiEdit::CascadeWin (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  GetMdiClientWnd ()->SendMessage (WM_MDICASCADE, 0, 0L);
  return 0;
}

uword
MdiEdit::ArrangeWin (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  GetMdiClientWnd ()->SendMessage (WM_MDIICONARRANGE, 0, 0L);
  return 0;
}

uword
MdiEdit::TileWin (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  GetMdiClientWnd ()->SendMessage (WM_MDITILE, 0, 0L);
  return 0;
}

HMENU
MdiEdit::GetWindowMenu ()
{
  return ::GetSubMenu (::GetMenu(GetHandle ()), window_menu_pos);
}

void
MdiEdit::RemoveEditDoc (EditDoc* edit_doc)
{
  if (edit_doc == docs)
  {
    docs = edit_doc->GetNext ();
  }
  else
  {
    EditDoc* prev = docs;
    for (EditDoc* ed = docs->GetNext (); ed != 0; ed = ed->GetNext ())
    {
      if (ed == edit_doc)
      {
	prev->SetNext (ed->GetNext ());
	break;
      }
      prev = ed;
    }
  }
  if (current_doc == edit_doc)
  {
    current_doc = 0;
  }
}

// -------------------------------------------------------------------- //
// calss MdiDoc
// -------------------------------------------------------------------- //
EditDoc::EditDoc (MdiEdit* mdi_frame_wnd, EditDoc* next)
: MdiChildWnd (mdi_frame_wnd)
{
  edit_wnd = 0;
  EditDoc::next = next;
}

EditDoc::~EditDoc ()
{
  delete edit_wnd;
}

DWORD
EditDoc::GetCreateWindowStyle ()
{
  return WS_MAXIMIZEBOX | WS_MINIMIZEBOX;
}

void
EditDoc::OpenChildWindow ()
{
  edit_wnd = new EditWnd (this);
  edit_wnd->OpenWindow ();
}

LPSTR
EditDoc::GetCaptionStr ()
{
  return edit_wnd->GetFileName ();
}

void
EditDoc::Resize (WORD left, WORD top, WORD right, WORD bottom)
{
  if (GetEditWnd ())
  {
    GetEditWnd ()->MoveWindow (0,
			       0,
			       right - left,
			       bottom - top);
  }
}

void
EditDoc::SetFocus ()
{
  GetEditWnd ()->SetFocus ();
  GetMdiEdit ()->SetCurrentEditDoc (this);
}

long
EditDoc::WindowProc (UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
   case WM_NCDESTROY:
    ((MdiEdit*)GetMdiFrameWnd ())->RemoveEditDoc (this);
    delete this;
    return 0L;

   case WM_CLOSE:
    if (!edit_wnd->CloseFile (0))
    {
      return 0L;      
    }
    break;

   case WM_COMMAND:
    if ((wparam == WndID_EditWnd) && 
	(edit_wnd->GetHandle () ==  LOWORD (lparam)))
    {
      return edit_wnd->WindowProc (msg, wparam, lparam);
    }
    return 0L;

   case WM_CHILDACTIVATE:
    if (wparam)
    {
      ((MdiEdit*)GetMdiFrameWnd ())->SetCurrentEditDoc (this);
    }
    break;

   default:
    break;
  }
  return MdiChildWnd::WindowProc (msg, wparam, lparam);
}

// -------------------------------------------------------------------- //
// Main
// -------------------------------------------------------------------- //
int
Main ()
{
  MdiEdit wnd;
  wnd.OpenWindow ();
  return MSWin::MessageLoop ();
}
