// -------------------------------------------------------------------- //
//                     Leaf class library for Windows
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// editwnd.cpp:
//
// class EditWnd
// -------------------------------------------------------------------- //
// $Header: /a/googol/d/1/proj/egypt/0/lw/src/RCS/editwnd.cpp,v 1.14 1994/05/08 16:43:41 kudou Exp $

#include "lw.h"
#include "filename.h"

#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <limits.h>

EditWnd::EditWnd (Wnd* parent)
{
  EditWnd::parent = parent;
  window_id = WndID_EditWnd;	// default

  h_lock_buf = 0;
  lock_buf_size = 0;
  lock_buf = 0;

  file_name_set_p = False;
  SetDefaultFileName ();
  ClearDirty ();
}

EditWnd::~EditWnd ()
{
  ;
}

void
EditWnd::SetDefaultFileName ()
{
  RStr fname(StrID_EditWnd_DefaultFileName);
  if (fname != 0)
  {
    strncpy (file_name, fname, max_path_length);
  }
}

HWND
EditWnd::GetParent ()
{
  return parent ? parent->GetHandle () : 0;
}

Wnd*
EditWnd::GetParentWnd ()
{
  return parent;
}

DWORD
EditWnd::GetCreateWindowStyle ()
{
  return (WS_CHILD | WS_VISIBLE | ES_MULTILINE | WS_VSCROLL | WS_HSCROLL);
}

void 
EditWnd::OpenWindow ()
{
  SetHandle (::CreateWindow ("EDIT",	// Edit Control window class name
			     NULL,
			     GetCreateWindowStyle (),
			     0,
			     0,
			     0,
			     0,
			     GetParent (),
			     window_id,
			     MSWin::GetInstance (),
			     0));
}

void 
EditWnd::SetFocus ()
{
  ::SetFocus (handle);
}

long 
EditWnd::WindowProc (UINT /*msg*/, WPARAM /*wparam*/, LPARAM lparam)
{
  WORD hi = HIWORD (lparam);
  switch (hi)
  {
   case EN_ERRSPACE:
    MessageBox (StrID_EditWnd_Err, StrID_EditWnd_LessMemory,
		MB_ICONHAND | MB_OK);
    break;
   case EN_MAXTEXT:
    MessageBox (StrID_EditWnd_Err, StrID_EditWnd_MaxText,
		MB_ICONHAND | MB_OK);
    break;
     
   case EN_CHANGE:
    SetDirty ();
    break;
     
   default:
    return 0;
  }
  return 1;
}

uword
EditWnd::OpenFile (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  if (GetDirty ())
  {
    if (!CloseFile (0))
    {
      return False;
    }
  }
  
  FileNameDlg dlg (file_name_set_p ? file_name : "*.*", this);
  if (dlg.Go ())
  {
    // file read
    int fh;
    if ((fh = open (dlg.GetFileName (), 
		    O_RDONLY | O_BINARY, S_IWRITE | S_IREAD)) == -1)
    {
      // open error
      MessageBox (dlg.GetFileName (),
		  StrID_EditWnd_CantOpen,
		  MB_OK | MB_ICONHAND);
      return False;
    }
    file_name_set_p = True;
    struct stat f_status;
    fstat (fh, &f_status);
    int read_size = (int)f_status.st_size;
    if (f_status.st_size >= INT_MAX)
    {
      // file size check
      read_size = INT_MAX - 1;
      MessageBox (dlg.GetFileName (),
		  StrID_EditWnd_TooBig,
		  MB_OK | MB_ICONHAND);
    }
    HANDLE h_read_buf = ::LocalAlloc (LMEM_MOVEABLE, read_size + 1);
    // get read buffer
    if (!h_read_buf)
    {
      // less memory
      MessageBox (dlg.GetFileName (),
		  StrID_EditWnd_ReadLessMemory,
		  MB_OK | MB_ICONHAND);
      close (fh);
      return False;
    }
    char* read_buf = ::LocalLock (h_read_buf);
    int read_status = read (fh, read_buf, read_size);
    read_buf[read_size] = '\0';
    LocalUnlock (h_read_buf);
    close (fh);
    if (read_status == -1)
    {
      // read error
      MessageBox (dlg.GetFileName (),
		  StrID_EditWnd_ReadError,
		  MB_OK | MB_ICONHAND);
    }
    HANDLE h_old_edit = SendMessage (EM_GETHANDLE, 0, 0l);
    ::LocalFree (h_old_edit);
    SendMessage (EM_SETHANDLE, h_read_buf, 0l);
    
    // update file name
    strncpy (file_name, dlg.GetFileName (), max_path_length);
    UpdateCaption ();
    ClearDirty ();
  }
  return True;
}

void
EditWnd::UpdateCaption ()
{
  if (GetParentWnd ())
  {
    GetParentWnd ()->SetWindowText (GetFileName ());
  }
  else
  {
    SetWindowText (GetFileName ());
  }
}

uword
EditWnd::SaveFile (bool menu_state)
{
  if (!file_name_set_p)
  {
    return SaveAsFile (menu_state);
  }
  
  if (menu_state)
  {
    if (GetDirty () &&
       (*file_name != '\0'))
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  
  int fh;
  if ((fh = open (GetFileName (),
		  O_CREAT | O_WRONLY | O_BINARY | O_TRUNC,
		  S_IWRITE | S_IREAD)) == -1)
  {
    MessageBox (GetFileName (),
		StrID_EditWnd_SaveOpenErr,
		MB_OK | MB_ICONHAND);
    return False;
  }
  HANDLE h_edit_buf = SendMessage (EM_GETHANDLE, 0, 0l);
  char* edit_buf   = ::LocalLock (h_edit_buf);
  write (fh, edit_buf, strlen(edit_buf));
  close (fh);
  ::LocalUnlock (h_edit_buf);
  ClearDirty ();
  return True;
}

uword
EditWnd::SaveAsFile (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }

  FileNameDlg dlg (file_name_set_p ? file_name : "*.*", this);  
  if (dlg.Go (True))
  {
    int fh;
    if ((fh = open (dlg.GetFileName (),
		    O_CREAT | O_WRONLY | O_BINARY | O_TRUNC,
		    S_IWRITE | S_IREAD)) == -1)
    {
      MessageBox (dlg.GetFileName (),
		  StrID_EditWnd_SaveAsOpenErr,
		  MB_OK | MB_ICONHAND);
      return False;
    }
    file_name_set_p = True;
    HANDLE h_edit_buf = SendMessage (EM_GETHANDLE, 0, 0l);
    char* edit_buf   = ::LocalLock (h_edit_buf);
    write (fh, edit_buf, strlen(edit_buf));
    close (fh);
    ::LocalUnlock (h_edit_buf);
    
    // update file name
    strncpy (file_name, dlg.GetFileName (), max_path_length);
    UpdateCaption ();
    ClearDirty ();
  }
  return True;
}

// cleate document
char* 
EditWnd::LockBuffer ()
{
  h_lock_buf = SendMessage (EM_GETHANDLE, 0, 0l);
  lock_buf = ::LocalLock (h_lock_buf);
  lock_buf_size = strlen (lock_buf);
  return lock_buf;
}

void 
EditWnd::UnlockBuffer ()
{
  if (h_lock_buf)
  {
    ::LocalUnlock (h_lock_buf);
    h_lock_buf = 0;
    lock_buf = 0;
    lock_buf_size = 0;
  }
}

uword
EditWnd::CloseFile (bool menu_state)
{
  if (menu_state)
  {
    return MF_ENABLED;
  }
  
  if (dirty)
  {
    int ret = MessageBoxPrintf (StrID_EditWnd_CloseFileMsg,
				file_name,
				MB_YESNOCANCEL | MB_DEFBUTTON1, file_name);
    switch (ret)
    {
     case IDOK:
     case IDYES:
      break;
     case IDNO:
      return True;
     case IDCANCEL:
      return False;
    }
    return SaveFile (0);
  }
  return True;
}

uword
EditWnd::CutEdit (bool menu_state)
{
  if (menu_state)
  {
    LONG l = SendMessage (EM_GETSEL, 0, 0L);
    if (HIWORD (l) != LOWORD (l))
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  SendMessage (WM_CUT, 0, 0L);
  return 0;
}

uword
EditWnd::CopyEdit (bool menu_state)
{
  if (menu_state)
  {
    LONG l = SendMessage (EM_GETSEL, 0, 0L);
    if (HIWORD (l) != LOWORD (l))
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  SendMessage (WM_COPY, 0, 0L);
  return 0;
}

uword
EditWnd::PasteEdit (bool menu_state)
{
  if (menu_state)
  {
    bool empty = True;
    if (::OpenClipboard (GetHandle ()))
    {
      HANDLE h = ::GetClipboardData (CF_TEXT);
      if (h)
      {
	empty = False;
      }
      ::CloseClipboard ();
    }
    if (!empty)
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  SendMessage (WM_PASTE, 0, 0L);
  return 0;
}

uword
EditWnd::DeleteEdit (bool menu_state)
{
  if (menu_state)
  {
    LONG l = SendMessage (EM_GETSEL, 0, 0L);
    if (HIWORD (l) != LOWORD (l))
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  SendMessage (WM_CLEAR, 0, 0L);
  return 0;
}

uword
EditWnd::UndoEdit (bool menu_state)
{
  if (menu_state)
  {
    if (SendMessage (EM_CANUNDO, 0, 0L))
    {
      return MF_ENABLED;
    }
    else
    {
      return MF_DISABLED | MF_GRAYED;
    }
  }
  SendMessage (WM_UNDO, 0, 0L);
  return 0;
}

