// -------------------------------------------------------------------- //
//                     Leaf class library for Windows
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// textwnd.cpp:
//
// class TextWnd
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/lw/src/RCS/textwnd.cpp,v 1.7 1994/07/17 13:05:06 kudou Exp $

#include "lw.h"
#include <memory.h>
#include <string.h>
#include <stdio.h> /* for vsprintf */

#define CR 0x0d
#define LF 0x0a
#define BS 0x08
#define DEL 0x7f

TextWnd::TextWnd (uword column_size, uword line_size)
{
  TextWnd::column_size = column_size;
  TextWnd::line_size = line_size;
  caret_column_pos = 0;
  caret_line_pos = 0;
  key_buffer_ptr = 0;
  vscroll_pos = 0;
  hscroll_pos = 0;
  set_vscroll_bar_p = False;
  set_hscroll_bar_p = False;
  tty_mode = RawMode;
  cooked_p = False;
  cook_buffer_ptr = 0;
  kanji_putchar_buf = 0;
  buf = new char[column_size * line_size];
  Clear ();
  SetFont ();
}

TextWnd::~TextWnd ()
{
  delete buf;
}
	  
void
TextWnd::Clear ()
{
  memset (buf, ' ', column_size * line_size);
}

long
TextWnd::WindowProc (UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
   case WM_KILLFOCUS:
    KillFocus ();
    break;

   case WM_GETMINMAXINFO:
    GetMinMaxInfo ((MINMAXINFO FAR*)lparam);
    break;
    
   case WM_CHAR:
    KeyIn (wparam, lparam);
    break;

   case WM_HSCROLL:
    HScroll (wparam, lparam);
    break;

   case WM_VSCROLL:
    VScroll (wparam, lparam);
    break;

   default:
    return BaseWnd::WindowProc (msg, wparam, lparam);
  }
  return 0L;
}

void
TextWnd::SetFont ()
{
  HWND dsktop = ::GetDesktopWindow ();
  TEXTMETRIC metric;
  HDC dc = ::GetDC (dsktop);
  SelectObject (dc, GetStockObject (SYSTEM_FIXED_FONT));
  ::GetTextMetrics (dc, &metric);
  font_width = metric.tmAveCharWidth;
  font_height = metric.tmHeight;
  ::ReleaseDC (dsktop, dc);
}

LPSTR
TextWnd::GetCaptionStr ()
{
  return "TextWnd";
}

DWORD
TextWnd::GetCreateWindowStyle ()
{
  return (WS_OVERLAPPEDWINDOW | WS_MAXIMIZEBOX | WS_MINIMIZEBOX |
	  WS_HSCROLL | WS_VSCROLL);
}

int 
TextWnd::GetCreateWindowXSize ()
{
  return (column_size * font_width +
	  ::GetSystemMetrics (SM_CXFRAME) * 2);
}

int 
TextWnd::GetCreateWindowYSize ()
{
  return (line_size * font_height +
	  ::GetSystemMetrics (SM_CYCAPTION) +
	  ::GetSystemMetrics (SM_CYFRAME) * 2);
}

void
TextWnd::GetMinMaxInfo (MINMAXINFO* info)
{
  info->ptMaxSize.x = GetCreateWindowXSize ();
  info->ptMaxSize.y = GetCreateWindowYSize ();
  info->ptMaxPosition.x = 0;
  info->ptMaxPosition.y = 0;
  info->ptMinTrackSize.x = ::GetSystemMetrics (SM_CXMIN);
  info->ptMinTrackSize.y = ::GetSystemMetrics (SM_CYMIN);
  info->ptMaxTrackSize.x = info->ptMaxSize.x;
  info->ptMaxTrackSize.y = info->ptMaxSize.y;
}

void
TextWnd::Display (Paint& pnt)
{
  ::HideCaret (GetHandle ());
  HDC dc = pnt.GetDC ();
  RECT rec;
  pnt.GetPaintRECT (&rec);
  COLORREF old_bk_col = ::SetBkColor (dc, ::GetSysColor (COLOR_WINDOW));
  COLORREF old_txt_col = ::SetTextColor (dc, ::GetSysColor (COLOR_WINDOWTEXT));
  HGDIOBJ old_fnt = ::SelectObject (dc, ::GetStockObject (SYSTEM_FIXED_FONT));
  uword line_from = rec.top / font_height + vscroll_pos;
  uword line_to = rec.bottom / font_height + vscroll_pos;
  if (line_from >= line_size)
  {
    line_from = line_size - 1;
  }
  if (line_to >= line_size)
  {
    line_to = line_size - 1;
  }
  int x_offset = - hscroll_pos * font_width;
  for (uword line = line_from; line <= line_to; line++)
  {
    char* l = GetLine (line);
    if (l[column_size - 1] != DEL)
    {
      ::TextOut (dc, x_offset, (line - vscroll_pos) * font_height,
		 l, column_size);
    }
    else
    {
      ::TextOut (dc, x_offset, (line - vscroll_pos) * font_height,
		 l, column_size - 1);
    }
  }
  ::SelectObject (dc, old_fnt);
  ::SetTextColor (dc, old_txt_col);
  ::SetBkColor (dc, old_bk_col);
  ::ShowCaret (GetHandle ());
}

void
TextWnd::SetFocus ()
{
  ::CreateCaret (GetHandle (), NULL, 1, font_height);
  ::SetCaretPos ((caret_column_pos - hscroll_pos) * font_width,
		 (caret_line_pos - vscroll_pos) * font_height);
  ::ShowCaret (GetHandle ());
}

void
TextWnd::KillFocus ()
{
  ::HideCaret (GetHandle ());
  ::DestroyCaret ();
}

inline word
Max (word a, word b)
{
  return (a > b) ? a : b;
}

inline word
Min (word a, word b)
{
  return (a < b) ? a : b;
}

void
TextWnd::Resize (WORD left, WORD top, WORD right, WORD bottom)
{
  if (!set_vscroll_bar_p)
  {
    word max_vscroll_pos = Max (line_size -
				GetWindowYSize () / font_height, 0);
    vscroll_pos = Max (0, Min (vscroll_pos, max_vscroll_pos)); 
    int h_min, h_max;
    ::GetScrollRange (GetHandle (), SB_HORZ, &h_min, &h_max);
    if (!((h_max == 0) && (h_min == 0)) &&
	(bottom - top + ::GetSystemMetrics (SM_CYHSCROLL)) / font_height >=
	line_size)
    {
      max_vscroll_pos = 0;
      vscroll_pos = 0;
    }
    set_vscroll_bar_p = True;
    ::SetScrollRange (GetHandle (), SB_VERT, 0, max_vscroll_pos, FALSE);
    ::SetScrollPos (GetHandle (), SB_VERT, vscroll_pos, TRUE);
    set_vscroll_bar_p = False;
  }
  if (!set_hscroll_bar_p)
  {
    word max_hscroll_pos = Max (column_size - 
				GetWindowXSize () / font_width, 0);
    hscroll_pos = Max (0, Min (hscroll_pos, max_hscroll_pos));
    int v_min, v_max;
    ::GetScrollRange (GetHandle (), SB_VERT, &v_min, &v_max);
    if (!((v_max == 0) && (v_min == 0)) &&
	(right - left + ::GetSystemMetrics (SM_CXVSCROLL)) / font_width >= 
	column_size)
    {
      max_hscroll_pos = 0;
      hscroll_pos = 0;
    }
    set_hscroll_bar_p = True;
    ::SetScrollRange (GetHandle (), SB_HORZ, 0, max_hscroll_pos, FALSE);
    ::SetScrollPos (GetHandle (), SB_HORZ, hscroll_pos, TRUE);
    set_hscroll_bar_p = False;
  }
  TouchCaret ();
}

void
TextWnd::HScroll (WPARAM wparam, LPARAM lparam)
{
  switch (wparam)
  {
   case SB_LINEUP:
    hscroll_pos--;
    break;
   case SB_LINEDOWN:
    hscroll_pos++;
    break;
   case SB_PAGEUP:
    hscroll_pos -= (GetWindowXSize () / font_width) / 2;
    break;
   case SB_PAGEDOWN:
    hscroll_pos += (GetWindowXSize () / font_width) / 2;
    break;
   case SB_THUMBPOSITION:
    hscroll_pos = LOWORD (lparam);
    break;
  }
  word max_hscroll_pos = Max (column_size - GetWindowXSize () / font_width, 0);
  hscroll_pos = Max (0, Min (hscroll_pos, max_hscroll_pos));
  if (hscroll_pos != ::GetScrollPos (GetHandle (), SB_HORZ))
  {
    ::SetScrollPos (GetHandle (), SB_HORZ, hscroll_pos, TRUE);
    ::InvalidateRect (GetHandle (), NULL, TRUE);
    TouchCaret ();
  }
}

void
TextWnd::VScroll (WPARAM wparam, LPARAM lparam)
{
  switch (wparam)
  {
   case SB_LINEUP:
    vscroll_pos--;
    break;
   case SB_LINEDOWN:
    vscroll_pos++;
    break;
   case SB_PAGEUP:
    vscroll_pos -= (GetWindowYSize () / font_height) / 2;
    break;
   case SB_PAGEDOWN:
    vscroll_pos += (GetWindowYSize () / font_height) / 2;
    break;
   case SB_THUMBPOSITION:
    vscroll_pos = LOWORD (lparam);
    break;
  }
  word max_vscroll_pos = Max (line_size - GetWindowYSize () / font_height, 0);
  vscroll_pos = Max (0, Min (vscroll_pos, max_vscroll_pos)); 
  if (vscroll_pos != ::GetScrollPos (GetHandle (), SB_VERT))
  {
    ::SetScrollPos (GetHandle (), SB_VERT, vscroll_pos, TRUE);
    ::InvalidateRect (GetHandle (), NULL, TRUE);
    TouchCaret ();
  }
}

void
TextWnd::KeyIn (WPARAM wparam, LPARAM /*lparam*/)
{
  if (key_buffer_ptr < key_buffer_size)
  {
    char c = (char)wparam;
    if (c == CR)
    {
      c = LF;
    }
    else if (c == LF)
    {
      c = CR;
    }
    key_buffer[key_buffer_ptr++] = c;
  }
  else
  {
    ::MessageBeep (MB_OK);
  }
}

char 
TextWnd::GetKeyBufferChar ()
{
  char ret = key_buffer[0];
  memmove (&key_buffer[0], &key_buffer[1], --key_buffer_ptr);
  return ret;
}

void
TextWnd::CookIn (char c)
{
  if (cook_buffer_ptr < cook_buffer_size)
  {
    cook_buffer[cook_buffer_ptr++] = c;
    PutChar (c);
  }
  else
  {
    ::MessageBeep (MB_OK);
  }
}

char 
TextWnd::GetCookBufferChar ()
{
  char ret = cook_buffer[0];
  memmove (&cook_buffer[0], &cook_buffer[1], --cook_buffer_ptr);
  if (ret == LF)
  {
    char* p = cook_buffer;
    for (uword i = 0; i < cook_buffer_ptr; i++)
    {
      if (*p == LF)
      {
	break;
      }
    }
    if (i == cook_buffer_ptr)
    {
      cooked_p = False;
    }
  }
  return ret;
}

void
TextWnd::BackwardChar ()
{
  if (caret_column_pos >= 1)
  {
    caret_column_pos --;
  }
  else
  {
    if (caret_line_pos >= 1)
    {
      caret_line_pos --;
      caret_column_pos = column_size - 1;
    }
    else
    {
      caret_column_pos = 0;
    }
  }
}

void
TextWnd::CookBS ()
{
  if (cook_buffer_ptr >= 1)
  {
    if ((caret_column_pos == 0) && (caret_line_pos != 0))
    {
      // for line end kanji
      char* l = GetLine (caret_line_pos - 1);
      if (l[column_size - 1] == DEL)
      {
	BackwardChar ();
	PutChar (' ');
	BackwardChar ();
      }
    }
    if ((cook_buffer_ptr >= 2) &&
	Kanji2P(cook_buffer[cook_buffer_ptr - 1]) &&
	Kanji1P(cook_buffer[cook_buffer_ptr - 2]))
    {
      cook_buffer_ptr -= 2;
      BackwardChar ();
      PutChar (' ');
      BackwardChar ();
    }
    else
    {
      cook_buffer_ptr--;
    }
    BackwardChar ();
    PutChar (' ');
    BackwardChar ();
    TouchCaret ();
  }
  else
  {
    ::MessageBeep (MB_OK);
  }
}

char
TextWnd::Cook ()
{
  if (cooked_p && (cook_buffer_ptr > 0))
  {
    return GetCookBufferChar ();
  }
  for (;;)
  {
    if (key_buffer_ptr > 0)
    {
      char c = GetKeyBufferChar ();
      if (c == BS)
      {
	CookBS ();
      }
      else
      {
	CookIn (c);
	if (c == LF)
	{
	  cooked_p = True;
	  return GetCookBufferChar ();
	}
      }
    }
    if (!MSWin::OneMessage ())
    {
      return EOF;
    }
  }
}

char
TextWnd::GetChar ()
{
  if (tty_mode == RawMode)
  {
    for (;;)
    {
      if (key_buffer_ptr > 0)
      {
	return GetKeyBufferChar ();
      }
      if (!MSWin::OneMessage ())
      {
	return EOF;
      }
    }
  }
  else
  {
    return Cook ();
  }
}

char*
TextWnd::Gets (char* read_buf, size_t read_buf_size)
{
  char* p = read_buf;
  for (size_t s = 0; s < read_buf_size; s++)
  {
    char c = GetChar ();
    if (c == EOF)
    {
      return 0;
    }
    if (c == LF)
    {
      *p = '\0';
      break;
    }
    *p++ = c;
  }
  return read_buf;
}

void
TextWnd::Printf (char* fmt, ...)
{
  va_list a;
  va_start (a, fmt);
  Vprintf (fmt, a);
  va_end (a);
}

void
TextWnd::Vprintf (char* fmt, va_list arg)
{
  static char buffer[1024];
  // windows wvsprintf doesn't support "%f"! Why?
//  wvsprintf (buffer, fmt, arg);
  vsprintf (buffer, fmt, arg);
  char* p = buffer;
  while (*p)
  {
    _PutChar (*p++);
  }
  TouchCaret ();
}

void
TextWnd::Puts (char* p)
{
  while (*p)
  {
    _PutChar (*p++);
  }
  TouchCaret ();
}

void
TextWnd::Puts (char* p, size_t size)
{
  for (size_t i = 0; i < size; i++)
  {
    _PutChar (*p++);
  }
  TouchCaret ();
}

void
TextWnd::PutChar (char c)
{
  _PutChar (c);
  TouchCaret ();
}

void
TextWnd::_PutChar (char c)
{
  switch (c)
  {
   case LF:
    if (++caret_line_pos >= line_size)
    {
      ScrollUp ();
      caret_line_pos = line_size - 1;
    }
    /* go down */

   case CR:
    caret_column_pos = 0;
    break;
    
   default:
    __PutChar (c);
    break;
  }
}

KanjiStatus
TextWnd::CheckKanji (char* buf, uword pos)
{
  KanjiStatus sta = NotKanji;
  char* p = buf;
  for (uword i = 0; i <= pos; i++)
  {
    char c = *p++;
    if ((sta == KanjiFirstByte) &&
	Kanji2P (c))
    {
      sta = KanjiSecondByte;
    }
    else if (Kanji1P (c))
    {
      sta = KanjiFirstByte;
    }
    else
    {
      sta = NotKanji;
    }
  }
  return sta;
}

void
TextWnd::__PutChar (char c)
{
  if (kanji_putchar_buf == 0)
  {
    if (Kanji1P (c))
    {
      // kanji 1 byte
      kanji_putchar_buf = c;
      return;
    }
  }
  else
  {
    if (Kanji2P(c))
    {
      // kanji 2 byte
      char* l = GetLine(caret_line_pos);
      if (caret_column_pos + 1 == column_size)
      {
	// for line end kanji
	l[caret_column_pos] = DEL;
	InvalidateCaretChar ();
	caret_column_pos = 0;
	if (++caret_line_pos >= line_size)
	{
	  ScrollUp ();
	  caret_line_pos = line_size - 1;
	}
      }
      l = GetLine(caret_line_pos);
      switch (CheckKanji (l, caret_column_pos))
      {
       case KanjiFirstByte:
	break;
       case KanjiSecondByte:
	l[caret_column_pos - 1] = ' ';
	InvalidateChar (caret_column_pos - 1, caret_line_pos);
	/* go down */
       case NotKanji:
	if (CheckKanji (l, caret_column_pos + 1) == KanjiFirstByte)
	{
	  l[caret_column_pos + 2] = ' ';
	  InvalidateChar (caret_column_pos + 2, caret_line_pos);
	}
	break;
      }
      l[caret_column_pos] = kanji_putchar_buf;
      l[caret_column_pos + 1] = c;
      InvalidateChar (caret_column_pos, caret_line_pos);
      InvalidateChar (caret_column_pos + 1, caret_line_pos);
      if ((caret_column_pos += 2) >= column_size)
      {
	caret_column_pos = 0;
	if (++caret_line_pos >= line_size)
	{
	  ScrollUp ();
	  caret_line_pos = line_size - 1;
	}
      }
      kanji_putchar_buf = 0;
      return;
    }
  }
  if (((ubyte)c) < ' ')
  {
    c = '.';
  }
  char* l = GetLine(caret_line_pos);
  switch (CheckKanji (l, caret_column_pos))
  {
   case KanjiFirstByte:
    l[caret_column_pos + 1] = ' ';
    InvalidateChar (caret_column_pos + 1, caret_line_pos);
    break;
   case KanjiSecondByte:
    l[caret_column_pos - 1] = ' ';
    InvalidateChar (caret_column_pos - 1, caret_line_pos);
    break;
   case NotKanji:
    break;
  }
  l[caret_column_pos] = c;
  InvalidateCaretChar ();
  if (++caret_column_pos >= column_size)
  {
    caret_column_pos = 0;
    if (++caret_line_pos >= line_size)
    {
      ScrollUp ();
      caret_line_pos = line_size - 1;
    }
  }
  kanji_putchar_buf = 0;
}

void
TextWnd::ScrollUp ()
{
  memmove (GetLine(0), GetLine(1), column_size * (line_size - 1));
  memset (GetLine(line_size - 1), ' ', column_size);
  ::ScrollWindow (GetHandle (), 0, -font_height, 0, 0);
}

void
TextWnd::InvalidateChar (uword column, uword line)
{
  RECT rec;
  rec.top = (line - vscroll_pos) * font_height;
  rec.left = (column - hscroll_pos) * font_width;
  rec.bottom = rec.top + font_height;
  rec.right = rec.left + font_width;
  ::InvalidateRect (GetHandle (), &rec, TRUE);
}

void
TextWnd::InvalidateCaretChar ()
{
  InvalidateChar (caret_column_pos, caret_line_pos);
}

void 
TextWnd::TouchCaret ()
{
  ::HideCaret (GetHandle ());
  ::SetCaretPos ((caret_column_pos - hscroll_pos) * font_width,
		 (caret_line_pos - vscroll_pos) * font_height);
  ::ShowCaret (GetHandle ());
}
