// -------------------------------------------------------------------- //
//                        Wool Object Oriented Lisp
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// wool1.cc:
//
// class Wool
// basic LISP Function
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/wool/RCS/wool1.cc,v 1.12 1994/05/06 02:43:22 kudou Exp $

#include "wool.h"
#include "flow.h"
extern "C"
{
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
}

Object*
Wool::Cons (Object* _car, Object* _cdr)
{
  GCLink car = _car;
  GCLink cdr = _cdr;

  return new (this) Pair (car, cdr);
}

Object*
Wool::Car (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    return 0;	// (car nil) -> nil
  }
  if (TypeP_ (x, WT_Pair))
  {
    return ((Pair*)((Object*)x))->Car ();
  }
  return Error ("Car : argument isn't ConsCell");
}

Object*
Wool::Cdr (Object* _x)
{
  GCLink x = _x;
  
  if (NullP (x))
  {
    return 0;	// (cdr nil) -> nil
  }
  if (TypeP_ (x, WT_Pair))
  {
    return ((Pair*)((Object*)x))->Cdr ();
  }
  return Error ("Cdr : argument isn't ConsCell");
}

Object*
Wool::Caar (Object* _x)
{
  GCLink x = _x;

  GCLink g1 = Car (x);
  return Car (g1);
}

Object*
Wool::Cadr (Object* _x)
{
  GCLink x = _x;

  GCLink g1 = Cdr (x);
  return Car (g1);
}

Object*
Wool::Cdar (Object* _x)
{
  GCLink x = _x;

  GCLink g1 = Car (x);
  return Cdr (g1);
}

Object*
Wool::Cddr (Object* _x)
{
  GCLink x = _x;

  GCLink g1 = Cdr (x);
  return Cdr (g1);
}

Object*
Wool::Cxr (Object* _x, char* xxx)
{
  GCLink x = _x;

  if (xxx[1] == '\0')
  {
    return (*xxx == 'a') ? Car (x) : Cdr (x);
  }
  GCLink g1 = Cxr (x, xxx + 1);
  return (*xxx == 'a') ? Car (g1) : Cdr (g1);
}

Object*
Wool::Eq (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  return (x == y) ? GetAtom ("t") : 0;
}

Object*
Wool::Atom_ (Object* _x)
{
  GCLink x = _x;

  return AtomP (x) ? GetAtom ("t") : 0;
}

Object*
Wool::Null (Object* _x)
{
  GCLink x = _x;

  return NullP (x) ? GetAtom ("t") : 0;
}

Object*
Wool::Rplaca (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (!TypeP (x, WT_Pair))
  {
    Error ("Rplaca : arg1 isn't Pair");
  }
  ((Pair*)((Object*)x))->ReplaceCar (y);
  return x;
}

Object*
Wool::Rplacd (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (!TypeP (x, WT_Pair))
  {
    Error ("Rplacd : arg1 isn't Pair");
    return 0;
  }
  ((Pair*)((Object*)x))->ReplaceCdr (y);
  return x;
}

Object*
Wool::Pairlis (Object* _x, Object* _y, Object* _a)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink a = _a;

  if (NullP (x))
  {
    return 0;
  }
  GCLink g1 = Car (x);
  GCLink g2 = Car (y);
  Object* car = Cons (g1, g2);
  g1 = Cdr (x);
  g2 = Cdr (y);
  Object* cdr = Pairlis (g1, g2, a);
  return Cons (car, cdr);
}

Object*
Wool::Assoc (Object* _x, Object* _a)
{
  GCLink x = _x;
  GCLink a = _a;

  GCLink g1;
  while (!NullP (a))
  {
    g1 = Caar (a);
    if (EqP (g1, x))
    {
      return Car (a);
    }
    a = Cdr (a);
  }
  return 0;
}

Object*
Wool::Quote (Object* x)
{
  return x;
}

Object* 
Wool::And (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  GCLink g1;
  GCLink ret;
  for (GCLink o = m; !NullP(o); o = Cdr (o))
  {
    g1 = Car (o);
    ret = Eval (g1, alist);
    if (NullP (ret))
    {
      return 0;
    }
  }
  return GetAtom ("t");
}

Object* 
Wool::Or (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  bool nil_p = True;
  GCLink g1;
  GCLink ret;
  for (GCLink o = m; !NullP(o); o = Cdr (o))
  {
    g1 = Car (o);
    ret = Eval (g1, alist);
    if (!NullP (ret))
    {
      nil_p = False;
    }
  }
  return nil_p ? 0 : GetAtom ("t");
}

Object*
Wool::Equal (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (EqP (x, y))
  {
    return GetAtom ("t");
  }
  if (AtomP (x))
  {
    return ((Object*)x)->Equal (y) ? GetAtom ("t") : 0; 
  }
  if (AtomP (y))
  {
    return 0;
  }
  GCLink g1 = Car (x);
  GCLink g2 = Car (y);
  if (Equal (g1, g2))
  {
    g1 = Cdr (x);
    g2 = Cdr (y);
    return Equal (g1, g2);
  }
  return 0;
}

Object*
Wool::List (Object* _m, Object* _alist)
{
  GCLink m = _m;
  GCLink alist = _alist;

  return Evlis (m, alist);
}

Object*
Wool::Append (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;
  
  if (NullP (x))
  {
    return y;
  }
  GCLink g1 = Car (x);
  GCLink g2 = Cdr (x);
  GCLink g3 = Append (g2, y);
  return Cons (g1, g3);
}

Object*
Wool::Nconc (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x))
  {
    return y;
  }
  Object* o = x;
  GCLink g1 = Cdr (o);
  while (!NullP (g1))
  {
    o = Cdr (o);
  }
  Rplacd (o, y);
  return x;
}

Object*
Wool::Not (Object* _x)
{
  GCLink x = _x;

  return NullP (x) ? GetAtom ("t") : 0;
}

Object*
Wool::Last (Object* _x)
{
  GCLink x = _x;

  GCLink g1;
  while (((g1 = Cdr (x)), !NullP (g1)))
  {
    x = Cdr (x);
  }
  return x;
}

Object*
Wool::Reverse (Object* _x)
{
  GCLink x = _x;

  if (AtomP (x))
  {
    return x;
  }
  GCLink ret = 0;
  GCLink g1;
  while (!NullP (x))
  {
    g1 = Car (x);
    ret = Cons (g1, ret);
    x = Cdr (x);
  }
  return ret;
}

Object*
Wool::Subst (Object* _new_s, Object* _old_s, Object* _x)
{
  GCLink new_s = _new_s;
  GCLink old_s = _old_s;
  GCLink x = _x;

  if (AtomP (x))
  {
    if (Eq (x, old_s))
    {
      return new_s;
    }
    return x;
  }
  GCLink rev = Reverse (x);
  GCLink ret = 0;
  GCLink g1;
  GCLink g2;
  while (!NullP (rev))
  {
    g1 = Car (rev);
    g2 = Subst (new_s, old_s, g1);
    ret = Cons (g2, ret);
    rev = Cdr (rev);
  }
  return ret;
}

Object*
Wool::Rand ()
{
  return new (this) Num ((long)rand () * (long)rand ());
}

Object*
Wool::Add1 (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    Error ("Add1 : argument is nil");
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
    return new (this) Num (((Num*)((Object*)x))->GetNum () + 1);
   case WT_FNum:
    return new (this) FNum (((FNum*)((Object*)x))->GetFNum () + 1);
   default:
    return Error ("Add1 : argument isn't number");
  }
}

Object*
Wool::Sub1 (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    Error ("Sub1 : argument is nil");
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
    return new (this) Num (((Num*)((Object*)x))->GetNum () - 1);
   case WT_FNum:
    return new (this) FNum (((FNum*)((Object*)x))->GetFNum () - 1);
   default:
    return Error ("Sub1 : argument isn't number");
  }
}

Object*
Wool::Difference (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x) || NullP (y))
  {
    Error ("Differance : arguments include nil");
  }
  WoolType x_t = ((Object*)x)->TypeOf ();
  WoolType y_t = ((Object*)y)->TypeOf ();
  if ((x_t == WT_Num) && (y_t == WT_Num))
  {
    int n = (((Num*)((Object*)x))->GetNum () -
	     ((Num*)((Object*)y))->GetNum ());
    return new (this) Num (n < 0 ? -n : n);
  }
  else if ((x_t == WT_Num || x_t == WT_FNum) &&
	   (y_t == WT_Num || y_t == WT_FNum))
  {
    double n = (((x_t == WT_Num) ? ((Num*)((Object*)x))->GetNum () : 
		 ((FNum*)((Object*)x))->GetFNum ()) -
		((y_t == WT_Num) ? ((Num*)((Object*)y))->GetNum () : 
		 ((FNum*)((Object*)y))->GetFNum ()));
    return new (this) FNum (n < 0 ? -n : n); 
  }
  return Error ("Differance : arguments include except number");
}

#define Operation(fname,_op_) \
  if (NullP (m))\
  {\
    Error (fname " : argument is nil");\
  }\
  long int reg = 0;\
  double freg = 0;\
  bool fnum_p = False;\
  GCLink o = Car (m);\
  if (NullP (o))\
  {\
    Error (fname " : arguments include nil");\
  }\
  switch (((Object*)o)->TypeOf ())\
  {\
   case WT_Num:\
    reg = ((Num*)((Object*)o))->GetNum ();\
    break;\
   case WT_FNum:\
    freg = ((FNum*)((Object*)o))->GetFNum ();\
    fnum_p = True;\
    break;\
   default:\
    Error ( fname " : arguments include except number");\
  }\
  m = Cdr (m);\
  while (!NullP (m))\
  {\
    o = Car (m);\
    if (NullP (o))\
    {\
      Error (fname " : arguments include nil");\
    }\
    switch (((Object*)o)->TypeOf ())\
    {\
     case WT_Num:\
      if (fnum_p)\
      {\
	freg _op_ ## = (long int)((Num*)((Object*)o))->GetNum ();\
      }\
      else\
      {\
	reg _op_ ## = ((Num*)((Object*)o))->GetNum ();\
      }\
      break;\
     case WT_FNum:\
      if (fnum_p)\
      {\
	freg _op_ ## = ((FNum*)((Object*)o))->GetFNum ();\
      }\
      else\
      {\
	freg = reg _op_ ((FNum*)((Object*)o))->GetFNum ();\
      }\
      fnum_p = True;\
      break;\
     default:\
      Error (fname " : arguments include except number");\
    }\
    m = Cdr (m);\
  }\
  return fnum_p ? (Object*)new (this) FNum (freg) : (Object*)new (this) Num (reg);

Object*
Wool::Plus (Object* _m)
{
  GCLink m = _m;

  Operation("Plus", +)
}

Object*
Wool::Minus (Object* _m)
{
  GCLink m = _m;

  GCLink g1 = Cdr (m);
  if (!NullP (m) && NullP (g1))
  {
    GCLink o = Car (m);
    if (((Object*)o)->TypeOf () == WT_Num)
    {
      return new (this) Num (- ((Num*)((Object*)o))->GetNum ());
    }
    if (((Object*)o)->TypeOf () == WT_FNum)
    {
      return new (this) FNum (- ((FNum*)((Object*)o))->GetFNum ());
    }
  }

  Operation ("Minus", -)
}

Object*
Wool::Times (Object* _m)
{
  GCLink m = _m;

  Operation ("Times", *)
}

Object*
Wool::Quotient (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x) || NullP (y))
  {
    Error ("Quotient : arguments include nil");
  }
  WoolType x_t = ((Object*)x)->TypeOf ();
  WoolType y_t = ((Object*)y)->TypeOf ();
  if ((x_t == WT_Num) && (y_t == WT_Num))
  {
    int n1 = ((Num*)((Object*)x))->GetNum ();
    int n2 = ((Num*)((Object*)y))->GetNum ();
    if (n2 == 0)
    {
      Error ("Quotient : divide by 0");
    }
    return new (this) Num (n1 / n2);
  }
  else if ((x_t == WT_Num || x_t == WT_FNum) &&
	   (y_t == WT_Num || y_t == WT_FNum))
  {
    double n1 = ((x_t == WT_Num) ? ((Num*)((Object*)x))->GetNum () :
		 ((FNum*)((Object*)x))->GetFNum ());
    double n2 = ((y_t == WT_Num) ? ((Num*)((Object*)y))->GetNum () :
		 ((FNum*)((Object*)y))->GetFNum ());
    if (n2 == 0)
    {
      Error ("Quotient : divide by 0");
    }
    return new (this) FNum (n1 / n2);
  }
  else
  {
    return Error ("Quotient : arguments include except number");
  }
}

Object*
Wool::Remainder (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x) || NullP (y))
  {
    Error ("Remainder : arguments include nil");
  }
  if (TypeP_ (x, WT_Num) && TypeP_ (y, WT_Num))
  {
    int n1 = ((Num*)((Object*)x))->GetNum ();
    int n2 = ((Num*)((Object*)y))->GetNum ();
    if (n2 == 0)
    {
      Error ("Remainder : divide by 0");
    }
    return new (this) Num (n1 % n2);
  }
  return Error ("Remainder : arguments include except number");
}

Object*
Wool::ZeroP (Object* _x)
{
  GCLink x = _x;
  
  if (NullP (x))
  {
    Error ("ZeroP : argument is null");
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
    return (((Num*)((Object*)x))->GetNum () == 0) ? GetAtom ("t") : 0;
   case WT_FNum:
    return (((FNum*)((Object*)x))->GetFNum () == 0) ? GetAtom ("t") : 0;
   default:
    return Error ("ZeroP : argument isn't number");
  }
}

Object*
Wool::MinusP (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    Error ("MinusP : argument is null");
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
    return (((Num*)((Object*)x))->GetNum () < 0) ? GetAtom ("t") : 0;
   case WT_FNum:
    return (((FNum*)((Object*)x))->GetFNum () < 0) ? GetAtom ("t") : 0;
   default:
    return Error ("MinusP : argument isn't number");
  }
}

#define Compare(fname,_op_) \
  if (NullP (x) || NullP (y))\
  {\
    Error (fname " : arguments include nil");\
  }\
  WoolType x_t = ((Object*)x)->TypeOf ();\
  WoolType y_t = ((Object*)y)->TypeOf ();\
  if ((x_t == WT_Num) && (y_t == WT_Num))\
  {\
    int n1 = ((Num*)((Object*)x))->GetNum ();\
    int n2 = ((Num*)((Object*)y))->GetNum ();\
    return (n1 _op_ n2) ? GetAtom ("t") : 0;\
  }\
  else if ((x_t == WT_Num || x_t == WT_FNum) &&\
	   (y_t == WT_Num || y_t == WT_FNum))\
  {\
    double n1 = ((x_t == WT_Num) ? ((Num*)((Object*)x))->GetNum () :\
		 ((FNum*)((Object*)x))->GetFNum ());\
    double n2 = ((y_t == WT_Num) ? ((Num*)((Object*)y))->GetNum () :\
		 ((FNum*)((Object*)y))->GetFNum ());\
    return (n1 _op_ n2) ? GetAtom ("t") : 0;\
  }\
  return Error (fname " : arguments include except number");

Object*
Wool::GreaterP (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  Compare("GreaterP", >)
}

Object*
Wool::LessP (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  Compare("LessP", <)
}

Object*
Wool::GreaterOrEqP (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  Compare ("GreaterOrEqP", >=)
}

Object*
Wool::LessOrEqP (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  Compare ("LessrOrEqP", <=)
}

Object*
Wool::Length (Object* _x)
{
  GCLink x = _x;

  int n = 0;
  while (!NullP (x))
  {
    n++;
    x = Cdr (x);
  }
  return new (this) Num (n);
}

Object*
Wool::NumberP (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    return 0;
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
   case WT_FNum:
    return GetAtom ("t");
   default:
    return 0;
  }
}

Object*
Wool::FloatP (Object* _x)
{
  GCLink x = _x;

  return TypeP (x, WT_FNum) ? GetAtom ("t") : 0;
}

Object*
Wool::Fix (Object* _x)
{
  GCLink x = _x;

  if (NullP (x))
  {
    Error ("Fix : argument is null");
  }
  switch (((Object*)x)->TypeOf ())
  {
   case WT_Num:
    return new (this) Num (((Num*)((Object*)x))->GetNum ());
   case WT_FNum:
    return new (this) Num ((int)((FNum*)((Object*)x))->GetFNum ());
   default:
    return Error ("Fix : argument isn't number");
  }
}

Object*
Wool::LeftShift (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x) || NullP (y))
  {
    Error ("LeftShift : arguments include nil");
  }
  if (TypeP_ (x, WT_Num) && TypeP_ (y, WT_Num))
  {
    int n1 = ((Num*)((Object*)x))->GetNum ();
    int n2 = ((Num*)((Object*)y))->GetNum ();
    return new (this) Num (n1 << n2);
  }
  return Error ("LeftShift : arguments include except number");
}

Object*
Wool::RightShift (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;

  if (NullP (x) || NullP (y))
  {
    Error ("RightShift : arguments include nil");
  }
  if (TypeP_ (x, WT_Num) && TypeP_ (y, WT_Num))
  {
    int n1 = ((Num*)((Object*)x))->GetNum ();
    int n2 = ((Num*)((Object*)y))->GetNum ();
    return new (this) Num (n1 >> n2);
  }
  return Error ("RightShift : arguments include except number");
}

#define MaxMinOp(fname,_op_) \
  if (NullP (m))\
  {\
    Error (fname " : argument is nil");\
  }\
  long int reg = 0;\
  double freg = 0;\
  bool fnum_p = False;\
  GCLink o = Car (m);\
  if (NullP (o))\
  {\
    Error (fname " : arguments include nil");\
  }\
  switch (((Object*)o)->TypeOf ())\
  {\
   case WT_Num:\
    reg = ((Num*)((Object*)o))->GetNum ();\
    break;\
   case WT_FNum:\
    freg = ((FNum*)((Object*)o))->GetFNum ();\
    fnum_p = True;\
    break;\
   default:\
    Error (fname " : arguments include except number");\
  }\
  while (!NullP (m))\
  {\
    o = Car (m);\
    if (NullP (o))\
    {\
      Error (fname " : arguments include nil");\
    }\
    switch (((Object*)o)->TypeOf ())\
    {\
     case WT_Num:\
      if (fnum_p)\
      {\
        if (reg _op_ ((Num*)((Object*)o))->GetNum ())\
	{\
	  freg = ((Num*)((Object*)o))->GetNum ();\
	}\
      }\
      else\
      {\
        if (reg _op_ ((Num*)((Object*)o))->GetNum ())\
	{\
	  reg = ((Num*)((Object*)o))->GetNum ();\
	}\
      }\
      break;\
     case WT_FNum:\
      if (fnum_p)\
      {\
        if (freg _op_ ((FNum*)((Object*)o))->GetFNum ())\
	{\
	  freg = ((FNum*)((Object*)o))->GetFNum ();\
	}\
      }\
      else\
      {\
        if (reg _op_ ((FNum*)((Object*)o))->GetFNum ())\
	{\
	  freg = ((FNum*)((Object*)o))->GetFNum ();\
	}\
      }\
      fnum_p = True;\
      break;\
     default:\
      Error (fname " : arguments include except number");\
    }\
    m = Cdr (m);\
  }\
  return fnum_p ? (Object*)new (this) FNum (freg) : (Object*)new (this) Num (reg);\

Object*
Wool::Max (Object* _m)
{
  GCLink m = _m;

  MaxMinOp ("Max", <)
}

Object*
Wool::Min (Object* _m)
{
  GCLink m = _m;

  MaxMinOp ("Min", >)
}

#ifdef MATHFUNC
extern "C"
{
#include <math.h>
#include <errno.h>
}

#define ArgCheck(x, n, s) \
if (TypeP (x, WT_Num))\
{\
  n = ((Num*)((Object*)x))->GetNum ();\
}\
else if (TypeP (x, WT_FNum))\
{\
  n = ((FNum*)((Object*)x))->GetFNum ();\
}\
else\
{\
  Error (s " : argument isn't number");\
}

#define MathFunc(fn, fn_str, cfn) \
Object* \
Wool:: ## fn (Object* _x)\
{\
  GCLink x = _x;\
\
  double n;\
  ArgCheck (x, n, fn_str);\
  return new (this) FNum (cfn (n));\
}

MathFunc (Acos, "Acos", acos)
MathFunc (Asin, "Asin", asin)
MathFunc (Atan, "Atan", atan)
MathFunc (Ceil, "Ceil", ceil)
MathFunc (Cos, "Cos", cos)
MathFunc (Cosh, "Cosh", cosh)
MathFunc (Exp, "Exp", exp)
MathFunc (Fabs, "Fabs", fabs)
MathFunc (Floor, "Floor", floor)
MathFunc (Log, "Log", log)
MathFunc (Log10, "Log10", log10)
MathFunc (Sin, "Sin", sin)
MathFunc (Sinh, "Sinh", sinh)
MathFunc (Sqrt, "Sqrt", sqrt)
MathFunc (Tan, "Tan", tan)
MathFunc (Tanh, "Tanh", tanh)

#define MathFunc2(fn, fn_str, cfn) \
Object* \
Wool:: ## fn (Object* _x, Object* _y)\
{\
  GCLink x = _x;\
  GCLink y = _y;\
\
  double n;\
  double m;\
  ArgCheck (x, n, fn_str);\
  ArgCheck (y, m, fn_str);\
  return new (this) FNum (cfn (n, m));\
}

MathFunc2 (Atan2, "Atan2", atan2)
MathFunc2 (Pow, "Pow", pow)

Object*
Wool::MathError ()
{
  return new (this) Num (errno);
}
#endif /* MATHFUNC */

void
Wool::Printf1 (char* fmt, Object* _o)
{
  GCLink o = _o;

  switch (((Object*)o)->TypeOf ())
  {
   case WT_Num:
    o_flow->Printf (fmt, ((Num*)((Object*)o))->GetNum ());
    break;
   case WT_FNum:
    o_flow->Printf (fmt, ((FNum*)((Object*)o))->GetFNum ());
    break;
   case WT_Str:
    o_flow->Printf (fmt, ((Str*)((Object*)o))->GetStr ());
    break;
   default:
    Error ("Printf : invalid argument type");
    break;
  }
}

Object*
Wool::Printf_ (Object* _fmt, Object* _arg)
{
  GCLink fmt = _fmt;
  GCLink arg = _arg;

  if (!TypeP (fmt, WT_Str))
  {
    Error ("Printf : first argument isn't string");
  }
  char* fmts = ((Str*)((Object*)fmt))->GetStr ();
  bool done = True;
  char* prev = fmts;
  while (done)
  {
    switch (*fmts)
    {
     case '\\':
      {
	*fmts = '\0';
	o_flow->Printf (prev);
	int skip = 2;
	switch (*(fmts + 1))
	{
	 case 'n':
	  o_flow->Printf ("\n");
	  break;
	 case 'r':
	  o_flow->Printf ("\r");
	  break;
	 case 't':
	  o_flow->Printf ("\t");
	  break;
	 case 'b':
	  o_flow->Printf ("\b");
	  break;
	 case 'f':
	  o_flow->Printf ("\f");
	  break;
	 case '\\':
	  o_flow->Printf ("\\");
	  break;
	 default:
	  skip = 1;
	  if (isdigit (*(fmts + 1)))
	  {
	    int i;
	    sscanf ((fmts + 1), "%o", &i);
	    o_flow ->Printf ("%c", i);
            skip = (isdigit (*(fmts + 2)) ? 
		    (isdigit (*(fmts + 3)) ? 4 : 3) : 2);
	  }
	  break;
	}
	*fmts = '\\';
	prev = fmts + skip;
	fmts += skip;
      }
      break;
     case '%':
      {
	char* p = fmts + 1;
	bool done2 = True;
	while (done2)
	{
	  switch (*p++)
	  {
	   case '\0':
	    Error ("Printf : invalid format string");
	   case 'd':
	   case 'i':
	   case 'u':
	   case 'o':
	   case 'x':
	   case 'X':
	   case 'f':
	   case 'e':
	   case 'g':
	   case 'G':
	   case 's':
	    done2 = False;
	    break;
 	  default:
	    break;
	  }
	}
	char c = *p;
	*p = '\0';
	GCLink g1 = Car (arg);
	Printf1 (prev, g1);
	arg = Cdr (arg);
	prev = p;
	*p = c;
	fmts++;
      }
      break;
     case '\0':
      done = False;
      o_flow->Printf (prev);
      break;
     default:
      fmts++;
      break;
    }
  }
  return 0;
}
