// -------------------------------------------------------------------- //
//                        Wool Object Oriented Lisp
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// wingrph.cc:
//
// for MS-Windows Graphic screen
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/wool/RCS/wingrph.cc,v 1.2 1994/05/06 02:43:28 kudou Exp $

#include "wool.h"

#ifdef MSWG
#include "wingrph.h"
#include <string.h>

#define MAX(a,b) (((a) > (b)) ? (a) : (b))
#define MIN(a,b) (((a) < (b)) ? (a) : (b))

#define DEBUG
#ifdef DEBUG
#include <stdlib.h>
#endif

// -------------------------------------------------------------------- //
// class GrphWnd
// -------------------------------------------------------------------- //
GrphWnd* GrphWnd::grph_wnd = 0;

void
GrphWnd::OpenGrphWnd ()
{
  if (!grph_wnd)
  {
    grph_wnd = new GrphWnd ();
    grph_wnd->OpenWindow ();
  }
}

void
GrphWnd::CloseGrphWnd ()
{
  if (!grph_wnd)
  {
    delete grph_wnd;
    grph_wnd = 0;
  }
}

GrphWnd::GrphWnd ()
{
  vscroll_offset = 0;
  hscroll_offset = 0;
  pen_col = RGB (0, 0, 0);
  fill_col = RGB (0, 0, 0);
  back_col = RGB (0xff, 0xff, 0xff);
  set_vscroll_bar_p = False;
  set_hscroll_bar_p = False;

  // create virtual screen
  HWND dsktop = ::GetDesktopWindow ();
  HDC dc = ::GetDC (dsktop);
  vscrn_width = ::GetSystemMetrics (SM_CXSCREEN);
  vscrn_height = ::GetSystemMetrics (SM_CYSCREEN);
  vscrn = ::CreateCompatibleBitmap (dc, vscrn_width, vscrn_height);
  if (vscrn)
  {
    HDC dc_mem = ::CreateCompatibleDC (dc);
    HGDIOBJ old = ::SelectObject (dc_mem, vscrn);
    RECT rec;
    rec.top = rec.left = 0;
    rec.right = vscrn_width;
    rec.bottom = vscrn_height;
    ::FillRect (dc_mem, &rec, ::GetStockObject (WHITE_BRUSH));
    ::SelectObject (dc_mem, old);
    ::DeleteDC (dc_mem);
  }
  ::ReleaseDC (dsktop, dc);
}

GrphWnd::~GrphWnd ()
{
  if (vscrn)
  {
    ::DeleteObject (vscrn);
  }
}

LPSTR
GrphWnd::GetCaptionStr ()
{
  return "WOOL Graphic Window";
}

void
GrphWnd::Display (HDC dc)
{
  if (vscrn)
  {
    HDC dc_mem = ::CreateCompatibleDC (dc);
    HGDIOBJ old = ::SelectObject (dc_mem, vscrn);
    ::BitBlt (dc, 0, 0, GetWindowXSize (), GetWindowYSize (),
	      dc_mem, hscroll_offset, vscroll_offset, SRCCOPY);
    ::SelectObject (dc_mem, old);
    ::DeleteDC (dc_mem);
  }
}

DWORD
GrphWnd::GetCreateWindowStyle ()
{
  return (WS_OVERLAPPEDWINDOW | WS_MAXIMIZEBOX | WS_MINIMIZEBOX |
	  WS_HSCROLL | WS_VSCROLL);
}

long
GrphWnd::WindowProc (UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
#ifdef DEBUG
   case WM_CHAR:
    SetPenCol (rand ());
    SetFillCol (rand ());
    SetBackCol (rand ());
    DrawBox (rand () % vscrn_width,
	     rand () % vscrn_height,
	     rand () % vscrn_width,
	     rand () % vscrn_width);
    DrawArcFill (rand () % vscrn_height,
		 rand () % vscrn_width,
		 (rand () % vscrn_height) / 2);
    break;
#endif
   case WM_HSCROLL:
#ifdef DEBUG
    DrawText (10, 10, "hello graphic window");
#endif
    HScroll (wparam, lparam);
    break;

   case WM_VSCROLL:
    VScroll (wparam, lparam);
    break;

   case WM_ERASEBKGND:
    /* do nothing */
    return 1;

   default:
    return BaseWnd::WindowProc (msg, wparam, lparam);
  }
  return 0L;
}

void
GrphWnd::Invalidate (int x1, int y1, int x2, int y2)
{
  RECT rec;
  rec.top =  MIN (y1, y2) - vscroll_offset;
  rec.left = MIN (x1, x2) - hscroll_offset;
  rec.bottom = MAX (y1, y2) - vscroll_offset;
  rec.right =  MAX (x1, x2) - hscroll_offset;
  InvalidateRect (&rec, FALSE);
}

#define DRAWPLOGUE \
  HDC wdc = GetDC ();\
  HDC dc = ::CreateCompatibleDC (wdc);

#define EPILOGUE \
  ::DeleteDC (dc);\
  ReleaseDC (wdc);

void
GrphWnd::DrawPixel (int x, int y)
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  ::SetPixel (dc, x, y, GetPenCol ());
  ::SelectObject (dc, old);
  EPILOGUE;
  Invalidate (x, y, x + 1, y + 1);
}

void
GrphWnd::DrawText (int x, int y, char* s)
{
  int size = strlen (s);
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  int old_txt_col = ::SetTextColor (dc, GetPenCol ());
  int old_bk_mode = ::SetBkMode (dc, OPAQUE);
  int old_bk_col = ::SetBkColor (dc, GetBackCol ());
  ::TextOut (dc, x, y, s, size);
  ::SetBkColor (dc, old_bk_col);
  ::SetBkMode (dc, old_bk_mode);
  ::SetTextColor (dc, old_txt_col);
  ::SelectObject (dc, old);
  int w = ::GetTextExtent (dc, s, size);
  TEXTMETRIC metric;
  ::GetTextMetrics (dc, &metric);
  EPILOGUE;
  Invalidate (x, y, x + w, y + metric.tmHeight);
}

void
GrphWnd::DrawTransparentText (int x, int y, char* s)
{
  int size = strlen (s);
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  int old_txt_col = ::SetTextColor (dc, GetPenCol ());
  int old_bk_mode = ::SetBkMode (dc, TRANSPARENT);
  ::TextOut (dc, x, y, s, size);
  ::SetBkMode (dc, old_bk_mode);
  ::SetTextColor (dc, old_txt_col);
  ::SelectObject (dc, old);
  int w = ::GetTextExtent (dc, s, size);
  TEXTMETRIC metric;
  ::GetTextMetrics (dc, &metric);
  EPILOGUE;
  Invalidate (x, y, x + w, y + metric.tmHeight);
}

void
GrphWnd::DrawLine (int x1, int y1, int x2, int y2)
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  HPEN pen = ::CreatePen (PS_SOLID, 1, GetPenCol ());
  HGDIOBJ old_pen = ::SelectObject (dc, pen);
  ::MoveTo (dc, x1, y1);
  ::LineTo (dc, x2, y2);
  ::SelectObject (dc, old_pen);
  ::DeleteObject (pen);
  ::SelectObject (dc, old);
  EPILOGUE;
  Invalidate (MIN(x1, x2), MIN(y1, y2), MAX(x1, x2) + 1, MAX(y1, y2) + 1);
}

void
GrphWnd::DrawBox (int x1, int y1, int x2, int y2)
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  HPEN pen = ::CreatePen (PS_SOLID, 1, GetPenCol ());
  HGDIOBJ old_pen = ::SelectObject (dc, pen);
  ::MoveTo (dc, x1, y1);
  ::LineTo (dc, x2, y1);
  ::LineTo (dc, x2, y2);
  ::LineTo (dc, x1, y2);
  ::LineTo (dc, x1, y1);
  ::SelectObject (dc, old_pen);
  ::DeleteObject (pen);
  ::SelectObject (dc, old);
  EPILOGUE;
  Invalidate (MIN(x1, x2), MIN(y1, y2), MAX(x1, x2) + 1, MAX(y1, y2) + 1);
}

void
GrphWnd::DrawBoxFill (int x1, int y1, int x2, int y2)
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  HPEN pen = ::CreatePen (PS_SOLID, 1, GetPenCol ());
  HGDIOBJ old_pen = ::SelectObject (dc, pen);
  HBRUSH br = ::CreateSolidBrush (GetFillCol ());
  RECT rec;
  rec.top = y1;
  rec.left = x1;
  rec.bottom = y2;
  rec.right = x2;
  ::FillRect (dc, &rec, br);
  ::DeleteObject (br);
  ::MoveTo (dc, x1, y1);
  ::LineTo (dc, x2, y1);
  ::LineTo (dc, x2, y2);
  ::LineTo (dc, x1, y2);
  ::LineTo (dc, x1, y1);
  ::SelectObject (dc, old_pen);
  ::DeleteObject (pen);
  ::SelectObject (dc, old);
  EPILOGUE;
  Invalidate (MIN(x1, x2), MIN(y1, y2), MAX(x1, x2) + 1, MAX(y1, y2) + 1);
}

void
GrphWnd::DrawArcFill (int x, int y, int r)
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  HPEN pen = ::CreatePen (PS_SOLID, 1, GetPenCol ());
  HGDIOBJ old_pen = ::SelectObject (dc, pen);
  HBRUSH br = ::CreateSolidBrush (GetFillCol ());
  HGDIOBJ old_br = ::SelectObject (dc, br);
  ::Ellipse (dc, x - r, y - r, x + r, y + r);
  ::SelectObject (dc, old_br);
  ::DeleteObject (br);
  ::SelectObject (dc, old_pen);
  ::DeleteObject (pen);
  ::SelectObject (dc, old);
  EPILOGUE;
  Invalidate (x - r, y - r, x + r + 1, y + r + 1);
}

void
GrphWnd::Clear ()
{
  DRAWPLOGUE;
  HGDIOBJ old = ::SelectObject (dc, vscrn);
  HBRUSH br = ::CreateSolidBrush (GetBackCol ());
  RECT rec;
  rec.top = rec.left = 0;
  rec.right = vscrn_width;
  rec.bottom = vscrn_height;
  ::FillRect (dc, &rec, br);
  ::DeleteObject (br);
  EPILOGUE;
  Invalidate (0, 0, vscrn_width, vscrn_height);
}

void
GrphWnd::Resize (WORD left, WORD top, WORD right, WORD bottom)
{
  if (!set_vscroll_bar_p)
  {
    word max_vscroll_offset = MAX (vscrn_height -
				   GetWindowYSize (), 0);
    vscroll_offset = MAX (0, MIN (vscroll_offset, max_vscroll_offset)); 
    int h_min, h_max;
    ::GetScrollRange (GetHandle (), SB_HORZ, &h_min, &h_max);
    if (!((h_max == 0) && (h_min == 0)) &&
	(bottom - top + ::GetSystemMetrics (SM_CYHSCROLL)) >=
	vscrn_height)
    {
      max_vscroll_offset = 0;
      vscroll_offset = 0;
    }
    set_vscroll_bar_p = True;
    ::SetScrollRange (GetHandle (), SB_VERT, 0, max_vscroll_offset, FALSE);
    ::SetScrollPos (GetHandle (), SB_VERT, vscroll_offset, TRUE);
    set_vscroll_bar_p = False;
  }
  if (!set_hscroll_bar_p)
  {
    word max_hscroll_offset = MAX (vscrn_width - 
				   GetWindowXSize (), 0);
    hscroll_offset = MAX (0, MIN (hscroll_offset, max_hscroll_offset));
    int v_min, v_max;
    ::GetScrollRange (GetHandle (), SB_VERT, &v_min, &v_max);
    if (!((v_max == 0) && (v_min == 0)) &&
	(right - left + ::GetSystemMetrics (SM_CXVSCROLL)) >= 
	vscrn_width)
    {
      max_hscroll_offset = 0;
      hscroll_offset = 0;
    }
    set_hscroll_bar_p = True;
    ::SetScrollRange (GetHandle (), SB_HORZ, 0, max_hscroll_offset, FALSE);
    ::SetScrollPos (GetHandle (), SB_HORZ, hscroll_offset, TRUE);
    set_hscroll_bar_p = False;
  }
}

void
GrphWnd::HScroll (WPARAM wparam, LPARAM lparam)
{
  word hscroll_offset_old = hscroll_offset;
  switch (wparam)
  {
   case SB_LINEUP:
    hscroll_offset -= scroll_unit;
    break;
   case SB_LINEDOWN:
    hscroll_offset += scroll_unit;
    break;
   case SB_PAGEUP:
    hscroll_offset -= (vscrn_width / 10);
    break;
   case SB_PAGEDOWN:
    hscroll_offset += (vscrn_height / 10);
    break;
   case SB_THUMBPOSITION:
    hscroll_offset = LOWORD (lparam);
    break;
  }
  word max_hscroll_offset = MAX (vscrn_width - GetWindowXSize (), 0);
  hscroll_offset = MAX (0, MIN (hscroll_offset, max_hscroll_offset));
  if (hscroll_offset != hscroll_offset_old)
  {
    ::SetScrollPos (GetHandle (), SB_HORZ, hscroll_offset, TRUE);
    int update = hscroll_offset_old - hscroll_offset;
    ::ScrollWindow (GetHandle (), update, 0, 0, 0);
    RECT rec;
    rec.top = 0;
    rec.bottom = GetWindowYSize ();
    if (update < 0)
    {
      rec.right = GetWindowXSize ();
      rec.left = rec.right + update; 
    }
    else
    {
      rec.left = 0;
      rec.right = update;
    }
    ::InvalidateRect (GetHandle (), &rec, FALSE);
  }
}

void
GrphWnd::VScroll (WPARAM wparam, LPARAM lparam)
{
  int vscroll_offset_old = vscroll_offset;
  switch (wparam)
  {
   case SB_LINEUP:
    vscroll_offset -= scroll_unit;
    break;
   case SB_LINEDOWN:
    vscroll_offset += scroll_unit;
    break;
   case SB_PAGEUP:
    vscroll_offset -= (vscrn_height / 10);
    break;
   case SB_PAGEDOWN:
    vscroll_offset += (vscrn_height / 10);
    break;
   case SB_THUMBPOSITION:
    vscroll_offset = LOWORD (lparam);
    break;
  }
  word max_vscroll_offset = MAX (vscrn_height - GetWindowYSize (), 0);
  vscroll_offset = MAX (0, MIN (vscroll_offset, max_vscroll_offset)); 
  if (vscroll_offset != vscroll_offset_old)
  {
    ::SetScrollPos (GetHandle (), SB_VERT, vscroll_offset, TRUE);
    int update = vscroll_offset_old - vscroll_offset;
    ::ScrollWindow (GetHandle (), 0, update, 0, 0);
    RECT rec;
    rec.left = 0;
    rec.right = GetWindowXSize ();
    if (update < 0)
    {
      rec.bottom = GetWindowXSize ();
      rec.top = rec.bottom + update; 
    }
    else
    {
      rec.top = 0;
      rec.bottom = update;
    }
    ::InvalidateRect (GetHandle (), &rec, FALSE);
  }
}

// -------------------------------------------------------------------- //
// class Wool
// -------------------------------------------------------------------- //
Object*
Wool::Rgb (Object* _r, Object* _g, Object* _b)
{
  GCLink r = _r;
  GCLink g = _g;
  GCLink b = _b;
  if (!TypeP (r, WT_Num) || 
      !TypeP (g, WT_Num) ||
      !TypeP (b, WT_Num))
  {
    Error ("Rgb : arg isn't number");
  }
  return new (this) Num (RGB ((int)((Num*)((Object*)r))->GetNum (),
			      (int)((Num*)((Object*)g))->GetNum (),
			      (int)((Num*)((Object*)b))->GetNum ()));
}

Object*
Wool::SetDrawPenColor (Object* _c)
{
  GCLink c = _c;
  if (!TypeP (c, WT_Num))
  {
    Error ("SetDrawPenColor : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->SetPenCol (((Num*)((Object*)c))->GetNum ());
  return 0;
}

Object*
Wool::SetDrawFillColor (Object* _c)
{
  GCLink c = _c;
  if (!TypeP (c, WT_Num))
  {
    Error ("SetDrawFillColor : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->SetFillCol (((Num*)((Object*)c))->GetNum ());
  return 0;
}

Object*
Wool::SetDrawBackColor (Object* _c)
{
  GCLink c = _c;
  if (!TypeP (c, WT_Num))
  {
    Error ("SetDrawBackColor : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->SetBackCol (((Num*)((Object*)c))->GetNum ());
  return 0;
}

Object*
Wool::DrawPixel (Object* _x, Object* _y)
{
  GCLink x = _x;
  GCLink y = _y;
  if (!TypeP (x, WT_Num) || !TypeP (x, WT_Num))
  {
    Error ("DrawPixel : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->DrawPixel (((Num*)((Object*)x))->GetNum (),
			    ((Num*)((Object*)y))->GetNum ());
  return 0;
}

Object*
Wool::DrawText (Object* _x, Object* _y, Object* _str)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink str = _str;
  if (!TypeP (x, WT_Num) || !TypeP (y, WT_Num) || !TypeP (str, WT_Str))
  {
    Error ("DrawText : arg error");
  }
  GrphWnd::GetGrphWnd ()->DrawText (((Num*)((Object*)x))->GetNum (),
			   ((Num*)((Object*)y))->GetNum (),
			   ((Str*)((Object*)str))->GetStr ());
  return 0;
}

Object*
Wool::DrawTransparentText (Object* _x, Object* _y, Object* _str)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink str = _str;
  if (!TypeP (x, WT_Num) || !TypeP (y, WT_Num) || !TypeP (str, WT_Str))
  {
    Error ("DrawText : arg error");
  }
  GrphWnd::GetGrphWnd ()->DrawTransparentText (((Num*)((Object*)x))->GetNum (),
				      ((Num*)((Object*)y))->GetNum (),
				      ((Str*)((Object*)str))->GetStr ());
  return 0;
}

Object*
Wool::DrawLine (Object* _x1 , Object* _y1, Object* _x2, Object* _y2)
{
  GCLink x1 = _x1;
  GCLink y1 = _y1;
  GCLink x2 = _x2;
  GCLink y2 = _y2;
  if (!TypeP (x1, WT_Num) || !TypeP (y1, WT_Num) ||
      !TypeP (x2, WT_Num) || !TypeP (y2, WT_Num))
  {
    Error ("DrawLine : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->DrawLine (((Num*)((Object*)x1))->GetNum (),
			   ((Num*)((Object*)y1))->GetNum (),
			   ((Num*)((Object*)x2))->GetNum (),
			   ((Num*)((Object*)y2))->GetNum ());
  return 0;
}

Object*
Wool::DrawBox (Object* _x1, Object* _y1, Object* _x2, Object* _y2)
{
  GCLink x1 = _x1;
  GCLink y1 = _y1;
  GCLink x2 = _x2;
  GCLink y2 = _y2;
  if (!TypeP (x1, WT_Num) || !TypeP (y1, WT_Num) ||
      !TypeP (x2, WT_Num) || !TypeP (y2, WT_Num))
  {
    Error ("DrawBox : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->DrawBox (((Num*)((Object*)x1))->GetNum (),
			  ((Num*)((Object*)y1))->GetNum (),
			  ((Num*)((Object*)x2))->GetNum (),
			  ((Num*)((Object*)y2))->GetNum ());
  return 0;
}

Object*
Wool::DrawBoxFill (Object* _x1, Object* _y1, Object* _x2, Object* _y2)
{
  GCLink x1 = _x1;
  GCLink y1 = _y1;
  GCLink x2 = _x2;
  GCLink y2 = _y2;
  if (!TypeP (x1, WT_Num) || !TypeP (y1, WT_Num) ||
      !TypeP (x2, WT_Num) || !TypeP (y2, WT_Num))
  {
    Error ("DrawBoxFill : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->DrawBoxFill (((Num*)((Object*)x1))->GetNum (),
			      ((Num*)((Object*)y1))->GetNum (),
			      ((Num*)((Object*)x2))->GetNum (),
			      ((Num*)((Object*)y2))->GetNum ());
  return 0;
}

Object*
Wool::DrawArcFill (Object* _x, Object* _y, Object* _r)
{
  GCLink x = _x;
  GCLink y = _y;
  GCLink r = _r;
  if (!TypeP (x, WT_Num) || !TypeP (y, WT_Num) || !TypeP (r, WT_Num))
  {
    Error ("DrawArcFill : arg isn't number");
  }
  GrphWnd::GetGrphWnd ()->DrawArcFill (((Num*)((Object*)x))->GetNum (),
			      ((Num*)((Object*)y))->GetNum (),
			      ((Num*)((Object*)r))->GetNum ());
  return 0;
}

Object*
Wool::ClearDrawWindow ()
{
  GrphWnd::GetGrphWnd ()->Clear ();
  return 0;
}

Object*
Wool::GetDrawWindowWidth ()
{
  return new (this) Num (GrphWnd::GetGrphWnd ()->GetDrawWindowWidth ());
}

Object*
Wool::GetDrawWindowHeight ()
{
  return new (this) Num (GrphWnd::GetGrphWnd ()->GetDrawWindowHeight ());
}

Object*
Wool::GetDrawScreenWidth ()
{
  return new (this) Num (GrphWnd::GetGrphWnd ()->GetDrawScreenWidht ());
}

Object*
Wool::GetDrawScreenHeight ()
{
  return new (this) Num (GrphWnd::GetGrphWnd ()->GetDrawScreenHeight ());
}

Object*
Wool::DlgMessage (Object* title, Object* str)
{
  if (!TypeP (title, WT_Str) || !TypeP (str, WT_Str))
  {
    Error ("DlgMessage : arg isn't string");
  }
  GrphWnd::GetGrphWnd ()->MessageBox (((Str*)((Object*)title))->GetStr (),
				      ((Str*)((Object*)str))->GetStr (), 
				      MB_OK);
  return 0;
}

Object*
Wool::DlgInput (Object* text, Object* title, Object* default_str)
{
  return 0;
}

#endif /* MSWG */
