// -------------------------------------------------------------------- //
//                        Wool Object Oriented Lisp
//           Copyright (c) 1993-4 by T.Kudou. All rights reserved.
//
// wool.h:
//
// WOOL(Wool Object Oriented Lisp)
// -------------------------------------------------------------------- //
// $Header: /d/1/proj/egypt/0/wool/RCS/wool.h,v 1.25 1994/07/17 13:07:28 kudou Exp $

#ifndef _wool_h
#define _wool_h

extern "C"
{
#include <stddef.h>
#include <setjmp.h>
#include <assert.h>
}

#include "typedef.h"
#include "conf.h"

#define WOOL_VERSION "WOOL version 0.15"

// forward ref.
class IFlow;
class OFlow;
class Wool;
class Free;

#ifndef NOALINE
#ifndef TWO_BYTE_ALINE
#ifndef FOUR_BYTE_ALINE
#define TWO_BYTE_ALINE
#endif
#endif
#endif

inline void*
MemAline (void* p)
{
#ifdef NOALINE  
  return p;
#endif
#ifdef TWO_BYTE_ALINE
  return (((long)p) & 1) ? ((char*)p + 1) : p;
#endif
#ifdef FOUR_BYTE_ALINE
  return (((long)p) & 3) ? ((char*)p + (((long)p) & 3)) : p;
#endif
}

inline size_t 
MemAlineSize (size_t s)
{
#ifdef NOALINE  
  return s;
#endif
#ifdef TWO_BYTE_ALINE
  return (s & 1) ? (s + 1) : s;
#endif
#ifdef FOUR_BYTE_ALINE
  return (s & 3) ? (s+ (s & 3)) : s;
#endif
}

enum WoolType
{
  WT_Object,
  WT_Pair,
  WT_Atom,
  WT_Num,
  WT_FNum,
  WT_Str,
  WT_Free,
  WT_Func,
};

void* operator new (size_t s, Wool* wool);

class Object
{
  // Object Mark
  int object_mark :1;

  // for Gabage Collection
  int gc_mark_p :1;
  
  // object number
  static int object_num;
  
 public:
  Object ();
  virtual ~Object ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf () = 0;
  virtual WoolType TypeOf () = 0;
  virtual void Print (Wool& wool) = 0;
  virtual bool Equal (Object* x);
  virtual void* GetNextPtr ();
  bool GCMarkP () { return gc_mark_p; }
  void ClearGCMark () { gc_mark_p = False; }
  virtual void SetGCMark ();
  static int GetObjectNumber () { return object_num; }
};

class Pair
: public Object
{
  Object* car;
  Object* cdr;
  
 public:
  Pair (Object* car = 0, Object* cdr = 0);
  virtual ~Pair ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  Object* Car () { return car; }
  Object* Cdr () { return cdr; }
  void ReplaceCar (Object* o) { car = o; }
  void ReplaceCdr (Object* o) { cdr = o; }
  virtual void SetGCMark ();
};

class Atom
: public Object
{
  Object* bind;
  char* name;
  Atom* next;	// atom list link

 public:
  Atom ();
  Atom (char* s, size_t n);
  virtual ~Atom ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  Object* GetBind () { return bind; }
  void SetBind (Object* x) { bind = x; }
  char* GetName () { return name; }
  virtual void SetGCMark ();
 
  // for atom list link
  Atom* GetNext () { return next; }
  void SetNext (Atom* next) { Atom::next = next; }
  bool Match (char* s, size_t n);
};

class Num
: public Object
{
  long int num;
  
 public:
  Num (long int n = 0);
  virtual ~Num ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  virtual bool Equal (Object* x);
  long int GetNum () { return num; }
};

class FNum
: public Object
{
  double fnum;
  
 public:
  FNum (double n = 0.0);
  virtual ~FNum ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  virtual bool Equal (Object* x);
  double GetFNum () { return fnum; }
};

class Str
: public Object
{
  char* str;
  size_t alloc_size;
  
 public:
  Str ();
  Str (char* s, size_t n);
  Str (char* s);
  virtual ~Str ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  virtual bool Equal (Object* x);
  char* GetStr () { return str; }
};

class GCLink
{
  Object* ptr;
  GCLink* next;
  static GCLink* top;

  GCLink* UnLink ();
 public:
  GCLink (Object* p = 0);
  GCLink (GCLink &l);
  ~GCLink ();
  operator Object* () { return ptr; }
  Object* GetPtr () { return ptr; } 
  Object* operator = (Object* o) { return ptr = o; }
  bool operator == (GCLink& g) { return ptr == g.ptr; }
  GCLink* GetNext () { return next; }
  static GCLink* GetTop () { return top; }
  static void SetGCMark ();
  static void ClearTop () { top = 0; }
};

inline 
GCLink::GCLink (Object* p)
{
  ptr = p;
  next = top;
  top = this;
}

inline
GCLink::GCLink (GCLink &l)
{
  ptr = l.ptr;
  next = top;
  top = this;
}

inline
GCLink::~GCLink ()
{
  top = (top == this) ? next : UnLink ();
}

// Object Pointer
#if 0
class Op
{
  Op* prev;
  Op* next;
  static GCLink* top;
 public:
  Op (Object* p = 0);
  ~Op ();
};
#endif

class Free
: public Object
{
  size_t size;
  Free* next;
  
 public:
  Free (size_t s, Free* next);
  virtual ~Free ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  void* GetMemory (size_t size);
  size_t GetSize () { return size; }
  Free* GetNext () { return next; }
  void SetNext(Free* f) { next = f; }
  virtual void* GetNextPtr ();
};

void* operator new (size_t size, Free* p);

class NameSpace
{
  enum { hash_table_size = 256 };
  Wool* wool;
  Atom* hash_table[hash_table_size];
  
  unsigned int Hash (char* name, size_t n);
 public:
  NameSpace (Wool* wool);
  Atom* GetAtom (char* name, size_t n);
  Atom* GetAtom (char* name);
  Atom* GetQuote () { return GetAtom ("quote"); }
  Atom* GetNil ()   { return GetAtom ("nil"); }
  Atom* GetT () { return GetAtom ("t"); }
  void SetGCMark ();
};

class WoolHeap
{
  WoolHeap* next;
  void* mem;
  Free* free_top;
  
#ifndef HEAP_UNIT
#define HEAP_UNIT 8192
#endif
  enum { alloc_size = HEAP_UNIT };
  
 public:
  WoolHeap (WoolHeap* next = 0);
  ~WoolHeap ();
  WoolHeap* GetNext () { return next; }
  void SetNext (WoolHeap* n) { next = n; }
  Free* GetFreeTop () { return free_top; }
  void SetFreeTop(Free* f) { free_top = f; }
  
  void* GetMemTop () { return mem; }
  void* GetEndPtr () { return (void*)((char*)mem + alloc_size); }
};

class Wool
{
  // flow
  IFlow* i_flow;
  OFlow* o_flow;
  OFlow* err_flow;
  
  // read
  enum { read_buf_size = 8192 };
  char read_buf[read_buf_size];
  char* line_end;
  char* scan_pos;
  
  // run flag
  bool run;
  bool eof_input;
  
  // atom table
  NameSpace atom_table;
  
  // heap
  WoolHeap* heap_top;
  
  enum GcMessageMode
  {
    None = 0,
    Show = 1,
    Detail = 2,
  }
  gc_message_mode;
  
 public:
  Wool (IFlow* i_flow, OFlow* o_flow, OFlow* err_flow);
  ~Wool ();
  void Run ();
  
 private:
  Atom* GetAtom (char* name);
  void InitAtom ();
  
 public:
  void Printf (char* f, ...);
  
 private:
  // error
  void Warnning (char* f, ...);
  Object* Error (char* f, ...);
  
  // lexer
  bool ReadLine ();
  void ReadByte (int n);
  bool ReadCheck ();
  enum Token
  {
    TKN_LeftBrace,
    TKN_RightBrace,
    TKN_Priod,
    TKN_Quote,
    TKN_Atom,
    TKN_Str,
    TKN_Num,
    TKN_FNum,
    TKN_Eof,
  };
  void YyLex (Token& token, GCLink& yylval);
  
  // perser
  Object* ReadSexp ();					// SUBR
  Object* ReadLst ();
  Object* ReadCdr ();
  Object* ReadPar (GCLink x);
  Object* ReadQuote ();
  
  // print
  void Print (Object* x);
  void PrintCdr (Object* x);
  Object* PrintSexp (Object* x);			// FSUBR
  
  bool AtomP (Object* x);
  bool NullP (Object* x) { return x == 0; }
  bool EqP (Object* x, Object* y) { return x == y; }
  bool TypeP (Object* x, WoolType t);
  // don't null check, special use only. 
  bool TypeP_ (GCLink& g, WoolType t) { return ((Object*)g)->TypeOf () == t; }
  
  // pure lisp function
  Object* Cons (Object* car, Object* cdr);		// SUBR
  Object* Atom_ (Object* x);				// SUBR
  Object* Null (Object* x);				// SUBR
  Object* Car (Object* x);				// SUBR
  Object* Cdr (Object* x);				// SUBR
  Object* Eq (Object* x, Object* y);			// SUBR
  Object* Rplaca (Object* x, Object* y);		// SUBR
  Object* Rplacd (Object* x, Object* y);		// SUBR
  
  // misc
  Object* EndRun ();					// SUBR
  Object* Version ();					// SUBR
  Object* GetBind (Object* atom, Object* attr);
  Object* SetBind (Object* atom, Object* attr, Object* value);
  Object* SetBind (char* atom_name, Object* attr, Object* value);
  Object* CSet (Object* atom, Object* value);		// SUBR
  Object* CSetQ (Object* atom, Object* value, Object* alist);// FSUBR
  Object* Set (Object* atom, Object* value, Object* alist);// SUBR
  Object* SetQ (Object* atom, Object* value, Object* alist);// FSUBR
  Object* Get (Object* atom);				// SUBR
  Object* Quote (Object* x);				// FSUBR

  enum FuncKind
  {
    FK_Subr = 0,
    FK_FSubr = 1,
    FK_Expr = 2,
    FK_FExpr = 3,
    FK_Macro = 4,
  };
  Object* GetFuncBind (Object* g0, FuncKind& fk);
  
  // for SUBR and FSUBR
  void InitSubr ();
  Object* CallSubr (Object* func, Object* e, Object* alist);
  Object* CallFSubr (Object* func, Object* e, Object* alist);
  
  // eval
  Object* Eval (Object* e, Object* alist);		// SUBR
  Object* Apply (Object* fn, Object* x, Object* alist);	// SUBR
  Object* Assoc (Object* x, Object* alist);		// SUBR
  Object* Evcon (Object* c, Object* alist);		// SUBR
  Object* Evlis (Object* m, Object* alist);		// SUBR
  Object* Pairlis (Object* x, Object* y, Object* alist);// SUBR
  Object* Define (Object* m);				// SUBR
  Object* Defun (Object* m);				// FSUBR
  Object* Let (Object* m, Object* alist);		// FSUBR
  Object* Progn (Object* m, Object* alist);		// FSUBR
  Object* Prog1 (Object* m, Object* alist);		// FSUBR
  Object* Prog2 (Object* m, Object* alist);		// FSUBR
  Object* While (Object* m, Object* alist);		// FSUBR
  Object* If (Object* m, Object* alist);		// FSUBR
  
  // cxxr cxxxr cxxxxr cxxxxxr
  Object* Cxr (Object* x, char* xxx);
  
  Object* Caar (Object* x);				// SUBR
  Object* Cadr (Object* x);				// SUBR
  Object* Cdar (Object* x);				// SUBR
  Object* Cddr (Object* x);				// SUBR
  Object* Caaar (Object* x) { return Cxr (x, "aaa"); }	// SUBR
  Object* Caadr (Object* x) { return Cxr (x, "aad"); }	// SUBR
  Object* Cadar (Object* x) { return Cxr (x, "ada"); }	// SUBR
  Object* Caddr (Object* x) { return Cxr (x, "add"); }	// SUBR
  Object* Cdaar (Object* x) { return Cxr (x, "daa"); }	// SUBR
  Object* Cdadr (Object* x) { return Cxr (x, "dad"); }	// SUBR
  Object* Cddar (Object* x) { return Cxr (x, "dda"); }	// SUBR
  Object* Cdddr (Object* x) { return Cxr (x, "ddd"); }	// SUBR
  
  Object* Caaaar (Object* x) { return Cxr (x, "aaaa"); }// SUBR
  Object* Caaadr (Object* x) { return Cxr (x, "aaad"); }// SUBR
  Object* Caadar (Object* x) { return Cxr (x, "aada"); }// SUBR
  Object* Caaddr (Object* x) { return Cxr (x, "aadd"); }// SUBR
  Object* Cadaar (Object* x) { return Cxr (x, "adaa"); }// SUBR
  Object* Cadadr (Object* x) { return Cxr (x, "adad"); }// SUBR
  Object* Caddar (Object* x) { return Cxr (x, "adda"); }// SUBR
  Object* Cadddr (Object* x) { return Cxr (x, "addd"); }// SUBR
  Object* Cdaaar (Object* x) { return Cxr (x, "daaa"); }// SUBR
  Object* Cdaadr (Object* x) { return Cxr (x, "daad"); }// SUBR
  Object* Cdadar (Object* x) { return Cxr (x, "dada"); }// SUBR
  Object* Cdaddr (Object* x) { return Cxr (x, "dadd"); }// SUBR
  Object* Cddaar (Object* x) { return Cxr (x, "ddaa"); }// SUBR
  Object* Cddadr (Object* x) { return Cxr (x, "ddad"); }// SUBR
  Object* Cdddar (Object* x) { return Cxr (x, "ddda"); }// SUBR
  Object* Cddddr (Object* x) { return Cxr (x, "dddd"); }// SUBR
  
  Object* Caaaaar (Object* x) { return Cxr (x, "aaaaa"); }// SUBR
  Object* Caaaadr (Object* x) { return Cxr (x, "aaaad"); }// SUBR
  Object* Caaadar (Object* x) { return Cxr (x, "aaada"); }// SUBR
  Object* Caaaddr (Object* x) { return Cxr (x, "aaadd"); }// SUBR
  Object* Caadaar (Object* x) { return Cxr (x, "aadaa"); }// SUBR
  Object* Caadadr (Object* x) { return Cxr (x, "aadad"); }// SUBR
  Object* Caaddar (Object* x) { return Cxr (x, "aadda"); }// SUBR
  Object* Caadddr (Object* x) { return Cxr (x, "aaddd"); }// SUBR
  Object* Cadaaar (Object* x) { return Cxr (x, "adaaa"); }// SUBR
  Object* Cadaadr (Object* x) { return Cxr (x, "adaad"); }// SUBR
  Object* Cadadar (Object* x) { return Cxr (x, "adada"); }// SUBR
  Object* Cadaddr (Object* x) { return Cxr (x, "adadd"); }// SUBR
  Object* Caddaar (Object* x) { return Cxr (x, "addaa"); }// SUBR
  Object* Caddadr (Object* x) { return Cxr (x, "addad"); }// SUBR
  Object* Cadddar (Object* x) { return Cxr (x, "addda"); }// SUBR
  Object* Caddddr (Object* x) { return Cxr (x, "adddd"); }// SUBR
  Object* Cdaaaar (Object* x) { return Cxr (x, "daaaa"); }// SUBR
  Object* Cdaaadr (Object* x) { return Cxr (x, "daaad"); }// SUBR
  Object* Cdaadar (Object* x) { return Cxr (x, "daada"); }// SUBR
  Object* Cdaaddr (Object* x) { return Cxr (x, "daadd"); }// SUBR
  Object* Cdadaar (Object* x) { return Cxr (x, "dadaa"); }// SUBR
  Object* Cdadadr (Object* x) { return Cxr (x, "dadad"); }// SUBR
  Object* Cdaddar (Object* x) { return Cxr (x, "dadda"); }// SUBR
  Object* Cdadddr (Object* x) { return Cxr (x, "daddd"); }// SUBR
  Object* Cddaaar (Object* x) { return Cxr (x, "ddaaa"); }// SUBR
  Object* Cddaadr (Object* x) { return Cxr (x, "ddaad"); }// SUBR
  Object* Cddadar (Object* x) { return Cxr (x, "ddada"); }// SUBR
  Object* Cddaddr (Object* x) { return Cxr (x, "ddadd"); }// SUBR
  Object* Cdddaar (Object* x) { return Cxr (x, "dddaa"); }// SUBR
  Object* Cdddadr (Object* x) { return Cxr (x, "dddad"); }// SUBR
  Object* Cddddar (Object* x) { return Cxr (x, "dddda"); }// SUBR
  Object* Cdddddr (Object* x) { return Cxr (x, "ddddd"); }// SUBR
  
  // test functions
#ifdef TEST_WOOL
 public:
  void TestYyLex ();
  void TestReadPrint ();
#endif
  
  // lisp functions
  Object* And (Object* m, Object* alist);		// FSUBR
  Object* Or (Object* m, Object* alist);		// FSUBR
  Object* Equal (Object* x, Object* y);			// SUBR
  Object* List (Object* m, Object* alist);		// FSUBR
  Object* Append (Object* x, Object* y);		// SUBR
  Object* Nconc (Object* x, Object* y);			// SUBR
  Object* Not (Object* x);				// SUBR
  Object* Last (Object* x);				// SUBR
  Object* Subst (Object* new_s, Object* old_s, Object* x); // SUBR
  Object* Reverse (Object* x);				// SUBR
  
  // numelical functions
  Object* Rand ();					// SUBR
  Object* Add1 (Object* x);				// SUBR
  Object* Sub1 (Object* x);				// SUBR
  Object* Minus (Object* m);				// SUBR
  Object* Plus (Object* m);				// SUBR
  Object* Difference (Object* x, Object* y);		// SUBR
  Object* Times (Object* m);				// SUBR
  Object* Quotient (Object* x, Object* y);		// SUBR
  Object* Remainder (Object* x, Object* y);		// SUBR
  Object* ZeroP (Object* x);				// SUBR
  Object* MinusP (Object* x);				// SUBR
  Object* GreaterP (Object* x, Object* y);		// SUBR
  Object* LessP (Object* x, Object* y);			// SUBR
  Object* GreaterOrEqP (Object* x, Object* y);		// SUBR
  Object* LessOrEqP (Object* x, Object* y);		// SUBR
  Object* Length (Object* x);				// SUBR
  Object* NumberP (Object* x);				// SUBR
  Object* FloatP (Object* x);				// SUBR
  Object* Fix (Object* x);				// SUBR
  Object* LeftShift (Object* x, Object* y);		// SUBR
  Object* RightShift (Object* x, Object* y);		// SUBR
  Object* Max (Object* m);				// SUBR
  Object* Min (Object* m);				// SUBR
#ifdef MATHFUNC
  Object* Acos (Object* x);				// SUBR
  Object* Asin (Object* x);				// SUBR
  Object* Atan (Object* x);				// SUBR
  Object* Atan2 (Object* y, Object* x);			// SUBR
  Object* Ceil (Object* x);				// SUBR
  Object* Cos (Object* x);				// SUBR
  Object* Cosh (Object* x);				// SUBR
  Object* Exp (Object* x);				// SUBR
  Object* Fabs (Object* x);				// SUBR
  Object* Floor (Object* x);				// SUBR
  Object* Log (Object* x);				// SUBR
  Object* Log10 (Object* x);				// SUBR
  Object* Pow (Object* x, Object* y);			// SUBR
  Object* Sin (Object* x);				// SUBR
  Object* Sinh (Object* x);				// SUBR
  Object* Sqrt (Object* x);				// SUBR
  Object* Tan (Object* x);				// SUBR
  Object* Tanh (Object* x);				// SUBR
  Object* MathError ();					// SUBR
#endif /* MATHFUNC */

  // print
  void Printf1 (char* fmt, Object* o);
  Object* Printf_ (Object* fmt, Object* args);		// FSUBR

  // for GC
  Object* GetObjectNumber ();				// SUBR
  bool GC ();
  Object* RunGC ();					// SUBR
  Object* SetGCMessageMode (Object* mode);		// SUBR
  
#ifdef MSWG
  Object* Rgb (Object* r, Object* g, Object* b);	// SUBR
  Object* SetDrawPenColor (Object* c);			// SUBR
  Object* SetDrawFillColor (Object* c);			// SUBR
  Object* SetDrawBackColor (Object* c);			// SUBR
  Object* DrawPixel (Object* x, Object* y);		// SUBR
  Object* DrawText (Object* x, Object* y, Object* str);	// SUBR
  Object* DrawTransparentText (Object* x, Object* y, Object* str);	// SUBR
  Object* DrawLine (Object* x1 , Object* y1, Object* x2, Object* y2);	// SUBR
  Object* DrawBox (Object* x1, Object* y1, Object* x2, Object* y2);	// SUBR
  Object* DrawBoxFill (Object* x1, Object* y1, Object* x2, Object* y2);	// SUBR
  Object* DrawArcFill (Object* x, Object* y, Object* r);		// SUBR
  Object* ClearDrawWindow ();				// SUBR
  Object* GetDrawWindowWidth ();			// SUBR
  Object* GetDrawWindowHeight ();			// SUBR
  Object* GetDrawScreenWidth ();			// SUBR
  Object* GetDrawScreenHeight ();			// SUBR
  Object* DlgMessage (Object* title, Object* str);	// SUBR
  Object* DlgInput (Object* text, Object* title, Object* default_str);	// SUBR

#endif /* MSWG */ 
 public:
  void* GetMemory (size_t size);
};

typedef Object* (Wool::*WoolFunc0)();
typedef Object* (Wool::*WoolFunc1)(Object*);
typedef Object* (Wool::*WoolFunc2)(Object*, Object*);
typedef Object* (Wool::*WoolFunc3)(Object*, Object*, Object*);
typedef Object* (Wool::*WoolFunc4)(Object*, Object*, Object*, Object*);

// for SUBR and FSUBR function
class Func
: public Object
{
  union
  {
    WoolFunc0 func0;
    WoolFunc1 func1;
    WoolFunc2 func2;
    WoolFunc3 func3;
    WoolFunc4 func4;
  } f;
  int arg_num;
 public:
  enum FncType
  {
    Normal        = 0x00,
    CallWithEnv   = 0x01,
    UnfixedArgNum = 0x02,
    UnfixedWithEnv = 0x01 | 0x02,
  };   
 private:
  int func_type;
  
 public:
  Func (WoolFunc0 func, int type);
  Func (WoolFunc1 func, int type);
  Func (WoolFunc2 func, int type);
  Func (WoolFunc3 func, int type);
  Func (WoolFunc4 func, int type);
  virtual ~Func ();
  void operator delete (void*) { ; }
  virtual size_t SizeOf ();
  virtual WoolType TypeOf ();
  virtual void Print (Wool& wool);
  Object* Func0 (Wool& wool);
  Object* Func1 (Wool& wool, Object* a1);
  Object* Func2 (Wool& wool, Object* a1, Object* a2);
  Object* Func3 (Wool& wool, Object* a1, Object* a2, Object* a3);
  Object* Func4 (Wool& wool, Object* a1, Object* a2, Object* a3, Object* a4);
  bool CallWithEnvP () { return func_type & CallWithEnv; }
  bool UnfixedArgNumP () { return func_type & UnfixedArgNum; }
  int GetArgNum () { return arg_num; }
};

#endif /* _wool_h */
